package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementRef;
import javax.xml.bind.annotation.XmlElementRefs;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlIDREF;

import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * Class aimed at putting together reflection info about a {@link ModelObject}
 * class.
 * 
 * @author mjambert
 * 
 */
final class JaxbReflectionInfo {

    public static final String JAXB_ATTRIBUTE_NAME_DEFAULT = "##default";
    public static final Class<?> JAXB_ATTRIBUTE_TYPE_DEFAULT = XmlElement.DEFAULT.class;

    private static final Map<Class<? extends ModelObject>, JaxbReflectionInfo> jaxbReflectionInfos = 
        new HashMap<Class<? extends ModelObject>, JaxbReflectionInfo>();

    private final Class<? extends ModelObject> modelObjectClass;
    private final Map<String, JaxbReflectionFieldInfo> childrenFieldInfos;
    private final Map<String, JaxbReflectionFieldInfo> attributeFieldInfos;

    JaxbReflectionInfo(Class<? extends ModelObject> modelObjectClass) {
        this.modelObjectClass = modelObjectClass;
        this.childrenFieldInfos = new LinkedHashMap<String, JaxbReflectionFieldInfo>();
        this.attributeFieldInfos = new LinkedHashMap<String, JaxbReflectionFieldInfo>();
        populateFields();
    }


    static JaxbReflectionInfo getReflectionInfo(
            Class<? extends ModelObject> modelObjectClass) {
        if (!jaxbReflectionInfos.containsKey(modelObjectClass)) {
            jaxbReflectionInfos.put(modelObjectClass, new JaxbReflectionInfo(
                    modelObjectClass));
        }
        return jaxbReflectionInfos.get(modelObjectClass);
    }    
    
    Collection<JaxbReflectionFieldInfo> getFieldInfos() {
        List<JaxbReflectionFieldInfo> fieldInfos = new ArrayList<JaxbReflectionFieldInfo>();
        fieldInfos.addAll(attributeFieldInfos.values());
        fieldInfos.addAll(childrenFieldInfos.values());
        return fieldInfos;
    }
    
    Collection<JaxbReflectionFieldInfo> getChildrenFieldInfos() {
        return childrenFieldInfos.values();
    }

    JaxbReflectionFieldInfo getChildFieldInfoByName(String name) {
        return childrenFieldInfos.get(name);
    }
    
    Collection<JaxbReflectionFieldInfo> getAttributeFields() {
        return attributeFieldInfos.values();
    }

    private static boolean isJaxbComplexTypeClass(Class<?> fieldClass) {
        return JAXBElement.class.equals(fieldClass)
                || ModelObject.class.isAssignableFrom(fieldClass);
    }

    private static boolean isComplexChildField(Field field) {
        // note that child enums, even if flagged with XmlElement* annotation,
        // will not be considered as a complexChild field because not
        // implementing
        // ModelObject.
        if (ModelObject.class.isAssignableFrom(field.getType())
                || isModelObjectCollectionField(field))
            return true;
        if (field.isAnnotationPresent(XmlElements.class)) {
            XmlElement[] xmlElements = field.getAnnotation(XmlElements.class)
                    .value();
            for (XmlElement xmlElement : xmlElements) {
                if (isJaxbComplexTypeClass(xmlElement.type()))
                    return true;
            }
            return false;
        }
        if (field.isAnnotationPresent(XmlElementRefs.class)) {
            XmlElementRef[] xmlElementRefs = field.getAnnotation(
                    XmlElementRefs.class).value();
            for (XmlElementRef xmlElementRef : xmlElementRefs) {
                if (isJaxbComplexTypeClass(xmlElementRef.type()))
                    return true;
            }
            return false;
        }
        
        if (isJaxbComplexTypeClass(getOverridenTypeOrType(field))) {
        	return true;
        }
        if (field.isAnnotationPresent(XmlAnyElement.class)) {
            return true;
        }
        if (isModelObjectIdRef(field)) {
        	// TODO : JAXB returns a resolved reference to the model object itself.
        	// we should consider having special reflection methods to get JAXB adoptive children.
        	// Otherly said, all JAXB children are not necessarily "natural" children.
        	return true;
        }
        
        return false;
    }

    
    private static Class<?> getOverridenTypeOrType(Field attributeField) {
    	Class<?> attributeType = attributeField.getType();
        if (attributeField.isAnnotationPresent(XmlElement.class)) {
        	attributeType = attributeField.getAnnotation(XmlElement.class)
                    .type();
        }
        if (attributeField.isAnnotationPresent(XmlElementRef.class)) {
        	attributeType = attributeField.getAnnotation(XmlElementRef.class)
                    .type();
        }

        if (JAXB_ATTRIBUTE_TYPE_DEFAULT.equals(attributeType)) {
        	attributeType = attributeField.getType();
        }
        return attributeType;
    	
    }
    
    
    private static boolean isAttributeField(Field field) {
        if (ModelObject.class.isAssignableFrom(field.getType())
                || isModelObjectCollectionField(field)
                || isModelObjectIdRef(field))
            return false;
        return true;
    }

    
    private static boolean isModelObjectIdRef(Field field) {
    	return field.isAnnotationPresent(XmlIDREF.class);
	}


	private static String getOverridenNameOrName(Field attributeField) {
        String attributeName = attributeField.getName();
        if (attributeField.isAnnotationPresent(XmlAttribute.class)) {
            attributeName = attributeField.getAnnotation(XmlAttribute.class)
                    .name();
        }
        if (attributeField.isAnnotationPresent(XmlElement.class)) {
            attributeName = attributeField.getAnnotation(XmlElement.class)
                    .name();
        }
        if (attributeField.isAnnotationPresent(XmlElementRef.class)) {
            attributeName = attributeField.getAnnotation(XmlElementRef.class)
                    .name();
        }

        if (JAXB_ATTRIBUTE_NAME_DEFAULT.equals(attributeName)) {
            attributeName = attributeField.getName();
        }
        return attributeName;
    }

    private void populateFields() {
        try {
            for (Field field : findAllFieldsIncludingInherited(modelObjectClass)) {
                field.setAccessible(true);
                String nameOrOverridenName = getOverridenNameOrName(field);
                if (isComplexChildField(field)) {
                    childrenFieldInfos.put(nameOrOverridenName,
                    		new JaxbReflectionFieldInfo(field,
                            JaxbReflectionFieldInfo.Type.CHILD, nameOrOverridenName));
                } else if (isAttributeField(field)) {
                    attributeFieldInfos.put(nameOrOverridenName,
                    		new JaxbReflectionFieldInfo(field,
                            JaxbReflectionFieldInfo.Type.ATTRIBUTE,
                            nameOrOverridenName));
                } else {
                    assert false : String.format(
                            "Unrecognized JAXB field '%s' in class '%s'.",
                            field.getName(), modelObjectClass.getSimpleName());
                }
            }
        } catch (IllegalArgumentException e) {
            throw new UncheckedException("Illegal argument.", e);
        }
    }

    private static List<Field> findAllFieldsIncludingInherited(
            Class<? extends ModelObject> modelObjectClass) {
        List<Field> fields = new LinkedList<Field>();
        Class<?> currentClass = modelObjectClass;
        while (true) {
            if (currentClass == null)
                break;
            if (currentClass.equals(AbstractJaxbModelObject.class))
                break;
            // insert at position 0 so that fields from superclass come first.
            fields.addAll(0, Arrays.asList(currentClass.getDeclaredFields()));
            currentClass = currentClass.getSuperclass();
        }
        return fields;
    }

    private static boolean isModelObjectCollectionField(Field field) {
        if (!Collection.class.isAssignableFrom(field.getType()))
            return false;
        Type collectionItemType = field.getGenericType();

        String collectionItemClassName = collectionItemType.toString();
        // TODO : only valid for single generic parameter collections.
        collectionItemClassName = collectionItemClassName.substring(
                collectionItemClassName.indexOf('<') + 1,
                collectionItemClassName.indexOf('>'));
        try {
            Class<?> collectionItemClass = Class
                    .forName(collectionItemClassName);
            return ModelObject.class.isAssignableFrom(collectionItemClass);
        } catch (ClassNotFoundException e) {
            return false;
        }
    }

}
