/**
 * 
 */
package com.ebmwebsourcing.easybox.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

final class ClassPathResourceResolver implements LSResourceResolver {

    private static Logger LOG = Logger.getLogger(ClassPathResourceResolver.class
            .getName());
    
	private static final DOMImplementationLS domImplementationLS;
	
	static {
		try {
			DOMImplementationRegistry registry = DOMImplementationRegistry
					.newInstance();
			domImplementationLS = (DOMImplementationLS) registry
					.getDOMImplementation("XML").getFeature("LS", "3.0");

		} catch (Exception e) {
			throw new UncheckedException(e);
		}

	}

    private final Set<String> originatingDirs;
    private final Map<String, String> originatingResourcePathByNamespaceURI;

    
	public ClassPathResourceResolver() {
		this.originatingDirs = new HashSet<String>();
		this.originatingResourcePathByNamespaceURI = new HashMap<String, String>();
	}
	

	public void addOriginatingDir(String originatingDir) {
		originatingDirs.add(originatingDir);
	}
	

	public void addOriginatingResourcePath(String namespaceURI, String resourcePath) {
	    originatingResourcePathByNamespaceURI.put(namespaceURI, resourcePath);
	}

    /**
     * Return a valid resource URL in classpath given a namespaceURI.
     * 
     * @param namespaceURI Searched namespace URI.
     * @return Resource URL or {@code null} if resource is not available in classpath. 
     */
    URL resolveResourceByNamespaceURI(String namespaceURI) {
        assert namespaceURI != null;
        String resourcePath = originatingResourcePathByNamespaceURI.get(namespaceURI);
        if (resourcePath == null) return null;
        URL resourceURL = getClass().getClassLoader().getResource(resourcePath);
        return resourceURL == null ? null : resourceURL;
    }

    
    
    /**
     * Return a valid resource URL in classpath given a system id.<br/>
     * 
     * System id may be complete (absolute path in classpath) or relative
     * (relative path in classpath. In latter case, all registered originating
     * dirs are tried to resolve relative path. 
     * 
     * @param systemId Absolute or "originating-dir relative" path in classpath.
     * @return Resource URL or {@code null} if resource is not available in classpath. 
     */
    URL resolveResourceBySystemId(String systemId) {
        assert systemId != null;
        URL resourceURL = getClass().getClassLoader().getResource(systemId);
        if (resourceURL != null) return resourceURL;
        for (String originatingDir : originatingDirs) {
            String absolutePath = originatingDir + "/" + systemId;
            resourceURL = getClass().getClassLoader().getResource(absolutePath);
            if (resourceURL != null) return resourceURL;
        }
        return null;
    }

    
	@Override
	public LSInput resolveResource(String type, String namespaceURI,
			String publicId, String systemId, String baseURI) {
	    URL resourceURL = null;
	    if (systemId != null) {
            resourceURL = resolveResourceBySystemId(systemId);
        } else if (namespaceURI != null) {
            resourceURL = resolveResourceByNamespaceURI(namespaceURI);
	    }  
	    if (resourceURL == null) {
	        LOG.warning(String.format("Cannot resolve resource { type='%s' , namespaceURI='%s', publicId='%s', systemId='%s', baseURI='%s'} from classpath.", 
	                type, namespaceURI, publicId, systemId, baseURI));
	        return null;
	    }
	    
        InputStream classPathInputStream;
        try {
            classPathInputStream = resourceURL.openStream();
            LSInput lsInput = domImplementationLS.createLSInput();
            lsInput.setByteStream(classPathInputStream);
            return lsInput;
        } catch (IOException e) {
            throw new UncheckedException(
                    String.format("Problem while opening stream from URL '%s'.", 
                    resourceURL.toString()), e);
        }
	}

}