package com.ebmwebsourcing.easybox.impl;

import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.xml.DOMHelper;
import com.ebmwebsourcing.easycommons.xml.DefaultNamespaceContext;

/**
 * Centralizes all JAXB schema bindings information,
 * extracted from a JAXB episode file. 
 * 
 * @author mjambert
 *
 */
final class JaxbSchemaBinding {

    private static final String EASYBOX_PREFIX_PACKAGE = "easybox";

    private static Logger LOG = Logger.getLogger(JaxbSchemaBinding.class.getName());

    private static final String JAXB_EPISODE_PATH = "META-INF/sun-jaxb.episode";
    private static final String JAXB_NAMESPACE_URI = "http://java.sun.com/xml/ns/jaxb";
    private static final String JAXB_PREFIX = "jaxb";

    private static final Map<String, JaxbSchemaBinding> JAXB_SCHEMA_BINDINGS_BY_NAMESPACE_URI;
    private static final XPath XPATH;

    static {
        XPathFactory xpathFactory = XPathFactory.newInstance();
        XPATH = xpathFactory.newXPath();
        DefaultNamespaceContext namespaceContext = new DefaultNamespaceContext();
        namespaceContext.bindNamespace(JAXB_PREFIX, JAXB_NAMESPACE_URI);
        XPATH.setNamespaceContext(namespaceContext);

        JAXB_SCHEMA_BINDINGS_BY_NAMESPACE_URI = new HashMap<String, JaxbSchemaBinding>();
        try {
            Enumeration<URL> episodesURLs = JaxbSchemaBinding.class.getClassLoader().getResources(JAXB_EPISODE_PATH);
            while (episodesURLs.hasMoreElements()) {
                URL episodeURL = episodesURLs.nextElement();
                for (JaxbSchemaBinding jaxbSchemaBinding : parseSchemaBindings(episodeURL)) {

                    if(jaxbSchemaBinding.getTargetPackage().getName().startsWith(EASYBOX_PREFIX_PACKAGE)) {
                        if(JAXB_SCHEMA_BINDINGS_BY_NAMESPACE_URI.get(jaxbSchemaBinding.getTargetNamespaceURI()) != null) { 
                            LOG.log(Level.WARNING, String.format("Impossible to add '%s', another JAXB episodes with the name is already registered for '%s':  '%s'", jaxbSchemaBinding.getTargetPackage().getName(), jaxbSchemaBinding.getTargetNamespaceURI(), JAXB_SCHEMA_BINDINGS_BY_NAMESPACE_URI.get(jaxbSchemaBinding.getTargetNamespaceURI()).getTargetPackage().getName()));
                        } else {
                            JAXB_SCHEMA_BINDINGS_BY_NAMESPACE_URI.put(jaxbSchemaBinding.getTargetNamespaceURI(), jaxbSchemaBinding);
                        }
                    } else {
                        LOG.log(Level.WARNING, String.format("Impossible to add '%s', The package name of this jaxb binding MUST start by '%s'.", jaxbSchemaBinding.getTargetPackage().getName(), EASYBOX_PREFIX_PACKAGE));
                    }

                }
            }
        } catch (IOException ioe) {
            LOG.log(Level.SEVERE, "Error while initializing JAXB episodes.", ioe);
        }
    }

    private final Map<Class<? extends ModelObject>, JaxbSchemaObjectBinding> jaxbSchemaObjectBindingsByClass;
    private final Map<QName, JaxbSchemaObjectBinding> jaxbSchemaObjectBindingsBySchemaType;

    private final String targetNamespaceURI;
    private final Package targetPackage;

    JaxbSchemaBinding(String targetNamespaceURI, Package targetPackage, Set<JaxbSchemaObjectBinding> jaxbSchemaObjectBindings) {
        this.targetNamespaceURI = targetNamespaceURI;
        this.targetPackage = targetPackage;
        this.jaxbSchemaObjectBindingsByClass = new HashMap<Class<? extends ModelObject>, JaxbSchemaObjectBinding>();
        this.jaxbSchemaObjectBindingsBySchemaType = new HashMap<QName, JaxbSchemaObjectBinding>();
        for (JaxbSchemaObjectBinding jaxbSchemaObjectBinding : jaxbSchemaObjectBindings) {
            this.jaxbSchemaObjectBindingsByClass.put(jaxbSchemaObjectBinding.getJaxbClass(), jaxbSchemaObjectBinding);
            this.jaxbSchemaObjectBindingsBySchemaType.put(jaxbSchemaObjectBinding.getSchemaType(), jaxbSchemaObjectBinding);
        }
    }

    static Map<String, JaxbSchemaBinding> getJaxbSchemaBindings() {
        return Collections.unmodifiableMap(JAXB_SCHEMA_BINDINGS_BY_NAMESPACE_URI);
    }

    public String getTargetNamespaceURI() {
        return targetNamespaceURI;
    }

    public Package getTargetPackage() {
        return targetPackage;
    }

    public JaxbSchemaObjectBinding getJaxbBindingByClass(Class<? extends ModelObject> jaxbObjectClass) {
        return jaxbSchemaObjectBindingsByClass.get(jaxbObjectClass);
    }

    public JaxbSchemaObjectBinding getJaxbBindingByQName(QName qname) {
        return jaxbSchemaObjectBindingsBySchemaType.get(qname);
    }




    @SuppressWarnings("unchecked")
    private static JaxbSchemaObjectBinding parseSchemaObjectBinding(Element schemaObjectBindingElement,
            String targetNamespaceURI) {
        try {
            String classBindingScd = schemaObjectBindingElement.getAttribute("scd");
            QName schemaType = new QName(targetNamespaceURI, 
                    classBindingScd.substring(classBindingScd.indexOf(':')));
            NodeList classElements = schemaObjectBindingElement.getElementsByTagNameNS(JAXB_NAMESPACE_URI, "class");

            if(classElements.getLength() == 0){
                classElements = schemaObjectBindingElement.getElementsByTagNameNS(JAXB_NAMESPACE_URI, "typesafeEnumClass");
            }

            assert classElements.getLength() == 1;
            Element classElement = (Element) classElements.item(0);
            String jaxbClassName = classElement.getAttribute("ref");
            Class<? extends AbstractJaxbModelObject> jaxbClass = (Class<? extends AbstractJaxbModelObject>) 
            Class.forName(jaxbClassName);
            JaxbSchemaObjectBinding jaxbSchemaObjectBinding = (classBindingScd.charAt(0) == '~') 
            ? new JaxbSchemaTypeBinding(schemaType, jaxbClass)
            : new JaxbSchemaElementBinding(schemaType, jaxbClass);
            return jaxbSchemaObjectBinding;
        } catch (ClassNotFoundException cnfe) {
            throw new UncheckedException("Problem while parsing jaxb episode (ClassNotFoundException).", cnfe);
        }

    }


    private static JaxbSchemaBinding parseSchemaBinding(Element schemaBindingElement) {
        String targetNamespaceURI = schemaBindingElement.lookupNamespaceURI("tns");
        Package targetPackage = null;
        try {
            String xpathStr = "jaxb:bindings[jaxb:class or jaxb:typesafeEnumClass]";
            NodeList schemaObjectBindingElements = (NodeList)
            XPATH.evaluate(xpathStr, schemaBindingElement, XPathConstants.NODESET);
            Set<JaxbSchemaObjectBinding> jaxbSchemaObjectBindings = new HashSet<JaxbSchemaObjectBinding>();

            for (int i = 0 ; i < schemaObjectBindingElements.getLength() ; ++i) {
                Element schemaObjectBindingElement = (Element) schemaObjectBindingElements.item(i);
                JaxbSchemaObjectBinding jaxbSchemaObjectBinding  =
                    parseSchemaObjectBinding(schemaObjectBindingElement, targetNamespaceURI);
                if (targetPackage != null) {
                    assert targetPackage.equals(jaxbSchemaObjectBinding.getJaxbClass().getPackage());
                } else {
                    targetPackage = jaxbSchemaObjectBinding.getJaxbClass().getPackage();
                }
                jaxbSchemaObjectBindings.add(jaxbSchemaObjectBinding);
            }
            return new JaxbSchemaBinding(targetNamespaceURI, targetPackage, jaxbSchemaObjectBindings);

        } catch (XPathExpressionException xee) {
            throw new UncheckedException("Problem while parsing jaxb episode (XPathExpressionException).", xee);
        } 
    }


    private static Set<JaxbSchemaBinding> parseSchemaBindings(URL episodeURL) {
        Set<JaxbSchemaBinding> schemaBindings = new HashSet<JaxbSchemaBinding>();
        DOMSource domSource = DOMHelper.parseAsDOMSource(episodeURL);
        try {
            String xpathStr = "//jaxb:bindings[starts-with(@scd, 'x-schema')]";
            NodeList schemaBindingsNodes = (NodeList)
            XPATH.evaluate(xpathStr, domSource.getNode(), XPathConstants.NODESET);
            for (int i = 0 ; i < schemaBindingsNodes.getLength() ; ++i) {
                assert schemaBindingsNodes.item(i) instanceof Element;
                schemaBindings.add(parseSchemaBinding((Element) schemaBindingsNodes.item(i)));
            }
            return schemaBindings;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException("Problem while parsing jaxb episode (XPathExpressionException).", xee);
        }
    }


}
