package com.ebmwebsourcing.easybox.impl;

import com.ebmwebsourcing.easybox.api.ClassMetadataConstants;
import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectBinding;
import com.ebmwebsourcing.easybox.api.XmlObjectFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

final class XmlObjectFactoryImpl implements XmlObjectFactory {

    private final XmlContext xmlContext;

    public XmlObjectFactoryImpl(XmlContext xmlContext) {
        this.xmlContext = xmlContext;
    }

    /**
     * Wrap any {@link ModelObject} object into an implementation of
     * {@link XmlObject}.
     * 
     * @param <X>
     *            Type of desired {@link XmlObject} interface.
     * @param ModelObject
     *            Model object to be wrapped.
     * @param xmlObjectClassOrInterface
     *            Desired {@link XmlObject} interface class. For convenience,
     *            one can also provide directly the desired implementation class
     *            to prevent unnecessary lookup in
     *            {@link XmlObjectClassRegistry}.
     * @return Newly created concrete {@link XmlObject}.
     */
    @SuppressWarnings("unchecked")
    public <X extends XmlObjectNode> X wrap(Object obj,
            Class<X> xmlObjectClassOrInterface) {
        if (obj instanceof ModelObject) {
            ModelObject modelObject = (ModelObject) obj;
            if (modelObject.getXmlObject() != null)
                return (X) modelObject.getXmlObject();
        }
        if (xmlObjectClassOrInterface.isInterface()) {
            return doWrapGivenXmlObjectInterfaceClass(obj,
                    xmlObjectClassOrInterface);
        } else {
            return doWrapGivenXmlObjectImplClass(obj, xmlObjectClassOrInterface);
        }
    }

    /**
     * Wrap any {@link ModelObject} object into an implementation of
     * {@link XmlObject}. Implementation class is guessed from known
     * meta-information. If a QName is provided, it can be used as well.
     * 
     * @param obj
     *            Internal model object to be wrapped.
     * @return Newly created concrete {@link XmlObject}.
     */
    public XmlObjectNode wrap(Object obj) {
        if (obj instanceof ModelObject) {
            ModelObject modelObject = (ModelObject) obj;
            if (modelObject.getXmlObject() != null)
                return modelObject.getXmlObject();
        }
        for (XmlObjectBinding xob : xmlContext.getXmlObjectBindings()) {
            if (xob.canWrap(obj)) {
                return xob.wrap(xmlContext, obj);
            }
        }
        throw new UncheckedException(
                String.format(
                        "Cannot find an XmlObjectBinding capable of wrapping object class '%s'.",
                        obj.getClass()));
    }

    private <X extends XmlObjectNode> X doWrapGivenXmlObjectImplClass(Object obj,
            Class<X> xmlObjectImplClass) {
        assert xmlObjectImplClass != null;
        XmlObjectBinding binding = xmlContext.getClassMetadata().get(
                xmlObjectImplClass,
                ClassMetadataConstants.IMPLEMENTATION_CLASS_BINDING);
        return binding.wrap(xmlContext, xmlObjectImplClass, obj);
    }

    private <X extends XmlObjectNode> X doWrapGivenXmlObjectInterfaceClass(
            Object obj, Class<X> xmlObjectInterfaceClass) {
        Class<X> xmlObjectImplClass = xmlContext.getClassMetadata().get(
                xmlObjectInterfaceClass,
                ClassMetadataConstants.INTERFACE_CLASS_IMPLEMENTATION_CLASS);
        assert xmlObjectImplClass != null : String.format(
                "No implementation class found for interface '%s'",
                xmlObjectInterfaceClass.getSimpleName());
        return doWrapGivenXmlObjectImplClass(obj, xmlObjectImplClass);
    }

    public <X extends XmlObjectNode> X create(Class<X> xmlObjectClassOrInterface) {
        if (xmlObjectClassOrInterface.isInterface()) {
            return doCreateGivenXmlObjectInterfaceClass(xmlObjectClassOrInterface);
        } else {
            return doCreateGivenXmlObjectImplClass(xmlObjectClassOrInterface);
        }
    }

    private <X extends XmlObjectNode> X doCreateGivenXmlObjectImplClass(
            Class<X> xmlObjectImplClass) {
        assert xmlObjectImplClass != null;
        XmlObjectBinding binding = xmlContext.getClassMetadata().get(
                xmlObjectImplClass,
                ClassMetadataConstants.IMPLEMENTATION_CLASS_BINDING);
        return binding.create(xmlContext, xmlObjectImplClass);
    }

    private <X extends XmlObjectNode> X doCreateGivenXmlObjectInterfaceClass(
            Class<X> xmlObjectInterfaceClass) {
        assert xmlObjectInterfaceClass != null;
        if (!xmlContext.getClassMetadata().has(xmlObjectInterfaceClass,
                ClassMetadataConstants.INTERFACE_CLASS_IMPLEMENTATION_CLASS)) {
            throw new UncheckedException(String.format(
                    "No implementation class found for interface '%s'",
                    xmlObjectInterfaceClass.getName()));
        }
        Class<X> xmlObjectImplClass = xmlContext.getClassMetadata().get(
                xmlObjectInterfaceClass,
                ClassMetadataConstants.INTERFACE_CLASS_IMPLEMENTATION_CLASS);
        return doCreateGivenXmlObjectImplClass(xmlObjectImplClass);
    }

}
