/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import net.sf.saxon.Configuration;
import net.sf.saxon.query.DynamicQueryContext;
import net.sf.saxon.query.StaticQueryContext;
import net.sf.saxon.query.XQueryExpression;
import net.sf.saxon.trans.XPathException;

import com.ebmwebsourcing.easybox.api.XQueryException;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.XmlObjectXQueryEvaluator;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.xml.DefaultNamespaceContext;

final class XmlObjectXQueryEvaluatorImpl implements XmlObjectXQueryEvaluator {

    private static Logger LOG = Logger.getLogger(XmlObjectXQueryEvaluatorImpl.class
            .getName());
    
    private final Configuration configuration;
    private final StaticQueryContext staticQueryContext;

    
    XmlObjectXQueryEvaluatorImpl(DefaultNamespaceContext namespaceContext) {
        this.configuration = new Configuration();
        staticQueryContext = configuration.newStaticQueryContext();
        for (Map.Entry<String, String> namespaceBinding : namespaceContext.getNamespaces().entrySet()) {
            staticQueryContext.declareNamespace(namespaceBinding.getKey(), namespaceBinding.getValue());
        }
    }

    
    @Override
    public <X extends XmlObjectNode> X selectSingleXmlObjectNode(
            XmlObject xmlObjectContextNode, String xqueryString, 
            Class<X> resultInterfaceClass) throws XQueryException {
        LOG.finest(String.format("Executing XQuery query '%s'.", xqueryString));
        XQueryExpression exp;
        try {
                exp = staticQueryContext.compileQuery(xqueryString);
            DynamicQueryContext dynamicContext = new DynamicQueryContext(configuration);
            dynamicContext.setContextItem(new XmlObjectNodeInfo(xmlObjectContextNode, configuration));
            Object result = exp.evaluateSingle(dynamicContext);
            if (result == null) {
                LOG.finest(String.format("Executed XQuery query successfully. 0 node found."));
                return null;
            } else if (result instanceof XmlObjectNodeInfo) {
                XmlObjectNode xon = ((XmlObjectNodeNodeInfo) result).getXmlObjectNode();
                if (!resultInterfaceClass.isInstance(xon)) {
                    throw new UncheckedException(
                            String.format(
                                    "XQuery expression '%s' does not select a single '%s' XML object node.",
                                    xqueryString,
                                    resultInterfaceClass.getName()));
                }
                LOG.finest(String.format("Executed XQuery expression successfully. 1 node found."));
                return resultInterfaceClass.cast(xon);
            } else {
                throw new UncheckedException(
                        String.format(
                                "XQuery expression '%s' does not select a single XML object node.",
                                xqueryString));
            }
        } catch (XPathException xpe) {
            throw new XQueryException(
                    String.format("Error while evaluating XQuery expression '%s'.", xqueryString), xpe);
        }
    }
    

    @SuppressWarnings("unchecked")
    @Override
    public <X extends XmlObjectNode> X[] selectXmlObjectNodes(XmlObject xmlObjectContextNode, String xqueryString, 
            Class<X> resultInterfaceClass) throws XQueryException {
        LOG.finest(String.format("Executing XQuery query '%s'.", xqueryString));
        List<X> listResult = new ArrayList<X>();
        XQueryExpression exp;
        try {
                exp = staticQueryContext.compileQuery(xqueryString);
            DynamicQueryContext dynamicContext = new DynamicQueryContext(configuration);
            dynamicContext.setContextItem(new XmlObjectNodeInfo(xmlObjectContextNode, configuration));
            List<?> result = exp.evaluate(dynamicContext);
            for (Object item : result) {
                if (item instanceof XmlObjectNodeNodeInfo) {
                    XmlObjectNode xon = ((XmlObjectNodeNodeInfo) item)
                            .getXmlObjectNode();
                    if (!resultInterfaceClass.isInstance(xon)) {
                        throw new UncheckedException(
                                String.format(
                                        "XQuery expression '%s' does not select '%s' XML object nodes.",
                                        xqueryString,
                                        resultInterfaceClass.getName()));
                    }
                    listResult.add(resultInterfaceClass.cast(xon));
                } else {
                    throw new UncheckedException(
                            String.format(
                                    "XQuery expression '%s' does not select XML object nodes.",
                                    xqueryString));
                }
            }       
        } catch (XPathException xpe) {
            throw new XQueryException(
                    String.format("Error while evaluating XQuery expression '%s'.", xqueryString), xpe);
        }
        LOG.finest(String.format("Executed XQuery query successfully. '%s' node(s) found.", listResult.size()));

        return listResult.toArray((X[]) Array.newInstance(resultInterfaceClass, listResult.size()));
    }
    
    
    @Override
    public List<?> evaluate(XmlObject xmlObjectContextNode, String xqueryString) throws XQueryException {
        LOG.finest(String.format("Executing XQuery query '%s'.", xqueryString));
        XQueryExpression exp;
        try {
                exp = staticQueryContext.compileQuery(xqueryString);
            DynamicQueryContext dynamicContext = new DynamicQueryContext(configuration);
            dynamicContext.setContextItem(new XmlObjectNodeInfo(xmlObjectContextNode, configuration));
            List<?> result = exp.evaluate(dynamicContext);
            LOG.finest(String.format("Executed XQuery expression successfully."));
            return result;
        } catch (XPathException xpe) {
            throw new XQueryException(
                    String.format("Error while evaluating XQuery expression '%s'.", xqueryString), xpe);
        }
        
        
    }
    
        
}
