/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;

import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easycommons.lang.ArrayHelper;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

public abstract class AbstractXmlObjectNodeImpl<Model extends ModelObject> implements XmlObjectNode {

    private final XmlContext xmlContext;

    private Model modelObject;

    private Object userData;

    private int xmlObjectIndex;

    private Integer[] xmlObjectBaseIndexes;

    protected AbstractXmlObjectNodeImpl(XmlContext xmlContext, Model modelObject) {
        assert xmlContext != null;
        this.xmlContext = xmlContext;
        this.userData = null;
        this.xmlObjectIndex = -1;
        this.xmlObjectBaseIndexes = null;
        setModelObject(modelObject);
    }

    public final XmlContext getXmlContext() {
        return xmlContext;
    }

    public Model getModelObject() {
        return modelObject;
    }

    protected final void setModelObject(Model modelObject) {
        if (modelObject == null) {
            modelObject = createCompliantModel();
        }
        this.modelObject = modelObject;
        if (getModelObject() != null) {
            ((AbstractModelObject) getModelObject()).setXmlObject(this);
        }
    }

    protected abstract Model createCompliantModel();

    @Override
    public final int hashCode() {
        // TODO
        return super.hashCode();
    }

    @Override
    public abstract boolean equals(Object obj);

    private void getXmlObjectBaseIndexes(List<Integer> indexes) {
        AbstractXmlObjectNodeImpl<?> parent = (AbstractXmlObjectNodeImpl<?>) getXmlObjectParent();
        if(parent != this) {
            if ((parent == null) || (parent.getXmlObjectBaseURI() != getXmlObjectBaseURI()) ) {
                indexes.add(0);
            } else {
                parent.getXmlObjectBaseIndexes(indexes);
                indexes.add(getXmlObjectIndex());
            }
        }
    }

    @Override
    public Integer[] getXmlObjectBaseIndexes() {
        if (xmlObjectBaseIndexes == null) {
            List<Integer> indexes = new ArrayList<Integer>();
            getXmlObjectBaseIndexes(indexes);
            xmlObjectBaseIndexes = indexes.toArray(new Integer[indexes.size()]);
        }
        return xmlObjectBaseIndexes;
    }

    @Override
    public final String getXmlObjectBaseXPath() {
        StringBuffer sb = new StringBuffer();
        for (int i : getXmlObjectBaseIndexes()) {
            sb.append("/*[").append(i + 1).append("]");
        }
        return sb.toString();
    }

    @Override
    public final XmlObject getXmlObjectBaseRoot() {
        XmlObject parent;
        URI currentURI;
        XmlObjectNode current = this;
        while (true) {
            parent = current.getXmlObjectParent();
            if (parent == null) {
                return (XmlObject) current;
            }
            currentURI = getXmlObjectBaseURI();
            if ((currentURI != null) && (!currentURI.equals(parent.getXmlObjectBaseURI()))) {
                return (XmlObject) current;
            }
            current = parent;
        }
    }

    @Override
    public final int getXmlObjectIndex() {
        if (xmlObjectIndex == -1) {
            xmlObjectIndex = guessIndexFromParent(getXmlObjectParent());
        }
        return xmlObjectIndex;

    }

    @Override
    public final XmlObjectNode getXmlObjectFollowingSibling() {
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return null;
        XmlObjectNode[] children = parent.getXmlObjectChildren();
        boolean found = false;
        for (XmlObjectNode child : children) {
            if (found)
                return child;
            if (child == this) {
                found = true;
            }
        }
        return null;
    }

    @Override
    public final XmlObjectNode getXmlObjectPrecedingSibling() {
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return null;
        XmlObjectNode[] children = parent.getXmlObjectChildren();
        XmlObjectNode previous = null;
        for (XmlObjectNode child : children) {
            if (child == this) {
                return previous;
            }
            previous = child;
        }
        return null;
    }

    @Override
    public final XmlObject[] getXmlObjectAncestors() {
        LinkedList<XmlObject> ancestors = new LinkedList<XmlObject>();
        XmlObjectNode current = this;
        XmlObject parent = null;
        while (true) {
            parent = current.getXmlObjectParent();
            if (parent == null)
                break;
            ancestors.addFirst(parent);
            current = parent;
        }
        return ancestors.toArray(new XmlObject[ancestors.size()]);
    }

    private void appendXmlObjectDescendants(XmlObjectNode xo, List<XmlObjectNode> result,
            IdentityHashMap<XmlObjectNode, Object> descendantsMap) {
        XmlObjectNode[] children = xo.getXmlObjectChildren();
        for (XmlObjectNode child : children) {
            if (descendantsMap.containsKey(child))
                continue;
            descendantsMap.put(child, null);
            result.add(child);
            appendXmlObjectDescendants(child, result, descendantsMap);
        }
    }

    @Override
    public final XmlObjectNode[] getXmlObjectDescendants() {
        List<XmlObjectNode> descendants = new LinkedList<XmlObjectNode>();
        appendXmlObjectDescendants(this, descendants, new IdentityHashMap<XmlObjectNode, Object>());
        return descendants.toArray(new XmlObjectNode[descendants.size()]);
    }

    @Override
    public final XmlObject getXmlObjectParent() {
        if (getModelObject() == null)
            return null;
        if (getModelObject().getParent() == null)
            return null;
        return (XmlObject) getXmlContext().getXmlObjectFactory().wrap(getModelObject().getParent());
    }

    @Override
    public final XmlObjectNode[] getXmlObjectFollowing() {
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return EMPTY_ARRAY;
        XmlObjectNode[] descendants = getXmlObjectDescendants();
        XmlObjectNode lastXmlObjectBeforeFollowing = this;
        if (descendants.length != 0) {
            lastXmlObjectBeforeFollowing = descendants[descendants.length - 1];
        }
        XmlObjectNode[] rootDescendants = getXmlObjectRoot().getXmlObjectDescendants();
        List<XmlObjectNode> following = new LinkedList<XmlObjectNode>();
        boolean lastXmlObjectBeforeFollowingFound = false;
        for (XmlObjectNode descendantXmlObject : rootDescendants) {
            if (descendantXmlObject == lastXmlObjectBeforeFollowing) {
                lastXmlObjectBeforeFollowingFound = true;
                continue;
            }
            if (lastXmlObjectBeforeFollowingFound == false)
                continue;
            following.add(descendantXmlObject);
        }
        return following.toArray(new XmlObjectNode[following.size()]);
    }

    @Override
    public final XmlObjectNode[] getXmlObjectFollowingSiblings() {
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return EMPTY_ARRAY;
        XmlObjectNode[] children = parent.getXmlObjectChildren();
        boolean found = false;
        List<XmlObjectNode> followingSiblings = new LinkedList<XmlObjectNode>();
        for (XmlObjectNode child : children) {
            if (found) {
                followingSiblings.add(child);
            }
            if (child == this) {
                found = true;
            }
        }
        return followingSiblings.toArray(new XmlObjectNode[followingSiblings.size()]);
    }

    @Override
    public final XmlObjectNode[] getXmlObjectPrecedingSiblings() {
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return EMPTY_ARRAY;
        XmlObjectNode[] children = parent.getXmlObjectChildren();
        List<XmlObjectNode> precedingSiblings = new LinkedList<XmlObjectNode>();
        for (XmlObjectNode child : children) {
            if (child == this)
                break;
            precedingSiblings.add(child);
        }
        return precedingSiblings.toArray(new XmlObjectNode[precedingSiblings.size()]);
    }

    @Override
    public final XmlObjectNode[] getXmlObjectPreceding() {
        // find all descendants of root and keep everything but ancestors, until
        // self.
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return EMPTY_ARRAY;
        XmlObjectNode lastXmlObjectAfterPreceding = this;
        XmlObjectNode[] rootDescendants = getXmlObjectRoot().getXmlObjectDescendants();
        List<XmlObjectNode> preceding = new LinkedList<XmlObjectNode>();
        XmlObjectNode[] ancestors = getXmlObjectAncestors();
        for (XmlObjectNode descendantXmlObject : rootDescendants) {
            if (descendantXmlObject == lastXmlObjectAfterPreceding) {
                break;
            }
            if (ArrayHelper.arrayContainsSameObject(ancestors, descendantXmlObject))
                continue;
            preceding.add(descendantXmlObject);
        }
        return preceding.toArray(new XmlObjectNode[preceding.size()]);
    }

    @Override
    public final XmlObjectNode[] getXmlObjectPrecedingOrAncestor() {
        // find all descendants of root and keep everything but ancestors, until
        // self.
        XmlObject parent = getXmlObjectParent();
        if (parent == null)
            return EMPTY_ARRAY;
        XmlObjectNode lastXmlObjectAfterPreceding = this;
        XmlObject root = getXmlObjectRoot();
        XmlObjectNode[] rootDescendants = root.getXmlObjectDescendants();
        List<XmlObjectNode> precedingOrAncestor = new LinkedList<XmlObjectNode>();
        precedingOrAncestor.add(root);
        for (XmlObjectNode descendantXmlObject : rootDescendants) {
            if (descendantXmlObject == lastXmlObjectAfterPreceding) {
                break;
            }
            precedingOrAncestor.add(descendantXmlObject);
        }
        return precedingOrAncestor.toArray(new XmlObjectNode[precedingOrAncestor.size()]);
    }

    @Override
    public final XmlObject getXmlObjectRoot() {
        XmlObject parent;
        XmlObjectNode current = this;
        while (true) {
            parent = current.getXmlObjectParent();
            if (parent == null)
                return (XmlObject) current;
            current = parent;
        }
    }

    @Override
    public final int compareTo(XmlObjectNode other) {
        Integer[] indexes = getXmlObjectBaseIndexes();
        Integer[] otherIndexes = other.getXmlObjectBaseIndexes();
        for (int i = 0; i < indexes.length; ++i) {
            int index = indexes[i];
            if (otherIndexes.length == i) {
                // other is inferior because path is shorter.
                return 1;
            }
            int otherIndex = otherIndexes[i];
            if (otherIndex < index) {
                return 1;
            }
            if (index < otherIndex) {
                return -1;
            }
        }
        if (otherIndexes.length > indexes.length) {
            return -1;
        }
        return 0;
    }

    @Override
    public abstract String toString();

    private final int guessIndexFromParent(XmlObject newParent) {
        if (newParent == null)
            return 0;
        XmlObjectNode[] children = newParent.getXmlObjectNaturalChildren();
        int i = 0;
        for (XmlObjectNode child : children) {
            if (child == this) {
                return i;
            }
            ++i;
        }
        children = newParent.getXmlObjectAdoptedChildren();
        i = 0;
        for (XmlObjectNode child : children) {
            if (child == this) {
                return i;
            }
            ++i;
        }
        throw new UncheckedException("XML object has a parent but no index!");
    }

    final void setAdoptiveParent(XmlObject parent, int adoptionIndex) {
        xmlObjectIndex = parent.getXmlObjectNaturalChildren().length + adoptionIndex;
        onAdoptiveParentChange(parent);
    }

    final void setNaturalParent(XmlObject parent) {
        xmlObjectIndex = -1;
        onNaturalParentChange(parent);
    }

    // TODO : this is not fair since extending this callback without invoking
    // super
    // can corrupt core behaviour.
    protected void onNaturalParentChange(XmlObject parent) {
    }

    // TODO : this is not fair since extending this callback without invoking
    // super
    // can corrupt core behaviour.
    protected void onAdoptiveParentChange(XmlObject parent) {
    }

    public XmlObjectNode[] getXmlObjectChildren() {
        final XmlObjectNode[] naturalChildren = getXmlObjectNaturalChildren();
        final XmlObjectNode[] adoptedChildren = getXmlObjectAdoptedChildren();
        if (adoptedChildren.length == 0) {
            return naturalChildren;
        }
        return ArrayHelper.mergeArrays(adoptedChildren, naturalChildren);
    }

    public XmlObjectNode[] getXmlObjectNaturalChildren() {
        return XmlObjectNode.EMPTY_ARRAY;
    }

    public XmlObjectNode[] getXmlObjectAdoptedChildren() {
        return XmlObjectNode.EMPTY_ARRAY;
    }

    @Override
    public String getXmlObjectTextContent() {
        XmlObjectNode[] descendants = getXmlObjectDescendants();
        StringBuffer sb = new StringBuffer();
        // use descendants rather than children to prevent infinite recursion...
        for (XmlObjectNode xon : descendants) {
            Object value = xon.getXmlObjectValue();
            sb.append(value == null ? "" : String.valueOf(value));
        }
        return sb.toString();
    }

    @Override
    public final boolean hasUserData() {
        return userData != null;
    }

    @Override
    public Object getUserData() {
        assert userData != null;
        return userData;
    }

    @Override
    public void setUserData(Object userData) {
        this.userData = userData;
    }

}
