/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.w3c.dom.Element;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.lang.reflect.ReflectionHelper;

public final class JaxbDuplicator {

    private final boolean allowTypeMorphing;
    private final IdentityHashMap<AbstractJaxbModelObject, AbstractJaxbModelObject> alreadyDuplicatedMap;

    
    private JaxbDuplicator(boolean allowTypeMorphing) {
        this.allowTypeMorphing = allowTypeMorphing;
        alreadyDuplicatedMap = new IdentityHashMap<AbstractJaxbModelObject, AbstractJaxbModelObject>();
    }

    @SuppressWarnings("unchecked")
    public static <T extends AbstractJaxbModelObject> T duplicate(T originalObject) {
        return (T) duplicateAs(originalObject, originalObject.getClass());
    }

    @SuppressWarnings("unchecked")
    public static <T extends AbstractJaxbModelObject> T duplicateAs(AbstractJaxbModelObject originalObject, Class<T> targetClass) {
        JaxbDuplicator jaxbDuplicator = new JaxbDuplicator(true);
        return (T) jaxbDuplicator.doDuplicateModelObject(originalObject, targetClass);
    }
    
    
    private static boolean isImmutableObject(Object originalObject) {
        if (originalObject instanceof Number)
            return true;
        if (originalObject instanceof Boolean)
            return true;
        if (originalObject instanceof Enum<?>)
            return true;
        return false;
    }

    @SuppressWarnings("unchecked")
    private Object doDuplicate(Object originalObject) {
        if (originalObject == null) {
            return null;

        } else if (isImmutableObject(originalObject)) {
            return originalObject;

        } else if (originalObject instanceof AbstractJaxbModelObject) {
            return doDuplicateModelObject((AbstractJaxbModelObject) originalObject);
        } else if (originalObject instanceof List<?>) {
            return doDuplicateList((List<Object>) originalObject);
        } else if (originalObject instanceof Map<?, ?>) {
            return doDuplicateMap((Map<Object, Object>) originalObject);
        } else if (originalObject instanceof Set<?>) {
            return doDuplicateSet((Set<Object>) originalObject);
        } else if (originalObject instanceof JAXBElement<?>) {
            return doDuplicateJAXBElement((JAXBElement<?>) originalObject);
        } else if (originalObject instanceof QName) {
            return doDuplicateQName((QName) originalObject);
        } else if (originalObject instanceof String) {
            return doDuplicateString((String) originalObject);
        } else if (originalObject instanceof Element) {
            Element originalElement = (Element) originalObject;
            Element clonedElement = (Element) originalElement.cloneNode(true);
            return clonedElement;
        } else {
            throw new UncheckedException(String.format(
                    "Cannot duplicate object of class '%s'.", originalObject
                            .getClass().getSimpleName()));
        }

    }

    private QName doDuplicateQName(QName originalObject) {
        return new QName(originalObject.getNamespaceURI(), originalObject
                .getLocalPart(), originalObject.getPrefix());
    }

    private String doDuplicateString(String originalObject) {
        return new String(originalObject);
    }

    @SuppressWarnings("unchecked")
    private List<?> doDuplicateList(List<Object> originalObject) {
        List<Object> duplicateObject;
        try {
            duplicateObject = ReflectionHelper.newInstance(originalObject
                    .getClass());
        } catch (InvocationTargetException e) {
            throw new UncheckedException("Cannot duplicate list.");
        }
        for (final Object element : originalObject) {
            final Object duplicateElement = doDuplicate(element);
            duplicateObject.add(duplicateElement);
        }
        return duplicateObject;
    }

    @SuppressWarnings("unchecked")
    private Set<?> doDuplicateSet(Set<Object> originalObject) {
        final Set<Object> duplicateObject;
        try {
            duplicateObject = ReflectionHelper.newInstance(originalObject
                    .getClass());
        } catch (InvocationTargetException e) {
            throw new UncheckedException("Cannot duplicate set.");
        }
        for (final Object element : originalObject) {
            final Object duplicateElement = doDuplicate(element);
            duplicateObject.add(duplicateElement);
        }
        return duplicateObject;
    }

    @SuppressWarnings("unchecked")
    private Map<?, ?> doDuplicateMap(Map<Object, Object> originalObject) {
        Map<Object, Object> duplicateObject;
        try {
            duplicateObject = ReflectionHelper.newInstance(originalObject
                    .getClass());
        } catch (InvocationTargetException e) {
            throw new UncheckedException("Cannot duplicate map.");
        }
        for (final Map.Entry<Object, Object> entry : originalObject.entrySet()) {
            duplicateObject.put(doDuplicate(entry.getKey()), doDuplicate(entry
                    .getValue()));
        }
        return duplicateObject;
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    private JAXBElement<?> doDuplicateJAXBElement(JAXBElement<?> originalObject) {
        if (originalObject == null)
            return null;
        final JAXBElement<?> sourceElement = (JAXBElement<?>) originalObject;
        final Object sourceObject = sourceElement.getValue();
        final Object copyObject = doDuplicate(sourceObject);
        final JAXBElement copyElement = new JAXBElement(
                sourceElement.getName(), sourceElement.getDeclaredType(),
                sourceElement.getScope(), copyObject);
        return copyElement;
    }

    
    private AbstractJaxbModelObject doDuplicateModelObject(
            AbstractJaxbModelObject originalModelObject) {
        return doDuplicateModelObject(originalModelObject, originalModelObject.getClass());
    }
    
    
    private AbstractJaxbModelObject doDuplicateModelObject(
            AbstractJaxbModelObject originalModelObject, Class<? extends AbstractJaxbModelObject> targetClass) {
        if (alreadyDuplicatedMap.containsKey(originalModelObject)) {
            return alreadyDuplicatedMap.get(originalModelObject);
        }
        Class<? extends AbstractJaxbModelObject> originalClass = (Class<? extends AbstractJaxbModelObject>) originalModelObject
                .getClass();
        AbstractJaxbModelObject duplicateModelObject;

        try {
            duplicateModelObject = ReflectionHelper.newInstance(targetClass);
        } catch (InvocationTargetException e) {
            throw new UncheckedException(
                    String
                            .format(
                                    "Cannot create new instance of class '%s' (InvocationTargetException).",
                                    targetClass.getSimpleName()), e);
        }

        // keep reference of duplicate in map.
        alreadyDuplicatedMap.put(originalModelObject, duplicateModelObject);

        // duplicate JAXB part using reflection.
        JaxbReflectionInfo reflectionInfo = JaxbReflectionInfo
                .getReflectionInfo(originalClass);
        for (JaxbReflectionFieldInfo fieldInfo : reflectionInfo.getFieldInfos()) {
            Object originalValue = ReflectionHelper.getFieldValue(
                    originalModelObject, fieldInfo.getField());
            Object duplicateValue = doDuplicate(originalValue);
            try {
                fieldInfo.getField().set(duplicateModelObject, duplicateValue);
            } catch (IllegalArgumentException e) {
                if (!allowTypeMorphing)
                    throw new UncheckedException(
                            String
                                    .format(
                                            "Cannot set reflectively value of field '%s' on class '%s' (IllegalArgumentException).",
                                            fieldInfo.getField().getName(), targetClass.getSimpleName()), e);
            } catch (IllegalAccessException e) {
                if (!allowTypeMorphing)
                    throw new UncheckedException(
                            String
                                    .format(
                                            "Cannot set reflectively value of field '%s' on class '%s' (IllegalAccessException).",
                                            fieldInfo.getField().getName(), targetClass.getSimpleName()), e);
            }    
        }

        // duplicate non JAXB part.
        
        // TODO : for binder : eventually go to the root, marshall as a new Node with binder
        // or go to the root duplicate DOM node and unmarshall back with a new binder. 
        //duplicateModelObject.setBinder(null);
        duplicateModelObject.setXmlObject(null);
        String originalURIStr = String
                .valueOf(originalModelObject.getBaseURI());
        if (originalURIStr != null) {
            try {
                duplicateModelObject.setBaseURI(new URI(String
                        .valueOf(originalURIStr)));
            } catch (URISyntaxException e) {
                throw new UncheckedException(String.format(
                        "Cannot duplicate URI '%s'.", originalURIStr));
            }
        }
        JAXBElement<?> originalJAXBElement = originalModelObject
                .getJAXBElement();
        if (originalJAXBElement != null) {
            assert originalJAXBElement.getValue() == originalModelObject;
            duplicateModelObject
                    .setJaxbElement(doDuplicateJAXBElement(originalJAXBElement));
        }
        // parent hierarchy is NOT duplicated !
        duplicateModelObject.setNaturalParent(null);
        duplicateModelObject.setAdoptiveParent(null);

        return duplicateModelObject;

    }

}
