/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;

import net.sf.saxon.Configuration;
import net.sf.saxon.xpath.XPathEvaluator;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.XmlObjectXPathEvaluator;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

final class XmlObjectXPathEvaluatorImpl implements XmlObjectXPathEvaluator {

    private static Logger LOG = Logger.getLogger(XmlContextImpl.class
            .getName());
    
    private final Configuration configuration;
    private final NamespaceContext namespaceContext;

    XmlObjectXPathEvaluatorImpl(NamespaceContext namespaceContext) {
        this.configuration = new Configuration();
        this.namespaceContext = namespaceContext;
    }

    @Override
    public <X extends XmlObjectNode> X selectSingleXmlObjectNode(
            XmlObject xmlObjectContextNode, String xpathStr, Class<X> resultInterfaceClass)
            throws XPathExpressionException {
        LOG.finest(String.format("Executing XPath query '%s'.", xpathStr));
        XPathEvaluator eval = new XPathEvaluator(configuration);
        eval.setNamespaceContext(namespaceContext);
        XmlObjectNodeNodeInfo nodeInfo = NodeInfoHelper
                .wrapXmlObjectAsNodeInfo(xmlObjectContextNode, configuration);
        try {
            Object result = eval.evaluate(xpathStr, nodeInfo,
                    XPathConstants.NODE);
            if (result == null) {
                LOG.finest(String.format("Executed XPath query successfully. 0 node found."));
                return null;
            } else if (result instanceof XmlObjectNodeInfo) {
                XmlObjectNode xon = ((XmlObjectNodeNodeInfo) result).getXmlObjectNode();
                if (!resultInterfaceClass.isInstance(xon)) {
                    throw new UncheckedException(
                            String.format(
                                    "XPath expression '%s' does not select a single '%s' XML object node.",
                                    xpathStr,
                                    resultInterfaceClass.getName()));
                }
                LOG.finest(String.format("Executed XPath query successfully. 1 node found."));
                return resultInterfaceClass.cast(xon);
            } else {
                throw new UncheckedException(
                        String.format(
                                "XPath expression '%s' does not select a single XML object node.",
                                xpathStr));
            }
        } catch (XPathExpressionException xpee) {
            throw new UncheckedException(xpee);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public <X extends XmlObjectNode> X[] selectXmlObjectNodes(
            XmlObject xmlObjectContextNode, String xpathStr,
            Class<X> resultInterfaceClass) throws XPathExpressionException {
        List<X> listResult = new ArrayList<X>();
        LOG.finest(String.format("Executing XPath query '%s'.", xpathStr));
        XPathEvaluator eval = new XPathEvaluator(configuration);
        eval.setNamespaceContext(namespaceContext);
        XmlObjectNodeNodeInfo nodeInfo = NodeInfoHelper
                .wrapXmlObjectAsNodeInfo(xmlObjectContextNode, configuration);
        Object result = eval.evaluate(xpathStr, nodeInfo,
                XPathConstants.NODESET);
        if (result instanceof List<?>) {
            List<?> list = (List<?>) result;
            for (Object item : list) {
                if (item instanceof XmlObjectNodeNodeInfo) {
                    XmlObjectNode xon = ((XmlObjectNodeNodeInfo) item)
                            .getXmlObjectNode();
                    if (!resultInterfaceClass.isInstance(xon)) {
                        throw new UncheckedException(
                                String.format(
                                        "XPath expression '%s' does not select '%s' XML object nodes.",
                                        xpathStr,
                                        resultInterfaceClass.getName()));
                    }
                    listResult.add(resultInterfaceClass.cast(xon));
                } else {
                    throw new UncheckedException(
                            String.format(
                                    "XPath expression '%s' does not select XML object nodes.",
                                    xpathStr));
                }
            }
        }
        LOG.finest(String.format("Executed XPath query successfully. %d node(s) found.", listResult.size()));
        
        return listResult.toArray((X[]) Array.newInstance(resultInterfaceClass, listResult.size()));
    }


}
