/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;

import javax.xml.bind.annotation.XmlAnyAttribute;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;

import com.ebmwebsourcing.easycommons.lang.reflect.ReflectionHelper;

final class JaxbReflectionFieldInfo {

	static enum Type { CHILD, ATTRIBUTE }

	private final Field field;
	private final Type type;
	private final String name;
	private final Method getterMethod;
	private final Method setterMethod;

	public JaxbReflectionFieldInfo(Field field, Type type, String name) {
		this.field = field;
		this.type = type;
		this.name = name;
		this.getterMethod = guessGetterMethod(field);
		this.setterMethod = guessSetterMethod(field);
	}

	private static Method guessGetterMethod(Field field) {
		String prefix = "get";
		if (Boolean.class.equals(field.getType()) || boolean.class.equals(field.getType())) {
			prefix = "is";
		}
		return guessPrefixedMethod(field, prefix);
	}


	private static Method guessSetterMethod(Field field) {
		if (Collection.class.isAssignableFrom(field.getType())) return null;
		if (field.isAnnotationPresent(XmlAnyAttribute.class)) return null;
		Class<?> fieldType = field.getType();
		if (fieldType == Boolean.class) fieldType = boolean.class;
		if (fieldType == Byte.class) fieldType = byte.class;
		if (fieldType == Short.class) fieldType = short.class;
		if (fieldType == Integer.class) fieldType = int.class;
		if (fieldType == Long.class) fieldType = long.class;
		if (fieldType == Double.class) fieldType = double.class;
		if (fieldType == Float.class) fieldType = float.class;
		return guessPrefixedMethod(field, "set", fieldType);
	}
	
	
	private static Method guessPrefixedMethod(Field field, String prefix, Class<?>... parameterTypes) {
		assert field != null;

		XmlElement elmtAnnotation = field.getAnnotation(XmlElement.class);
		XmlAttribute attAnnotation = field.getAnnotation(XmlAttribute.class);
		String fieldName = null;
		if(elmtAnnotation != null && elmtAnnotation.name() != null 
				&& !elmtAnnotation.name().equals(JaxbReflectionInfo.JAXB_ATTRIBUTE_NAME_DEFAULT)
				&& !elmtAnnotation.name().contains("-")) {
			fieldName = elmtAnnotation.name();
		} 
		else if (attAnnotation != null && attAnnotation.name()!=null && 
				!attAnnotation.name().equals(JaxbReflectionInfo.JAXB_ATTRIBUTE_NAME_DEFAULT) && !attAnnotation.name().contains("-")) {
			fieldName = attAnnotation.name();
		}
		else {
			fieldName = field.getName().replaceAll("^_*", "");
		}
		if (fieldName.length() == 1) {
			return ReflectionHelper.getPublicMethod(field.getDeclaringClass(), 
					prefix + fieldName.substring(0, 1).toUpperCase(), parameterTypes);
		} else {
			return ReflectionHelper.getPublicMethod(field.getDeclaringClass(), prefix + 
					toCamelCase(fieldName), 
					parameterTypes);
		}    
	}


	public Field getField() {
		return field;
	}

	public Type getType() {
		return type;
	}

	public String getName() {
		return name;
	}

	public Method getGetterMethod() {
		return getterMethod;
	}

	public Method getSetterMethod() {
		return setterMethod;
	}
	
	static String toCamelCase(String s){
		   String[] parts = s.split("_");
		   String camelCaseString = "";
		   for (String part : parts){
		      camelCaseString = camelCaseString + toProperCase(part);
		   }
		   return camelCaseString;
		}

	static String toProperCase(String s) {
	    return s.substring(0, 1).toUpperCase() + s.substring(1);
	}



}
