/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;

import javax.xml.bind.Binder;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.ebmwebsourcing.easybox.api.ModelObject;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectReader;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

final class XmlObjectReaderImpl implements XmlObjectReader {

    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY;

    static {
        DOCUMENT_BUILDER_FACTORY = DocumentBuilderFactory.newInstance();
        DOCUMENT_BUILDER_FACTORY.setNamespaceAware(true);
    }

    private XmlContextImpl xmlContext;
    private ReadMode readMode;
    private EntityResolver entityResolver;

    public XmlObjectReaderImpl(XmlContextImpl context) {
        this.xmlContext = context;
        this.readMode = ReadMode.FULL;

    }

    
    @Override
    public void setEntityResolver(EntityResolver entityResolver) {
        this.entityResolver = entityResolver;
    }
    
    
    private DocumentBuilder createDocumentBuilder() {
        try {
            // TODO : to be optimized.
            DocumentBuilder documentBuilder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            if (entityResolver != null) {
                documentBuilder.setEntityResolver(entityResolver);
            }
            return documentBuilder;
        } catch (ParserConfigurationException e) {
            throw new UncheckedException(
                    "Cannot create DocumentBuilder (ParserConfigurationException).",
                    e);
        } 
    }
    

    @Override
    public ReadMode getReadMode() {
        return readMode;
    }

    @Override
    public void setReadMode(ReadMode readMode) {
        this.readMode = readMode;
    }

    private AbstractJaxbModelObject asAbstractJaxbObject(Object o) {
        AbstractJaxbModelObject ajo = null;
        if (o instanceof JAXBElement<?>) {
            ajo = (AbstractJaxbModelObject) ((JAXBElement<?>) o).getValue();
        } else if (!(o instanceof ModelObject)) {
            throw new UncheckedException("Object is not a model object!");
        } else {
            ajo = (AbstractJaxbModelObject) o;
        }
        return ajo;
    }

    private void finalizeReading(XmlObject xmlObject, String baseURI) {
        if (baseURI != null) {
            ((AbstractJaxbXmlObjectImpl<?>) xmlObject).setXmlObjectBaseURI(URI
                    .create(baseURI));
        }
        if (ReadMode.FULL.equals(readMode)) {
            readFully(xmlObject);
        }
    }

    private void readFully(XmlObject xmlObject) {
        xmlObject.getXmlObjectDescendants();
    }

    @Override
    public <X extends XmlObject> X readDocument(URL url,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        try {
            X xmlObject = readDocument(url.openStream(),
                    xmlObjectInterfaceClass, url.toString());
            return xmlObject;
        } catch (IOException ioe) {
            throw new XmlObjectReadException(ioe);
        }
    }

    @Override
    public <X extends XmlObject> X readDocument(InputStream inputStream,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        return readDocument(inputStream, xmlObjectInterfaceClass, null);
    }

    @SuppressWarnings("unchecked")
    public <X extends XmlObject> X readDocument(InputSource inputSource,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        try {
            DocumentBuilder documentBuilder = createDocumentBuilder();
            
            Binder<Node> binder = newBinder();
            Object unmarshalled = binder.unmarshal(documentBuilder.parse(inputSource));
            AbstractJaxbModelObject ajo = asAbstractJaxbObject(unmarshalled);
            ajo.setBinder(binder);

            XmlObject newXmlObject = (XmlObject) xmlContext.getXmlObjectFactory().wrap(ajo);
            finalizeReading(newXmlObject, inputSource.getSystemId());

            return (X) newXmlObject;
        } catch (JAXBException je) {
            throw new XmlObjectReadException(je);
        } catch (SAXException se) {
            throw new XmlObjectReadException(se);
        } catch (IOException ioe) {
            throw new XmlObjectReadException(ioe);
        }
    }

    private <X extends XmlObject> X readDocument(InputStream inputStream,
            Class<X> xmlObjectInterfaceClass, String systemId)
            throws XmlObjectReadException {
        InputSource inputSource = new InputSource(inputStream);
        inputSource.setSystemId(systemId);
        return readDocument(inputSource, xmlObjectInterfaceClass);
    }

    @Override
    public <X extends XmlObject> X readFragment(URL url,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        try {
            X xmlObject = readFragment(url.openStream(),
                    xmlObjectInterfaceClass, url.toString());
            return xmlObject;
        } catch (IOException ioe) {
            throw new XmlObjectReadException(ioe);
        }
    }

    @Override
    public <X extends XmlObject> X readFragment(InputStream inputStream,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        return readFragment(inputStream, xmlObjectInterfaceClass, null);
    }

    @SuppressWarnings("unchecked")
    public <X extends XmlObject> X readFragment(InputSource inputSource,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        try {
            DocumentBuilder documentBuilder = createDocumentBuilder();
            AbstractJaxbXmlObjectImpl<AbstractJaxbModelObject> newXmlObject = (AbstractJaxbXmlObjectImpl<AbstractJaxbModelObject>) xmlContext
                    .getXmlObjectFactory().create(xmlObjectInterfaceClass);
            // create a dummy model, but which should be valid from a type point
            // of view.
            Class<? extends AbstractJaxbModelObject> modelClass = newXmlObject
                    .getModelObject().getClass();

            Binder<Node> binder = newBinder();
            Object unmarshalled = binder.unmarshal(documentBuilder.parse(inputSource), modelClass);
            AbstractJaxbModelObject jaxbModelObject = asAbstractJaxbObject(unmarshalled);
            jaxbModelObject.setBinder(binder);
            newXmlObject.setModelObject(jaxbModelObject);

            finalizeReading(newXmlObject, inputSource.getSystemId());

            return xmlObjectInterfaceClass.cast(newXmlObject);
        } catch (JAXBException je) {
            throw new XmlObjectReadException(je);
        } catch (SAXException se) {
            throw new XmlObjectReadException(se);
        } catch (IOException ioe) {
            throw new XmlObjectReadException(ioe);
        }
    }

    private <X extends XmlObject> X readFragment(InputStream inputStream,
            Class<X> xmlObjectInterfaceClass, String systemId)
            throws XmlObjectReadException {
        InputSource inputSource = new InputSource(inputStream);
        inputSource.setSystemId(systemId);
        return readFragment(inputSource, xmlObjectInterfaceClass);
    }


    private Binder<Node> newBinder() {
        return xmlContext.newBinder();
    }

    
    @Override
    public <X extends XmlObject> X readDocument(Document document,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        try {
            Binder<Node> binder = newBinder();
            Object unmarshalled = binder.unmarshal(document);
            
            AbstractJaxbModelObject jaxbModelObject = asAbstractJaxbObject(unmarshalled);
            jaxbModelObject.setBinder(binder);

            XmlObject newXmlObject = (XmlObject) xmlContext.getXmlObjectFactory().wrap(jaxbModelObject);
            finalizeReading(newXmlObject, document.getBaseURI());

            return xmlObjectInterfaceClass.cast(newXmlObject);
        } catch (JAXBException je) {
            throw new XmlObjectReadException(je);
        }
    }

    


    @SuppressWarnings("unchecked")
    public <X extends XmlObject> X readFragment(Document document,
            Class<X> xmlObjectInterfaceClass) throws XmlObjectReadException {
        try {
            AbstractJaxbXmlObjectImpl<AbstractJaxbModelObject> newXmlObject = (AbstractJaxbXmlObjectImpl<AbstractJaxbModelObject>) xmlContext
                    .getXmlObjectFactory().create(xmlObjectInterfaceClass);
            // create a dummy model, but which should be valid from a type point
            // of view.
            Class<? extends AbstractJaxbModelObject> modelClass = newXmlObject
                    .getModelObject().getClass();

            Binder<Node> binder = newBinder();
            Object unmarshalled = binder.unmarshal(document, modelClass);
            AbstractJaxbModelObject jaxbModelObject = asAbstractJaxbObject(unmarshalled);
            jaxbModelObject.setBinder(binder);
            newXmlObject.setModelObject(jaxbModelObject);

            finalizeReading(newXmlObject, document.getBaseURI());

            return xmlObjectInterfaceClass.cast(newXmlObject);
        } catch (JAXBException je) {
            throw new XmlObjectReadException(je);
        }
    }  
}
