/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.impl;

import java.io.InputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ServiceLoader;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.ValidatorHandler;

import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectBinding;
import com.ebmwebsourcing.easybox.api.XmlObjectSchemaBinding;
import com.ebmwebsourcing.easybox.api.XmlObjectValidationException;
import com.ebmwebsourcing.easybox.api.XmlObjectValidator;
import com.ebmwebsourcing.easybox.api.XmlObjectWriteException;
import com.ebmwebsourcing.easybox.api.XmlObjectWriter;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

public class XmlObjectValidatorImpl implements XmlObjectValidator {


	private static final class FailFastErrorHandler extends DefaultHandler {
		
		@Override
		public void error(SAXParseException e) throws SAXException {
			fatalError(e);
		}
		
		@Override
		public void fatalError(SAXParseException e) throws SAXException {
			throw new FailFastValidationException(e);
		}
	}
	

	private static final class FailFastValidationException extends RuntimeException {
		
		private static final long serialVersionUID = 633136600651546753L;

		public FailFastValidationException(SAXParseException e) {
			super(e);
		}
		
		@Override
		public SAXParseException getCause() {
			return (SAXParseException) super.getCause();
		}
	}
	
	private static final Schema schema = initializeSchema();
	
	
	private static Schema initializeSchema() {
		Schema schema = null;
		SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		ClassPathResourceResolver cprr = new ClassPathResourceResolver();
		ServiceLoader<XmlObjectBinding> loader = ServiceLoader
        .load(XmlObjectBinding.class);
		Iterator<XmlObjectBinding> it = loader.iterator();
		List<Source> schemaSources = new LinkedList<Source>();
		while (it.hasNext()) {
		    XmlObjectBinding xob = it.next();
		    if (!(xob instanceof XmlObjectSchemaBinding)) continue;
		    XmlObjectSchemaBinding xosb = (XmlObjectSchemaBinding) xob;
		    String schemaResourcePath = xosb.getOriginatingSchemaDir() + "/" + xosb.getOriginatingSchemaName(); 
			cprr.addOriginatingDir(xosb.getOriginatingSchemaDir());
			cprr.addOriginatingResourcePath(xosb.getOriginatingSchemaNamespaceURI(), schemaResourcePath);

			InputStream schemaInputStream = XmlObjectValidatorImpl.class.getClassLoader().getResourceAsStream(
			        schemaResourcePath);
			if (schemaInputStream == null) {
				// TODO : log error
				continue;
			}
			schemaSources.add(new StreamSource(schemaInputStream));
		}		
		schemaFactory.setResourceResolver(cprr);
		try {
			schema = schemaFactory.newSchema(schemaSources.toArray(new Source[schemaSources.size()]));
		} catch (SAXException e) {
			// TODO : log error;
			e.printStackTrace();
		}
		return schema;
	}

	private XmlContextImpl context;
	
	XmlObjectValidatorImpl(XmlContextImpl context) {
		this.context = context;
	}


	public void failFastValidate(XmlObject xmlObject) throws XmlObjectValidationException {
		FailFastErrorHandler failFastErrorHandler = new FailFastErrorHandler();
		try {
			validate(xmlObject, failFastErrorHandler);
		} catch (FailFastValidationException ffve) {
			throw new XmlObjectValidationException("xml object is not valid!", ffve.getCause());
		}
	}
	
	
	@Override
	public void validate(XmlObject xmlObject, ErrorHandler errorHandler) {
		ValidatorHandler validatorHandler = schema.newValidatorHandler();
		validatorHandler.setErrorHandler(errorHandler);
        XmlObjectWriter writer = context.createWriter();
		try {
            writer.writeDocument(xmlObject, validatorHandler);
        } catch (XmlObjectWriteException e) {
            throw new UncheckedException("Unexpected problem while  validating XmlObject.");
        }
	}
	
	
}
