/**
* easy BPEL software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easybpel.model.bpel.api.inout;

import java.net.URI;
import java.util.Map;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELException;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;

/**
 * This interface describes a collection of methods that enable conversion of a
 * WSDL document (in XML, following the WSDL schema described in the WSDL
 * specification) into a WSDL model.
 * 
 * @author Nicolas Salatge - eBM WebSourcing
 */
public interface BPELReader {

    /**
     * Constants for the Message Exchange Patterns.
     * 
     */
    public enum FeatureConstants {
        VERBOSE("com.ebmwebsourcing.easybpel.model.bpel.verbose"), IMPORT_DOCUMENTS(
                "com.ebmwebsourcing.easybpel.model.bpel.importDocuments");

        private String value = null;

        /**
         * Creates a new instance of {@link FeatureConstants}
         * 
         * @param value
         */
        private FeatureConstants(final String value) {
            this.value = value;
        }

        /**
         * 
         * @return
         */
        public String value() {
            return this.value;
        }

        /**
         * Please use this equals method instead of using :<code>
         * value().equals(value)
         * </code> which is
         * almost 10 times slower...
         * 
         * @param value
         * @return
         */
        public boolean equals(final String val) {
            return this.value().equals(val);
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return this.value;
        }
    }

    /**
     * Sets the specified feature to the specified value.
     * <p>
     * The minimum features that must be supported are:
     * <p>
     * <table border=1>
     * <tr>
     * <th>Name</th>
     * <th>DescriptionImpl</th>
     * <th>Default Value</th>
     * </tr>
     * <tr>
     * <td><center>com.ebmwebsourcing.easybpel.model.bpel.verbose</center></td>
     * <td>If set to true, status messages will be displayed.</td>
     * <td><center>type: boolean - default value: false</center></td>
     * </tr>
     * <tr>
     * <td><center>com.ebmwebsourcing.easybpel.model.bpel.importDocuments</center></td>
     * <td>If set to true, imported WSDL documents will be retrieved and
     * processed.</td>
     * <td><center>type: boolean - default value: true</center></td>
     * </tr>
     * <tr>
     * <td>
     * <center>com.ebmwebsourcing.easybpel.model.bpel.pathDirectoryOfImportLocations
     * </center></td>
     * <td>If the location is set, imported WSDL documents will be retrieved at
     * this location (Set the importDocuments Features at true).</td>
     * <td><center>type: String</center></td>
     * </tr>
     * </table>
     * <p>
     * All feature names must be fully-qualified, Java package style. All names
     * starting with om.ebmwebsourcing. are reserved for features defined by the
     * specification. It is recommended that implementation- specific features
     * be fully-qualified to match the package name of that implementation. For
     * example: com.abc.featureName
     * 
     * @param name
     *            the name of the feature to be set.
     * @param value
     *            the value to set the feature to.
     * @throws BPELException
     *             TODO
     * @see #getFeature(String)
     */
    void setFeature(FeatureConstants name, Object value) throws BPELException;

    /**
     * Gets the value of the specified feature.
     * 
     * @param name
     *            the name of the feature to get the value of.
     * @return the value of feature
     * @throws IllegalArgumentException
     *             if the feature name is not recognized.
     * @see #setFeature(String, boolean)
     */
    Object getFeature(FeatureConstants name);

    /**
     * Gets all features.
     * 
     * @return the features
     * @see #setFeature(String, boolean)
     */
    Map<FeatureConstants, Object> getFeatures();

    /**
     * Read the WSDL document accessible via the specified URI into a BPEL
     * definition.
     * 
     * @param wsdlURI
     *            a URI (can be a filename or URL) pointing to a BPEL XML
     *            definition.
     * @return the definition.
     */
    BPELProcess readBPEL(URI bpelURI) throws BPELException;

    /**
     * Read the specified WSDL document into a WSDL definition.
     * 
     * @param documentBaseURI
     *            the document base URI of the WSDL definition described by the
     *            document. Will be set as the documentBaseURI of the returned
     *            Definition. Can be null, in which case it will be ignored.
     * @param wsdlDocument
     *            the WSDL document, an XML document obeying the WSDL schema.
     * @return the definition described in the document.
     */
    BPELProcess readBPEL(Document bpelDocument) throws BPELException;

    /**
     * Read a BPEL document into a BPEL definition.
     * 
     * @param bpelURI
     *            uri of the bpel
     * @param inputSource
     *            an InputSource pointing to the WSDL document, an XML document
     *            obeying the WSDL schema.
     * @return the definition described in the document pointed to by the
     *         InputSource.
     */
    BPELProcess readBPEL(URI bpelURI, InputSource inputSource) throws BPELException;


}
