/**
 * easy BPEL software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */
package com.ebmwebsourcing.easybpmn.bpmn2bpel;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.xml.namespace.QName;

import org.jdom.JDOMException;
import org.jdom.output.DOMOutputter;
import org.ow2.easywsdl.schema.api.abstractElmt.AbstractSchemaElementImpl;
import org.petalslink.abslayer.Factory;
import org.petalslink.abslayer.service.api.Description;
import org.w3c.dom.Document;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.with.WithName;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELElement;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELElementImpl;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELException;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Activity;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Assign;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Empty;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Exit;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Flow;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.If;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Invoke;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Pick;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Receive;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Reply;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Scope;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Sequence;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Wait;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.While;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4pick.OnAlarm;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4pick.OnMessage;
import com.ebmwebsourcing.easybpel.model.bpel.api.partnerLink.PartnerLink;
import com.ebmwebsourcing.easybpel.model.bpel.api.wsdlImports.Import;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TAssign;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TBooleanExpr;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TCopy;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TDeadlineExpr;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TDurationExpr;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TEmpty;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TExit;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TFlow;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TFrom;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TIf;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TImport;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TInvoke;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TOnAlarmPick;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TOnMessage;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TPick;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TProcess;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TReceive;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TReply;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TScope;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TSequence;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TTo;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TVariable;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TWait;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TWhile;
import com.ebmwebsourcing.easybpel.model.bpel.impl.BPELFactoryImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.BPELProcessImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.AssignImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.EmptyImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ExitImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.FlowImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.IfImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.InvokeImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.PickImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ReceiveImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ReplyImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ScopeImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.SequenceImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.WaitImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.WhileImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.CopyImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.FromImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.ToImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4pick.OnAlarmImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4pick.OnMessageImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.variable.AbstractVariableImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.variable.ElementVariableImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.variable.TypeVariableImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.wsdlImports.ImportImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.api.BPELExpression;
import com.ebmwebsourcing.easybpel.xpath.exp.api.XPathExpressionException;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.BPELExpressionImpl;
import com.ebmwebsourcing.easybpmn.bpmn20.api.BPMNException;
import com.ebmwebsourcing.easybpmn.bpmn20.api.DefinitionsHelper;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Assignment;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.BaseElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.BoundaryEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.CallActivity;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.CatchEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Collaboration;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataInputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataOutputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EndEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Event;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventBasedGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ExclusiveGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowNode;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Gateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Interface;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateCatchEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateThrowEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ItemDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Message;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.MessageEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Operation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ParallelGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Participant;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Process;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ReceiveTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SendTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ServiceTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.StartEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Task;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TerminateEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ThrowEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TimerEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.ItemAwareElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithDefaultSequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithFlowElements;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.extension.ExtensionAdapter;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.wsdl.WSDLGenerator;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.wsdl.WSDLGeneratorHelper;
import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;
import com.ebmwebsourcing.easyplnk20.api.element.PartnerLinkType;
import com.ebmwebsourcing.easyschema.xsd2xml.XSD2XML;
import com.ebmwebsourcing.easyschema10.api.SchemaHelper;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;
import com.ebmwebsourcing.easywsdl11.api.Constants;



public class BPELGenerator  {
	
	/**
	 * Equivalent to {@link #generate(Definitions, String)} with String argument null
	 * @param def
	 * @return the list of BPEL processes generated from the given BPMN definitions
	 * @throws BPMNException
	 */
	public List<BPELProcess> generate(Definitions def) throws BPMNException {
		System.out.println("***** definitions are: "+ def.getId());
		return generate(def,null);
	}

	/**
     * Generates the BPEL processes corresponding to the given BPMN definitions
     * Also writes the BPEL and WSDL files into the given path
     * @param def
     * @param path If path is null no file will be written
     * @return the list of BPEL processes generated from the given BPMN definitions
     * @throws BPMNException
     */
	public List<BPELProcess> generate(Definitions def, String writingPath) throws BPMNException {
		List<BPELProcess> res = new ArrayList<BPELProcess>();
		GenerationProperties gp = new GenerationProperties(def);
		
		BPELGeneratorHelper.addDefaultExternalParticipant(def);
		
		WSDLGenerator wsdlGen = new WSDLGenerator();
		Set<com.ebmwebsourcing.easywsdl11.api.element.Definitions> descs = wsdlGen.generate(def, gp);
		Map<QName,Map<String, XmlObject>> schemaImports = new HashMap<QName, Map<String,XmlObject>>();
		
		for(Collaboration c: def.getCollaborations()) {
			for(Participant participant: c.getParticipant()) {
			    //generate the bpel process out of the bpmn process if it exists
			    if(!participant.hasProcessRef()) {
			        continue;
			    }
			    //and if it is not empty
			    else {
			        Process p = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(),participant.getProcessRef(), Process.class);
			        if(p.getFlowNode()==null || p.getFlowNode().length==0) {
			            continue;
			        }
			    }
			    
			    //generate the bpel process actual content
				BPELProcess bpel = this.generate(participant, gp);
				res.add(bpel);
				
				//add its imports : wsdl and schemas
				for(com.ebmwebsourcing.easywsdl11.api.element.Definitions defs : descs){
				    addDescriptionImport(bpel, defs);
				}
				
				schemaImports.put(bpel.getQName(), new HashMap<String, XmlObject>());
				for(QName elt : gp.getElementsToImport()){
                    List<Element> elts = bpel.getImports().findElementsInAllSchema(elt);
                    if(elts==null || elts.isEmpty()){
                        for(com.ebmwebsourcing.easybpmn.bpmn20.api.element.Import impt : def.getImports()){
                            if(impt.isXSDImport() && impt.getNamespace().equals(elt.getNamespaceURI())){
                                Schema s = impt.getSchema();
                                if(SchemaHelper.findElementByQName(s, elt)!=null) {
                                    schemaImports.get(bpel.getQName()).put(impt.getLocation(), s);
                                    addSchemaImport(bpel, s, impt.getLocation());
                                    break;
                                }
                            }
                            if(impt.isWSDL11Import() && impt.getNamespace().equals(elt.getNamespaceURI())) {
                                com.ebmwebsourcing.easywsdl11.api.element.Definitions wsdlDefs = impt.getWSDL11Definitions();
                                if(SchemaHelper.findElementByQName(wsdlDefs, elt)!=null) {
                                    schemaImports.get(bpel.getQName()).put(impt.getLocation(), wsdlDefs);
                                    addDescriptionImport(bpel, wsdlDefs);
                                    break;
                                }
                            }
                        }
                    }
                }
				
			}
		}
		
		//writes the BPEL files and their imports (recursively) into the output directory
		if(writingPath!=null){
		    try{
    		    for(com.ebmwebsourcing.easywsdl11.api.element.Definitions defs : descs){

    		        if(defs.hasTypes()){
    		        	for(Schema schema : defs.getTypes().getAnyXmlObjects(Schema.class)){
    		        		Map<String, Schema> impts = WSDLGeneratorHelper.findImportedSchemas(schema);
    		        		for(Entry<String, Schema> impt : impts.entrySet()){
    		        			BPELProcessWriterHelper.writeFile(writingPath, impt.getKey(), impt.getValue());
    		        		}
    		        	}
    		        }
    		        
    		        BPELProcessWriterHelper.writeFile(writingPath, defs.getName()+".wsdl", defs);
    		    }
    		    
    		    for(BPELProcess bpel : res){
                    Document doc = BPELFactoryImpl.getInstance().newBPELWriter().getDocument(bpel);
                    BPELProcessWriterHelper.writeFile(writingPath, bpel.getName()+".bpel", XMLPrettyPrinter.prettyPrint(doc));
                    
                    for(String location : schemaImports.get(bpel.getQName()).keySet()){
                        BPELProcessWriterHelper.writeFile(writingPath, BPELGeneratorHelper.getFileNameFromImportLocation(location), schemaImports.get(bpel.getQName()).get(location));
                    }
                }
		    }
		    catch(Exception e){
		        throw new BPMNException("A problem occured during the process writing.",e);
		    }
		}
		
		return res;
	}

	private BPELProcess generate(Participant participant, GenerationProperties gp)
			throws BPMNException {
		Process p = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(),
		        participant.getProcessRef(), Process.class);
		
		String processName = BPELGeneratorHelper.getBPELProcessName(p);
		BPELProcess res = new BPELProcessImpl(URI.create(processName), new TProcess(), null, null);
		ExtensionAdapter.getInstance().adaptExtensions(p, res);
		res.setName(processName);
		res.setTargetNamespace(gp.getBPMNDefinitions().getTargetNamespace());
		gp.setBpelProcess(res);
		
		gp.setCurrentParticipant(participant);
		
		generatePartnerLinks(participant, gp);

		// create Process
		this.generateProcess(res, p, gp);

		return res;
	}
	
	/**
	 * Generate the PartnerLinks out of the ParnternLinkTypes from the wsdl imports
	 * @param participant
	 * @param gp
	 * @throws BPMNException
	 */
	public void generatePartnerLinks(Participant participant, GenerationProperties gp) throws BPMNException {
	    for(PartnerLinkType plt : gp.getPartnerLinkTypes().keySet()) {
	        if(plt.getRoles()!=null && plt.getRoles().length>0) {
	            PartnerLink pl = gp.getBpelProcess().createPartnerLink();
	            pl.setName(plt.getName().replace(WSDLGeneratorHelper.PLTSuffix, ""));
	            
	            String role = plt.getRoles()[0].getName();
	            
	            Participant pltParticipant = gp.getPartnerLinkTypes().get(plt);
	            if(pltParticipant==participant) {
	            	pl.setMyRole(role);
	            }
	            else {
	            	pl.setPartnerRole(role);
	            }
	            pl.setPartnerLinkType(new QName(BPELGeneratorHelper.getPartnerLinkTypeNS(plt), plt.getName()));
	        }
	    }
	}

	private void generateProcess(BPELProcess bpel, Process bpmn,
			GenerationProperties gp) throws BPMNException {
		try {
			// create main sequence
			Sequence seqMain = new SequenceImpl(new TSequence(), bpel);
			seqMain.setName("main");
			bpel.setActivity(seqMain);

			// create pick
			List<FlowNode> starts = this.detectAllStartingNodes(bpmn,gp); 
			gp.setLoopers(BPELGeneratorHelper.findLoopDivergers(bpmn, starts));
			
			if(starts.size()==1) {
			    generateActivity(starts.get(0), seqMain, gp);
			}
			else if(starts.size()>1){
			    generateStartingPick(starts, seqMain, true, gp);
			}
		} catch (BPELException e) {
			throw new BPMNException("An error occured during BPEL process generation.", e);
		}
	}
	
	public List<FlowNode> detectAllStartingNodes(WithFlowElements bpmn, GenerationProperties gp) throws BPMNException {
	    List<FlowNode> l = new ArrayList<FlowNode>();
	    for(FlowNode fn : bpmn.getFlowElementsByClass(FlowNode.class)) {
	        if(fn instanceof StartEvent) {
	            if(BPELGeneratorHelper.getEventDefinition((StartEvent) fn, gp.getBPMNDefinitions())==null) {
	                l.addAll(fn.getOutgoingFlowNodes());
	            }
	            else{
	                l.add(fn);
	            }
	        }
	        else if(fn.getIncomingFlowNodes().isEmpty()) { //TODO this condition is not good : loops might loop on a start receive task 
	            l.add(fn);
	        }
	    }
	    return l;
	}

	private Activity generateNextActivity(FlowElement a, BPELElement parent,
			GenerationProperties gp) throws BPELException, BPMNException {
		System.out.println("************* FlowElement: " + a.getName()
				+ " --- instance of class: " + a.getClass());
		Activity ua = null;
		if(a instanceof com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity) {
		    com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity t = (com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity)a;
			System.out.println("************* Taskes: "+ t.getName());
			if(t.getOutgoingFlowNodes().size() > 1) {
				Flow f = generateFlow(t, parent, gp);
				ua = f;
			} else if(t.getOutgoingFlowNodes().size() == 1) {
				Sequence seq = generateSequence(t, parent, gp);
				ua = seq;
			} 
		} else if(a instanceof Gateway) {
			Gateway g = (Gateway)a;
			System.out.println("************* gateways: "+ g.getName());
			if(g.getOutgoingFlowNodes().size() > 1) {
				Activity f = generateActivityFromGateway(g, (Activity) parent, gp);
				ua = f;
			} else if(g.getOutgoingFlowNodes().size() == 1) {
				// end of gateway
				Sequence seq = new SequenceImpl(new TSequence(), parent);
				this.addActivityInParent(seq, parent);
				ua = generateActivity(g.getOutgoingFlowNodes().get(0), seq, gp);
			} 
		}
		else if (a instanceof Event){
		    Event e = (Event)a;
		    System.out.println("************* events: "+ e.getName());
		    if(e.getOutgoingFlowNodes().size() !=0 ){
		        if(e.getOutgoingFlowNodes().size() > 1) {
		            Flow f = generateFlow(e, parent, gp);
		            ua = f;
		        } else if(e.getOutgoingFlowNodes().size() == 1) {
		            // end of gateway
		            ua = generateActivity(e.getOutgoingFlowNodes().get(0), (Activity)parent, gp);
		        }
		    }
		}
		else {
		    throw new BPMNException("This type of flow element is not supported yet. "+a.getClass());
		}
		return ua;
	}

	private Sequence generateSequence(FlowNode a, BPELElement parent,
			GenerationProperties gp) throws BPELException, BPMNException {
		Sequence seq = null;
		System.out.println("******* Flow node is: "+ a.getName());
		if(parent instanceof Sequence) {
			seq = (Sequence) parent;
		} else {
			seq = new SequenceImpl(new TSequence(), parent);
			this.addActivityInParent(seq, parent);
		}
		
		generateActivity((FlowElement)a.getOutgoingFlowNodes().get(0), seq, gp);

		return seq;
	}

	private Flow generateFlow(FlowNode fn, BPELElement parent,
	        GenerationProperties gp) throws BPELException, BPMNException {
	    Flow flow = new FlowImpl(new TFlow(), parent);
	    this.addActivityInParent(flow, (Activity) parent);
	    
	    Process proc = DefinitionsHelper.getParentProcess(fn);
	    List<FlowNode> ifNodes = new ArrayList<FlowNode>();
	    
	    for(FlowElement elmt : fn.getOutgoingFlowNodes()) {
	        SequenceFlow sf = DefinitionsHelper.getSequenceFlow(proc, fn.getId(), elmt.getId());
	        if(sf.hasConditionExpression() || (fn instanceof WithDefaultSequenceFlow && ((WithDefaultSequenceFlow)fn).getDefault().equals(sf))) {
	            ifNodes.add((FlowNode) elmt);
	        }
	        else {
	            generateActivity(elmt, flow, gp);
	        }
	    }

	    if(!ifNodes.isEmpty()) {
	        generateIf(ifNodes, flow, fn, (fn instanceof WithDefaultSequenceFlow ? ((WithDefaultSequenceFlow)fn).getDefault() : null), gp);
	    }
	    
	    return flow;
	}

	public Activity generateActivity(FlowElement flowElement,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		if(flowElement instanceof Task) {
			return generateActivityFromTask((Task)flowElement, seqOrFlowOrScope, gp);
		} else if(flowElement instanceof CallActivity) {
		    return generateScope((CallActivity)flowElement, seqOrFlowOrScope, gp);
		} else if(flowElement instanceof Gateway) {
			return generateActivityFromGateway((Gateway) flowElement, seqOrFlowOrScope, gp);
		} else if(flowElement instanceof Event) {
			return generateActivityFromEvent((Event) flowElement, seqOrFlowOrScope, gp);
		} else {
			throw new BPMNException("Sorry but this element: " + flowElement.getClass() + " cannot be generated into a bpel activity");
		}
	}
	
	
	private Activity generateActivityFromEvent(Event a,
	        Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
	    Activity res = null;
	    if(a instanceof BoundaryEvent) {
	    	throw new BPMNException("Sorry, but Boundary events are not supported yet.");
	    }
	    else if(a instanceof CatchEvent) {
	    	EventDefinition ed = BPELGeneratorHelper.getEventDefinition((CatchEvent) a, gp.getBPMNDefinitions());
	    	if(ed instanceof MessageEventDefinition){
	    		res = this.generateReceiveIntEvent((CatchEvent)a, (MessageEventDefinition)ed, seqOrFlowOrScope, gp);
	    	} else if (ed instanceof TimerEventDefinition){
	    		//generate wait
	    		res = this.generateWait(a, (TimerEventDefinition)ed, seqOrFlowOrScope, gp);
	    	} else {
	    		if(ed!=null) {
	    			throw new BPMNException("Sorry, but this kind of event is not supported yet : "+ed.getXmlObjectQName());
	    		}
	    	}
	    	//TODO other events
	        
	        generateAssign(a.getName(), ((CatchEvent)a).getDataOutputAssociation(), seqOrFlowOrScope, gp);
	    }
	    else if(a instanceof ThrowEvent) {
	        generateAssign(a.getName(), ((ThrowEvent)a).getDataInputAssociation(), seqOrFlowOrScope, gp);
	        
	        EventDefinition ed = BPELGeneratorHelper.getEventDefinition((ThrowEvent)a,gp.getBPMNDefinitions());
	        if (ed instanceof MessageEventDefinition){
	        	res = this.generateSendIntEvent((ThrowEvent)a, (MessageEventDefinition)ed, seqOrFlowOrScope, gp);
	        }
	        else if(ed instanceof TerminateEventDefinition) {
	        	Exit exit = new ExitImpl(new TExit(), seqOrFlowOrScope);
	        	ExtensionAdapter.getInstance().adaptExtensions(a, exit);
	        	addActivityInParent(exit, seqOrFlowOrScope);
	        } else {
	        	if(ed!=null) {
	        		throw new BPMNException("Sorry, but this kind of event is not supported yet : "+ed.getXmlObjectQName());
	        	}
	        }
	        //TODO other events
	    }
	    
	    // generate next activities
	    this.generateNextActivity(a, seqOrFlowOrScope, gp);
		return res;
	}

	private Wait generateWait(WithName wn, TimerEventDefinition ed, Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException{
	    Wait wait = new WaitImpl(new TWait(), seqOrFlowOrScope);
	    ExtensionAdapter.getInstance().adaptExtensions((BaseElement) wn, wait);
	    this.addActivityInParent(wait, (Activity) seqOrFlowOrScope);
	    wait.setName(wn.getName());
	    
	    if(ed.hasTimeDate()){
	    	TDeadlineExpr expModel = new TDeadlineExpr();
	        expModel.getContent().add(ed.getTimeDate().getTextContent());
	        BPELExpression exp = new BPELExpressionImpl(expModel, wait);
	        wait.setUntil(exp);
	    }
	    else if(ed.hasTimeDuration()) {
	    	TDurationExpr expModel = new TDurationExpr();
	        expModel.getContent().add(ed.getTimeDuration().getTextContent());
	        BPELExpression exp = new BPELExpressionImpl(expModel, wait);
	        wait.setFor(exp);
	    }
	    else if(ed.hasTimeCycle()) {
	        throw new BPMNException("No BPEL Activity corresponds to a BPMN Timer Event with a cycle Timer Event Definition.");
	    }
	    else {
	        throw new BPMNException("A timer definition must have its Duration, its Date or its Cycle set.");
	    }
	    
        return wait;
	}
	
	private Activity generateSendIntEvent(ThrowEvent e, MessageEventDefinition ed,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
	    if(e instanceof EndEvent || BPELGeneratorHelper.isReply(ed,gp.getCurrentParticipant(),gp.getBPMNDefinitions())) {
	        Reply reply = generateReply(e.getName(), ed.getOperationRef(), e.getDataInput(), seqOrFlowOrScope, gp);
	        ExtensionAdapter.getInstance().adaptExtensions(e, reply);
	        return reply;
	    }
	    else if(e instanceof IntermediateThrowEvent) {
	        return generateInOnlyInvoke(e.getName(), ed.getOperationRef(), e.getDataInput(), seqOrFlowOrScope, gp);
	    }
	    throw new BPMNException("Unknown type of event : "+e.getClass());
	}

	private Receive generateReceiveIntEvent(CatchEvent a, MessageEventDefinition ed,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
	    assert a!=null;
	    boolean instantiate = a instanceof StartEvent;
		Receive receive = generateReceive(a.getName(),instantiate,ed.getOperationRef(),a.getDataOutput(),seqOrFlowOrScope,gp);
		ExtensionAdapter.getInstance().adaptExtensions(a, receive);
        return receive;
	}

	private Activity generateActivityFromGateway(Gateway gateway,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		Activity res = null;
		if(gateway instanceof ExclusiveGateway) {
			ExclusiveGateway eg = (ExclusiveGateway)gateway;
			
			if(gp.getAlreadyGeneratedLoops().contains(gateway)){
				if(gp.getAlreadyGeneratedLoopExit().contains(gateway)){
					return null;
				}
				gp.getAlreadyGeneratedLoopExit().add(eg);
				return generateActivity(gp.getLoopExit(eg), gp.getCurrentWhileParent(), gp);
			}
			
			if(gp.isLoopDiverger(eg)){
				gp.getAlreadyGeneratedLoops().add(eg);
				generateWhile(eg,gp.getLoopingFlows(eg),seqOrFlowOrScope,gp);
			}
			else{
			    List<FlowNode> outgoing = eg.getOutgoingFlowNodes();
			    if(outgoing.size()==1) {
			        res = generateActivity(outgoing.get(0), seqOrFlowOrScope, gp);
			    }
			    else {
			        res = this.generateIf(eg, seqOrFlowOrScope, gp);
			    }
			}
		} 
		else if(gateway instanceof ParallelGateway) {
			res = this.generateFlow((ParallelGateway)gateway, seqOrFlowOrScope, gp);
		}
		else if(gateway instanceof EventBasedGateway) {
		    if(gp.getAlreadyGeneratedLoops().contains(gateway)) {
		        return null;
		    }
		    if(gp.isLoopDiverger(gateway)){
		        gp.getAlreadyGeneratedLoops().add(gateway);
		        return generateWhile((EventBasedGateway)gateway, seqOrFlowOrScope, gp);
		    }
		    else {
		        res = this.generatePick((EventBasedGateway)gateway, seqOrFlowOrScope, gp);
		    }
		}
		else {
			throw new BPMNException("Sorry but this element: " + gateway.getClass() + " cannot be generated into a bpel activity");
		}
		return res;
	}
	
	private Activity generateStartingPick(Collection<FlowNode> starts, Activity parent, boolean createInstance,
            GenerationProperties gp) throws BPELException, BPMNException {
	    Pick pick  = new PickImpl(new TPick(), parent);
	    pick.setCreateInstance(createInstance);
        this.addActivityInParent(pick, parent);
	    for(FlowNode fn: starts) {
            if(fn instanceof CatchEvent) {
            	CatchEvent e = (CatchEvent)fn;
                EventDefinition ed = BPELGeneratorHelper.getEventDefinition(e,gp.getBPMNDefinitions());
                if(ed instanceof MessageEventDefinition) {
                    generateOnMessage(fn, ((MessageEventDefinition)ed).getOperationRef(),
                            ((CatchEvent)fn).getDataOutputAssociation(), null, e.getDataOutput(), pick, gp);
                }
                else {
                	generateOnAlarm(fn, ((TimerEventDefinition)ed), pick, gp);
                }
            }
            else if(fn instanceof ReceiveTask) {
                generateOnMessage((ReceiveTask) fn, pick, gp);
            }
            else if(fn instanceof EventBasedGateway) {
                return generatePick((EventBasedGateway)fn, parent, gp);
            }
            else {
                throw new BPMNException("A BPEL process should start with catch events or receive tasks. "+fn.getClass());
            }
	    }
	    return pick;
	}
	
	private Activity generatePick(EventBasedGateway gateway, Activity parent,
            GenerationProperties gp) throws BPELException, BPMNException {
	    Pick pick  = new PickImpl(new TPick(), parent);
	    pick.setCreateInstance(false);
        this.addActivityInParent(pick, parent);
        FlowElement loopExit = null;
        if(gp.isLoopDiverger(gateway)) {
            loopExit = gp.getLoopExit(gateway);
        }
        for(FlowNode fn: gateway.getOutgoingFlowNodes()) {
            BPELElement e = null;
            if(fn instanceof IntermediateCatchEvent) {
            	CatchEvent event = (CatchEvent)fn;
                EventDefinition ed = BPELGeneratorHelper.getEventDefinition(event,gp.getBPMNDefinitions());
                if(ed instanceof MessageEventDefinition) {
                    e = generateOnMessage(fn, ((MessageEventDefinition)ed).getOperationRef(),
                            ((IntermediateCatchEvent)fn).getDataOutputAssociation(), null, event.getDataOutput(), pick, gp);
                }
                else if(ed instanceof TimerEventDefinition) {
                	e = generateOnAlarm(fn, ((TimerEventDefinition)ed), pick, gp);
                }
                else {
                    throw new BPMNException("An event based gateway should be connected to intermediate message catching events or to receive tasks.");
                }
            }
            else if(fn instanceof ReceiveTask) {
                e = generateOnMessage((ReceiveTask) fn, pick, gp);
            }
            else {
                throw new BPMNException("An event based gateway should be connected to intermediate message catching events or to receive tasks.");
            }
            if(loopExit!=null && loopExit.equals(fn)) {
                Variable var = createOrFindBooleanVariable(gateway, gp);
                AssignImpl assign = new AssignImpl(new TAssign(), e);
                CopyImpl copy = new CopyImpl(new TCopy(), assign);
                FromImpl from = new FromImpl(new TFrom(), copy);
                from.getModel().getContent().add("true()");
                ToImpl to = new ToImpl(new TTo(), copy);
                to.getModel().setVariable(var.getName());
                copy.setFrom(from);
                copy.setTo(to);
                assign.addCopy(copy);
                this.addActivityInParent(assign, e);
            }
        }
        return pick;
    }
	
	private Scope generateScope(CallActivity callActi, Activity seqOrFlowOrScope, 
	        GenerationProperties gp) throws BPELException, BPMNException {
	    QName qn = callActi.getCalledElement();
	    Process proc = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), qn, Process.class);
	    
	    Scope scope = new ScopeImpl(new TScope(), seqOrFlowOrScope);
	    ExtensionAdapter.getInstance().adaptExtensions(callActi, scope);
	    this.addActivityInParent(scope, seqOrFlowOrScope);
	    
	    // create main sequence
	    Sequence seqMain = new SequenceImpl(new TSequence(), scope);
	    this.addActivityInParent(seqMain, scope);

	    Definitions defs = gp.getBPMNDefinitions();
	    gp.setBPMNDefinitions(DefinitionsHelper.getParentDefinitions(proc));
	    
	    // create pick
	    List<FlowNode> starts = this.detectAllStartingNodes(proc,gp); 
	    if(starts.size()==1) {
	        generateActivity(starts.iterator().next(), seqMain, gp);
	    }
	    else if(starts.size()>1){
	        generateStartingPick(starts, seqMain, false, gp);
	    }
	    
	    gp.setBPMNDefinitions(defs);
	    
	    this.generateNextActivity(callActi, seqOrFlowOrScope, gp);
	    
	    return scope;
	}
	
	private While generateWhile(EventBasedGateway ebg, Activity seqOrFlowOrScope,
	        GenerationProperties gp) throws BPELException, BPMNException {
	    TWhile model = new TWhile();
        While while_ = new WhileImpl(model, seqOrFlowOrScope);
        while_.setName(ebg.getName());
        
        gp.setCurrentWhileParent(seqOrFlowOrScope);
        
        Variable var = createOrFindBooleanVariable(ebg, gp);
        AssignImpl assign = new AssignImpl(new TAssign(), seqOrFlowOrScope);
        CopyImpl copy = new CopyImpl(new TCopy(), assign);
        FromImpl from = new FromImpl(new TFrom(), copy);
        from.getModel().getContent().add("false()");
        ToImpl to = new ToImpl(new TTo(), copy);
        to.getModel().setVariable(var.getName());
        copy.setFrom(from);
        copy.setTo(to);
        assign.addCopy(copy);
        this.addActivityInParent(assign, seqOrFlowOrScope);
        
        String loopCondition = "not($"+var.getName()+")";
        TBooleanExpr expModel = new TBooleanExpr();
        expModel.getContent().add(loopCondition);
        BPELExpression cond = new BPELExpressionImpl(expModel, while_);
        while_.setCondition(cond);
        
        this.addActivityInParent(while_, seqOrFlowOrScope);
        
        this.generatePick(ebg, while_, gp);
        
        return while_;
	}

    private void generateWhile(ExclusiveGateway eg, Set<SequenceFlow> loopingFlows,
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
		System.out.println("*************While for : "+eg.getId());
		
		TWhile model = new TWhile();
		While while_ = new WhileImpl(model, seqOrFlowOrScope);
		while_.setName(eg.getName());
		
		gp.setCurrentWhileParent(seqOrFlowOrScope);
		
		Process p = DefinitionsHelper.getParentProcess(eg);
		SequenceFlow nonlooper = null;
		for(SequenceFlow sf : p.getFlowElementsByClass(SequenceFlow.class)){
			if(sf.getSourceRef().getId().equals(eg.getId()) && !loopingFlows.contains(sf)){
				nonlooper = sf;
				break;
			}
		}
		String loopCondition = "not("+generateExpression(nonlooper.getConditionExpression().getTextContent(),gp)+")";
		
		TBooleanExpr expModel = new TBooleanExpr();
        expModel.getContent().add(loopCondition);
        BPELExpression cond = new BPELExpressionImpl(expModel, while_);
        while_.setCondition(cond);
		
		addActivityInParent(while_, seqOrFlowOrScope);
		
		generateWhileInsideIf(eg, loopingFlows, while_ , model, gp);
	}
	
	
	private void generateWhileInsideIf(ExclusiveGateway eg, Collection<SequenceFlow> loopingFlows, While while_ ,
			TWhile model, GenerationProperties gp) throws BPELException, BPMNException {
		TSequence sm = new TSequence();
		Sequence seq = new SequenceImpl(sm, while_);
		while_.setActivity(seq);
		model.setSequence(sm);
		
		List<FlowNode> activities = new ArrayList<FlowNode>();
		for(SequenceFlow sf : loopingFlows){
			activities.add(sf.getTargetRef());
		}
		
		generateIf(activities, seq, eg, eg.getDefault(),  gp);
	}
	
	
	private Flow getCurrentFlow(final BPELElement parent) {
		Flow res = null;
		if(parent != null) {
			if(parent instanceof Flow) {
				res = (Flow) parent;
			} else {
				res = getCurrentFlow((BPELElement) ((AbstractSchemaElementImpl<?>)parent).getParent());
			}
		}
		return res;
	}

	private If getCurrentIf(final BPELElement parent) {
		If res = null;
		if(parent != null) {
			if(parent instanceof If) {
				res = (If) parent;
			} else {
				res = getCurrentIf((BPELElement) ((AbstractSchemaElementImpl<?>)parent).getParent());
			}
		}
		return res;
	}

	private Flow generateFlow(ParallelGateway gateway,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
	    Flow flowA = null;
	    System.out.println("found gateways: "+ gateway.getId());
	    List<FlowNode> activities = gateway.getOutgoingFlowNodes();
	    if(activities.size() > 0 && gateway.getIncomingFlowNodes().size() == 1) {
	        flowA = new FlowImpl(new TFlow(), seqOrFlowOrScope);
	        this.addActivityInParent(flowA, (Activity) seqOrFlowOrScope);
	        System.out.println("flow out size = " + activities.size());
	        for(FlowElement elmt: activities) {
	            Sequence seq = new SequenceImpl(new TSequence(), flowA);
	            this.addActivityInParent(seq, flowA);
	            this.generateActivity(elmt, seq, gp);
	        }			

	    } else {
	        // save end of gateway
	        if(!gp.getGatewaysAlreadyGenerated().contains(gateway)) {
	            gp.getGatewaysAlreadyGenerated().add(gateway);
	            this.generateNextActivity(gateway, (BPELElement) ((BPELElementImpl<?>)getCurrentFlow(seqOrFlowOrScope)).getParent(), gp);
	        }
	    }		
	    return flowA;
	}
	
	private If generateIf(List<FlowNode> activities, Activity seqOrFlowOrScope, FlowElement forker,
			SequenceFlow defaultFlow, GenerationProperties gp) throws BPELException, BPMNException {
		
		If ifA = new IfImpl(new TIf(), seqOrFlowOrScope);
		this.addActivityInParent(ifA, (Activity) seqOrFlowOrScope);
		System.out.println("if out size = " + activities.size());
		boolean first = true;
		for(FlowElement elmt: activities) {
			Sequence seq = new SequenceImpl(new TSequence(), ifA);
			
			this.generateActivity(elmt, seq, gp);
			
			try{
			    System.out.println("elmt.getId().toString() = " + elmt.getId().toString());
			    SequenceFlow sf = DefinitionsHelper.getSequenceFlow(DefinitionsHelper.getParentProcess(elmt), forker.getId() , elmt.getId());
			    
			    if(sf.hasConditionExpression()) {
			    	TBooleanExpr model = new TBooleanExpr();
		        	model.getContent().add(generateExpression(sf.getConditionExpression().getTextContent(),gp));
                    BPELExpression exp = new BPELExpressionImpl(model, ifA);
			        if(first) {
	                    ifA.addIfActivity(exp, seq);
	                    first = false;
	                } else {
	                    ifA.addElseIfActivity(exp, seq);
	                }
			    }
			    else if(defaultFlow!=null && (defaultFlow.getTargetRef().getId().toString().equals(elmt.getId().toString()))) {
			        ifA.setElseActivity(seq);
			    }
			    else {
			        throw new BPMNException("A sequence flow going out of an exclusive gateway must either have a condition or be the default of the gateway.");
			    }
			}
			catch(XPathExpressionException e) {
			    throw new BPELException(e);
			}
		}
		
		return ifA;
	}

	private If generateIf(ExclusiveGateway gateway, Activity seqOrFlowOrScope,
	        GenerationProperties gp) throws BPELException, BPMNException {
	    System.out.println("found gateways: "+ gateway.getId());
	    List<FlowNode> activities = gateway.getOutgoingFlowNodes();

	    if(activities.size() > 0 && gateway.getIncomingFlowNodes().size() == 1) {
	        return generateIf(activities, seqOrFlowOrScope, gateway, gateway.getDefault(), gp);
	    } else {
	        // save end of gateway
	        if(!gp.getGatewaysAlreadyGenerated().contains(gateway)) {
	            gp.getGatewaysAlreadyGenerated().add(gateway);
	            this.generateNextActivity(gateway, (BPELElement) ((BPELElementImpl<?>)getCurrentIf(seqOrFlowOrScope)).getParent(), gp);
	        }
	    }
	    return null;
	}

	private Activity generateActivityFromTask(Task a, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException, BPMNException {
		Activity res = null;
		if(a != null) {

		    if(a.hasDataInputAssociation()) {
                this.generateAssign(a.getName(), a.getDataInputAssociation(), seqOrFlowOrScope, gp); 
            }

			if(a instanceof ServiceTask) {
				// generate invoke
				res = this.generateInvoke((ServiceTask)a, seqOrFlowOrScope, gp);
			} else if(a instanceof ReceiveTask) {
				// generate receive
				res = this.generateReceive((ReceiveTask) a, seqOrFlowOrScope, gp);
			} else if(a instanceof SendTask) {
				// generate invoke or reply
				if(BPELGeneratorHelper.isReply((SendTask)a,gp.getCurrentParticipant(),gp.getBPMNDefinitions())){
					res = this.generateReply((SendTask)a, seqOrFlowOrScope, gp);
				}
				else{
					res = this.generateInOnlyInvoke((SendTask)a, seqOrFlowOrScope, gp);
				}
			} else {
				// generate empty
				res = this.generateEmpty(a, seqOrFlowOrScope, gp);
			}

			if(a.hasDataOutputAssociation()) {
                this.generateAssign(a.getName(), a.getDataOutputAssociation(), seqOrFlowOrScope, gp); 
            }

			// generate next activities
			this.generateNextActivity(a, seqOrFlowOrScope, gp);

		}
		return res;
	}
	
	private Receive generateReceive(ReceiveTask a, 
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		assert a != null;
		boolean instantiate = a.isInstantiate() || detectAllStartingNodes(DefinitionsHelper.getParentContainer(a),gp).contains(a);
		Receive receive;
		if(a.getIoSpecification()!=null) {
			receive = generateReceive(a.getName(), instantiate, a.getOperationRef(), a.getIoSpecification().getDataOutput(), seqOrFlowOrScope, gp);
		}
		else {
			receive = generateReceive(a.getName(), instantiate, a.getOperationRef(), new ItemAwareElement[]{}, seqOrFlowOrScope, gp);
		}
		ExtensionAdapter.getInstance().adaptExtensions(a, receive);
		return receive;
	}
	
	private Receive generateReceive(String name, boolean createInstance, QName operationRef,
            ItemAwareElement[] outputs, Activity seqOrFlowOrScope, GenerationProperties gp)
            throws BPELException, BPMNException {
	    Receive receive = new ReceiveImpl(new TReceive(), seqOrFlowOrScope);
        this.addActivityInParent(receive, (Activity) seqOrFlowOrScope);
        receive.setName(name);
        receive.setCreateInstance(createInstance);
        
        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            receive.setInterface(itf.getImplementationRef());
            receive.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            receive.setOperation(op.getName());
            receive.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        receive.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                receive.getInterface(), gp.getBpelProcess(), gp.getPartnerLinkTypes().keySet()).getName());
        
        Message msgIn = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class);
        Variable inputVariable = createOrFindVariableFromMessage(msgIn, outputs, gp);
        receive.setInputVariable(inputVariable.getName());
        
        gp.getInitializedVariables().add(inputVariable);
        
        return receive;
	}

	private Invoke generateInvoke(ServiceTask a, 
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
		Invoke invoke;
		if(a.getIoSpecification()!=null) {
			invoke = generateInvoke(a.getName(), a.getOperationRef(), a.getIoSpecification().getDataInput(), a.getIoSpecification().getDataOutput(), seqOrFlowOrScope, gp);
		}
		else {
			invoke = generateInvoke(a.getName(), a.getOperationRef(), new ItemAwareElement[]{}, new ItemAwareElement[]{}, seqOrFlowOrScope, gp);
		}
	    ExtensionAdapter.getInstance().adaptExtensions(a, invoke);
        return invoke;
	}

	private Invoke generateInOnlyInvoke(SendTask a, 
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		Invoke invoke;
		if(a.getIoSpecification()!=null) {
			invoke = generateInOnlyInvoke(a.getName(), a.getOperationRef(), a.getIoSpecification().getDataInput(), seqOrFlowOrScope, gp);
		}
		else {
			invoke = generateInOnlyInvoke(a.getName(), a.getOperationRef(), new ItemAwareElement[]{}, seqOrFlowOrScope, gp);
		}
		ExtensionAdapter.getInstance().adaptExtensions(a, invoke);
        return invoke;
	}
	
	private Invoke generateInOnlyInvoke(String name, QName operationRef, ItemAwareElement[] input,
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
	    Invoke invoke = new InvokeImpl(new TInvoke(), seqOrFlowOrScope);
        invoke.setName(name);
        
        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            invoke.setInterface(itf.getImplementationRef());
            invoke.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            invoke.setOperation(op.getName());
            invoke.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        invoke.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                invoke.getInterface(), gp.getBpelProcess(), gp.getPartnerLinkTypes().keySet()).getName());
        
        Message msgIn = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class);
        Variable inputVariable = createOrFindVariableFromMessage(msgIn, input, gp);
        invoke.setInputVariable(inputVariable.getName());
        
        if(!gp.getInitializedVariables().contains(inputVariable)) {
        	generateVariableInitializationAssign(inputVariable,seqOrFlowOrScope,gp);
        }
        
        this.addActivityInParent(invoke, (Activity) seqOrFlowOrScope);
        
        return invoke;
	}
	
	private Invoke generateInvoke(String name, QName operationRef, ItemAwareElement[] input, ItemAwareElement[] output,
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
		Invoke invoke = generateInOnlyInvoke(name, operationRef, input, seqOrFlowOrScope, gp); 
		
		Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
		
		Message msgOut = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getOutMessageRef(), Message.class);
		Variable outputVar = createOrFindVariableFromMessage(msgOut, output, gp);
		invoke.setOutputVariable(outputVar.getName());

		gp.getInitializedVariables().add(outputVar);
		
		return invoke;
	}

	private Empty generateEmpty(Task a, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException {
		Empty empty = new EmptyImpl(new TEmpty(), seqOrFlowOrScope);
		ExtensionAdapter.getInstance().adaptExtensions(a, empty);
		this.addActivityInParent(empty, (Activity) seqOrFlowOrScope);
		empty.setName(a.getName());
		return empty;
	}

	private Reply generateReply(SendTask a, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException, BPMNException {
		Reply reply;
		if(a.getIoSpecification()!=null) {
			reply = generateReply(a.getName(), a.getOperationRef(), a.getIoSpecification().getDataInput(), seqOrFlowOrScope, gp);
		}
		else {
			reply = generateReply(a.getName(), a.getOperationRef(), new ItemAwareElement[]{}, seqOrFlowOrScope, gp); 
		}
		ExtensionAdapter.getInstance().adaptExtensions(a, reply);
        return reply;
	}
	
	private Reply generateReply(String name, QName operationRef, ItemAwareElement[] inputs,
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
	    Reply reply = new ReplyImpl(new TReply(), seqOrFlowOrScope);
        reply.setName(name);

        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            reply.setInterface(itf.getImplementationRef());
            reply.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            reply.setOperation(op.getName());
            reply.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        reply.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                reply.getInterface(), gp.getBpelProcess(), gp.getPartnerLinkTypes().keySet()).getName());
        
        Message msg = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getOutMessageRef(), Message.class);
        Variable var = this.createOrFindVariableFromMessage(msg,inputs,gp);
        reply.setOutputVariable(var.getName());
        
        if(!gp.getInitializedVariables().contains(var)) {
        	generateVariableInitializationAssign(var,seqOrFlowOrScope,gp);
        }
        
        this.addActivityInParent(reply, (Activity) seqOrFlowOrScope);

        return reply;
	}

	private void addActivityInParent(BPELElement child,
			BPELElement seqOrFlowOrScope)
			throws BPELException {
		if (seqOrFlowOrScope instanceof Sequence) {
			((Sequence) seqOrFlowOrScope).addActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof Flow) {
			((Flow) seqOrFlowOrScope).addActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof OnMessage) {
			((OnMessage) seqOrFlowOrScope).setActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof OnAlarm) {
			((OnAlarm) seqOrFlowOrScope).setActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof Pick) {
			if(child instanceof OnAlarm) {
				((Pick) seqOrFlowOrScope).addOnAlarm((OnAlarm) child);
			}
			else if(child instanceof OnMessage) {
				((Pick) seqOrFlowOrScope).addOnMessage((OnMessage) child);
			}
		} else if(seqOrFlowOrScope instanceof While) {
			((While)seqOrFlowOrScope).setActivity((Activity)child);
		} else if(seqOrFlowOrScope instanceof Scope) {
		    ((Scope)seqOrFlowOrScope).setActivity((Activity)child);
		}
		else {
			throw new BPELException("Sorry but this type of element is not handled yet : "+seqOrFlowOrScope.getClass());
		}
	}

	private Assign generateAssign(String activityName,
			DataAssociation[] dataAssociations, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException, BPMNException {
		AssignImpl assign = null;
		if(dataAssociations.length > 0) {
			assign = new AssignImpl(new TAssign(), seqOrFlowOrScope);
			for(DataAssociation asso: dataAssociations) {
			    for(ItemAwareElement iae : asso.getSourceRef()) {
			    	//this should not create variables : just in case...
			        createOrFindVariableFromItemAwareElement(iae, gp);
			    }
			    Variable var = createOrFindVariableFromItemAwareElement(asso.getTargetRef(), gp);
			    if(asso.hasAssignment()) gp.getInitializedVariables().add(var);
			    
				for(Assignment a: asso.getAssignment()) {
					CopyImpl copy = new CopyImpl(new TCopy(), assign);
					
					FromImpl from = new FromImpl(new TFrom(), copy);
			        from.getModel().getContent().add(generateExpression(a.getFrom().getTextContent(),gp));
			        copy.setFrom(from);
					
			        ToImpl to = new ToImpl(new TTo(), copy);
			        to.getModel().getContent().add(generateExpression(a.getTo().getTextContent(),gp));
			        copy.setTo(to);

					assign.addCopy(copy);
				}
				
				if(asso.hasTransformation()) {
				    System.out.println("WARNING : the transformations defined in BPMN dataAssociations are not handled yet.");
				}
			}
			this.addActivityInParent(assign, seqOrFlowOrScope);
		}
		return assign;
	}
	
	private void generateVariableInitializationAssign(Variable variable,
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
		AssignImpl assign = new AssignImpl(new TAssign(), seqOrFlowOrScope);
        CopyImpl copy = new CopyImpl(new TCopy(), assign);
        FromImpl from = new FromImpl(new TFrom(), copy);

        Element elt = gp.getBPMNDefinitions().findImportedElement(((AbstractVariableImpl)variable).getModel().getElement());
        
        org.jdom.Element jdomElt = XSD2XML.newInstance().generateElement(elt, SchemaHelper.findParentSchema(elt), "?");
        Document d;
		try {
			d = new DOMOutputter().output(new org.jdom.Document(jdomElt));
		} catch (JDOMException e) {
			e.printStackTrace();
			throw new BPELException(e);
		}
        from.getModel().getContent().add(d.getDocumentElement());
        
        ToImpl to = new ToImpl(new TTo(), copy);
        to.getModel().setVariable(variable.getName());
        copy.setFrom(from);
        copy.setTo(to);
        assign.addCopy(copy);
        this.addActivityInParent(assign, seqOrFlowOrScope);
        
        gp.getInitializedVariables().add(variable);
	}
	
    private String generateExpression(String xpathExpression, GenerationProperties gp){
        String result = xpathExpression;
        for(String associationExtrmityId : gp.getVariableMapping().keySet()) {
            result = result.replace("$"+associationExtrmityId, "$"+gp.getVariableMapping().get(associationExtrmityId));
        }
        return result;
    }

	private Variable createOrFindVariableFromItemAwareElement(ItemAwareElement iae,
			GenerationProperties gp) throws BPMNException {
		ItemDefinition itemDef = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), iae.getItemSubjectRef(), ItemDefinition.class);
	    Variable var = createOrFindVariableFromItemDefinition(((BaseElement)iae).getId(),itemDef, gp);
	    gp.getVariableMapping().put(((BaseElement)iae).getId(), var.getName());
	    return var;
	}
	
	private Variable createOrFindVariableFromItemDefinition(String name, ItemDefinition itemDef, GenerationProperties gp) {
		QName element = itemDef.getStructureRef();
	    gp.getElementsToImport().add(element);
	    
	    String varName = BPELGeneratorHelper.getVariableName(name);
	    
		for (Variable var : gp.getBpelProcess().getVariables()) {
//			if (var.getElement()!=null && var.getElement().equals(element) && var.getName().equals(varName)) {
			if (var.getName().equals(varName)) {
			    return var;
			}
		}
		
		TVariable model = new TVariable();
		model.setName(varName);
		model.setElement(element);
		Variable var = new ElementVariableImpl(model, gp.getBpelProcess());
		gp.getBpelProcess().addVariable(var);
		
		//Variable var = gp.getBpelProcess().createBPELElementVariable(varName,element, VariableType.ELEMENT);
		return var;
	}

	private OnMessage generateOnMessage(ReceiveTask rt, Pick pick,
			GenerationProperties gp) throws BPMNException, BPELException {
		if(rt.getIoSpecification()!=null) {
			return generateOnMessage(rt, rt.getOperationRef(), rt.getDataOutputAssociation(), rt.getDataInputAssociation(), rt.getIoSpecification().getDataOutput(), pick, gp);
		}
		else {
			return generateOnMessage(rt, rt.getOperationRef(), rt.getDataOutputAssociation(), rt.getDataInputAssociation(), new ItemAwareElement[]{}, pick, gp);
		}
	}
	
	private OnMessage generateOnMessage(FlowNode fn, QName operationRef, DataOutputAssociation[] doas,
	        DataInputAssociation[] dias, ItemAwareElement[] output, Pick pick, GenerationProperties gp) throws BPELException, BPMNException {
	    OnMessage onMessage = new OnMessageImpl(new TOnMessage(),(BPELElementImpl<?>) pick);
	    ExtensionAdapter.getInstance().adaptExtensions(fn, onMessage);
        this.addActivityInParent(onMessage, pick);
        
        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            onMessage.setInterface(itf.getImplementationRef());
            onMessage.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            onMessage.setOperation(op.getName());
            onMessage.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        
        onMessage.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                onMessage.getInterface(), gp.getBpelProcess(),
                gp.getPartnerLinkTypes().keySet()).getName());
        
        Message msg = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class);
        Variable var = this.createOrFindVariableFromMessage(msg, output, gp);
        onMessage.setInputVariable(var.getName());
        
        gp.getInitializedVariables().add(var);

        Sequence seq = new SequenceImpl(new TSequence(), onMessage);

        if(dias!=null) {
            this.generateAssign(fn.getName(), dias, seq, gp);
        }
        if(doas!=null) {
            this.generateAssign(fn.getName(), doas, seq, gp);
        }
        
        this.generateNextActivity(fn, seq, gp);
        
        if(!seq.getActivities().isEmpty()) {
        	this.addActivityInParent(seq, onMessage);
        }

        return onMessage;
	}
	
	private OnAlarm generateOnAlarm(FlowNode fn, TimerEventDefinition ed, Pick pick, GenerationProperties gp) throws BPELException, BPMNException{
		OnAlarm alarm = new OnAlarmImpl(new TOnAlarmPick(), (BPELElementImpl<?>)pick);
		ExtensionAdapter.getInstance().adaptExtensions(fn, alarm);
	    this.addActivityInParent(alarm, (Activity) pick);
	    
	    if(ed.hasTimeDate()){
	    	TDeadlineExpr expModel = new TDeadlineExpr();
	        expModel.getContent().add(ed.getTimeDate().getTextContent());
	        BPELExpression exp = new BPELExpressionImpl(expModel, alarm);
	        alarm.setUntil(exp);
	    }
	    else if(ed.hasTimeDuration()) {
	    	TDurationExpr expModel = new TDurationExpr();
	        expModel.getContent().add(ed.getTimeDuration().getTextContent());
	        BPELExpression exp = new BPELExpressionImpl(expModel, alarm);
	        alarm.setFor(exp);
	    }
	    else if(ed.hasTimeCycle()){
	        throw new BPMNException("No BPEL Activity corresponds to a BPMN Timer Event with a cycle Timer Event Definition."); 
	    }
	    else {
	        throw new BPMNException("A timer definition must have its Duration, its Date or its Cycle set.");
	    }
	    
	    Sequence seq = new SequenceImpl(new TSequence(), alarm);
        
        this.generateNextActivity(fn, seq, gp);
        
        if(!seq.getActivities().isEmpty()) {
        	this.addActivityInParent(seq, alarm);
        }
	    
        return alarm;
	}

	private Variable createOrFindVariableFromMessage(Message message, ItemAwareElement[] data,
			GenerationProperties gp) throws BPMNException {
		for(ItemAwareElement iae : data) {
			if(iae.getItemSubjectRef().equals(message.getItemRef())) {
				return createOrFindVariableFromItemAwareElement(iae, gp);
			}
		}
		ItemDefinition itemDef = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), message.getItemRef(), ItemDefinition.class);
		return createOrFindVariableFromItemDefinition(message.getName(), itemDef, gp);
	}
	
	private Variable createOrFindBooleanVariable(FlowElement fe, GenerationProperties gp)
	    throws BPMNException {
	    String varName = BPELGeneratorHelper.getVariableName(fe.getId());
	    for(Variable var : gp.getBpelProcess().getVariables()) {
	        if(var.getName().equals(varName)) {
	            return var;
	        }
	    }
	    
	    TVariable model = new TVariable();
		model.setName(varName);
		model.setType(new QName(com.ebmwebsourcing.easyschema10.api.Constants.XMLSCHEMA_NS_URI, "boolean"));
		Variable var = new TypeVariableImpl(model, gp.getBpelProcess());
		gp.getBpelProcess().addVariable(var);
		return var;
	    
	    //return gp.getBpelProcess().createBPELElementVariable(varName, new QName(com.ebmwebsourcing.easyschema10.api.Constants.XMLSCHEMA_NS_URI, "boolean"), VariableType.TYPE);
	}
	
	private void addDescriptionImport(BPELProcess p, com.ebmwebsourcing.easywsdl11.api.element.Definitions desc){
		Import imp = new ImportImpl(new TImport(), p);
		imp.setImportType(URI.create(Constants.WSDL11_NS_URI));
		imp.setLocation(URI.create(desc.getName()+".wsdl"));
		imp.setNamespace(URI.create(desc.getTargetNamespace()));
		((ImportImpl)imp).setDescription((Description)Factory.getInstance().wrap(desc));
		
		p.addImport(imp);
	}
	
	private void addSchemaImport(BPELProcess p, Schema schema, String location){
        Import imp = new ImportImpl(new TImport(), p);
        imp.setImportType(URI.create(com.ebmwebsourcing.easyschema10.api.Constants.XMLSCHEMA_NS_URI));
        imp.setLocation(URI.create(location));
        imp.setNamespace(URI.create(schema.getTargetNamespace()));
        ((ImportImpl)imp).setSchema2(schema);
        
        p.addImport(imp);
    }
	
}
