/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author Marc Jambert - EBM WebSourcing
 */
public final class ByteChannelDumpHelper {

    private ByteChannelDumpHelper() {
    }

    public static final FileChannel createDumpFileChannel(File dumpFile) {
        assert dumpFile != null;
        assert !dumpFile.isDirectory() : "Dump file can't be a directory";
        dumpFile.getParentFile().mkdirs();
        FileOutputStream dumpFileOutputStream;
        try {
            dumpFileOutputStream = new FileOutputStream(dumpFile);
        } catch (FileNotFoundException e) {
            throw new UncheckedException(e);
        }
        return dumpFileOutputStream.getChannel();
    }

    public static final int dumpAndRead(ReadableByteChannel channelToDump, ByteBuffer readBuffer,
            FileChannel dumpChannel) throws IOException {
        ensureChannelIsOpen(channelToDump);
        ByteBuffer readBufferView = readBuffer.duplicate();
        int nbBytesRead = channelToDump.read(readBuffer);
        if (nbBytesRead != -1) {
            readBufferView.limit(readBufferView.position() + nbBytesRead);
            dumpChannel.write(readBufferView);
        }
        return nbBytesRead;
    }

    public static final int dumpAndWrite(WritableByteChannel channelToDump, ByteBuffer writeBuffer,
            FileChannel dumpChannel) throws IOException {
        ensureChannelIsOpen(channelToDump);
        ByteBuffer writeBufferView = writeBuffer.duplicate();
        dumpChannel.write(writeBufferView);
        int nbBytesWritten = channelToDump.write(writeBuffer);
        return nbBytesWritten;
    }
    
    
    private static final void ensureChannelIsOpen(Channel channel) throws ClosedChannelException {
        if (!channel.isOpen()) {
            throw new ClosedChannelException();
        }
    }
    

}
