/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.sca.helper.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.oasisopen.sca.ServiceReference;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.IllegalBindingException;
import org.objectweb.fractal.api.control.IllegalContentException;
import org.objectweb.fractal.api.control.IllegalLifeCycleException;
import org.objectweb.fractal.api.control.LifeCycleController;
import org.objectweb.fractal.api.control.NameController;
import org.objectweb.fractal.api.control.SuperController;
import org.objectweb.fractal.api.factory.Factory;
import org.objectweb.fractal.julia.ComponentInterface;
import org.objectweb.fractal.julia.Interceptor;
import org.objectweb.fractal.util.Fractal;
import org.ow2.frascati.tinfi.api.control.SCAContentController;
import org.ow2.frascati.tinfi.api.control.SCAPropertyController;
import org.ow2.frascati.tinfi.control.content.SCAExtendedContentController;
import org.ow2.frascati.tinfi.oasis.ServiceReferenceImpl;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAComponent;
import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;



/**
 * This class helps fractal management.
 * 
 * @author Nicolas Salatge - EBM WebSourcing
 */
public class SCAHelper {


	/**
	 * the FCAPPL binding name.
	 */
	public static final String FCAPPL_BINDING = "fcAppl";

	/**
	 * The singleton internal instance.
	 */
	private static SCAHelper scaHelper = new SCAHelper();


	/**
	 * Create a SCAHelper instance.
	 * 
	 * @throws ADLException :
	 *             occurs when it is impossible to get the fractal explorer
	 */
	private SCAHelper() {
		super();
	}

	/**
	 * Get a SCAHelper instance.
	 * 
	 * @return the SCAHelper instance
	 * @throws ADLException :
	 *             occurs when it is impossible to get the fractal explorer
	 */
	public static SCAHelper getSCAHelper() throws SCAException {
		return SCAHelper.scaHelper;
	}


	/**
	 * Create a new component.
	 * 
	 * @param name
	 *            The class name of the component
	 * @return the new component
	 * @throws ADLException :
	 *             occurs when the framework of Petals is incorrect
	 */
	public synchronized Component createNewComponent(final String name,
			final Map<String, Object> ctxt) throws SCAException {
		try {
			String s = name + "Factory";
			Class<?> cl = Class.forName(s);
			Factory f = (Factory) cl.newInstance();
			Component c = f.newFcInstance();
			changeName(c, name);
			this.setContext(c, ctxt);
			return c;
		}
		catch( Exception e ) {
			throw new SCAException(e);
		}
	}


	private void setContext(Component fractalBehaviour,
			Map<String, Object> context) throws SCAException {
		if(context != null) {
			for(Entry<String, Object> entry: context.entrySet()) {
				this.setProperty(fractalBehaviour, entry.getKey(), entry.getValue());
			}
		}
	}

	public void setProperty(Component cmp, String propertyName, Object propertyValue) throws SCAException {
		try {
			SCAPropertyController scapc = (SCAPropertyController) cmp.getFcInterface(SCAPropertyController.NAME);
			if(!scapc.isDeclared(propertyName)) {
				throw new SCAException("This property \'" + propertyName + "`' is not declared in component:" + cmp);
			}
			scapc.setValue(propertyName, propertyValue);
		} catch (NoSuchInterfaceException e) {
			throw new SCAException(e);
		}
	}

	public <T> T getPropertyValue(Component cmp, String propertyName, Class<T> clazz) throws SCAException {
		T res = null;
		try {
			SCAPropertyController scapc = (SCAPropertyController) cmp.getFcInterface(SCAPropertyController.NAME);
			if(!scapc.isDeclared(propertyName)) {
				throw new SCAException("This property \'" + propertyName + "`' is not declared in component:" + cmp);
			}
			
			Object value = scapc.getValue(propertyName);
			
			if(value != null) {
				if(clazz.isInstance(value)) {
					res = (T) value;
				} else {
					throw new SCAException("The property " + propertyName + " is not an instance of " + clazz + ": " + value);
				}
			}
		} catch (NoSuchInterfaceException e) {
			throw new SCAException(e);
		}
		return res ;
	}

	/**
	 * Change the name of the component
	 * 
	 * @param name
	 * @throws SCAException
	 */
	public void changeName(final Component comp, final String name) throws SCAException {
		try {
			final NameController nc = Fractal.getNameController(comp);
			nc.setFcName(name);
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to change the name ot this fractal componet: "
							+ comp, e);
		}
	}

	public String getName(final Component comp) {
		String res = null;
		try {
			if (comp != null) {
				final NameController nc = Fractal.getNameController(comp);
				res = nc.getFcName();
			}
		} catch (final NoSuchInterfaceException e) {
			// Do nothing
		}
		return res;
	}


	/**
	 * A utility function allowing to get a component from any content
	 * controller.
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param prefix
	 *            prefix used to find component(s)
	 * @return the component, null if not found
	 */
	public List<Component> getComponentsByNamePrefix(final Component parent,
			final String prefix) {
		// the component to be returned
		final List<Component> matchedComponents = new ArrayList<Component>();

		try {
			final ContentController parentContentController = Fractal
					.getContentController(parent);

			// the subcomponent content controller
			ContentController subContentController = null;

			// List content controller subcomponents
			for (final Component component : parentContentController
					.getFcSubComponents()) {

				// if the component is a composite, search the matching
				// component
				// recursively
				try {
					subContentController = Fractal
							.getContentController(component);

					if (subContentController.getFcSubComponents().length > 0) {
						matchedComponents.addAll(this.getComponentsByName(
								component, prefix));
					}
				} catch (final NoSuchInterfaceException e1) {
					// do nothing, return null
				}

				try {
					final String componentName = Fractal.getNameController(component)
							.getFcName();
					if ((componentName != null) && (componentName.startsWith(prefix))) {
						matchedComponents.add(component);
						break;
					}
				} catch (final NoSuchInterfaceException e) {
					// do nothing, return null
					// matchedComponent = null;
				}
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
			// matchedComponent = null;
		}
		return matchedComponents;
	}
	
	/**
	 * A utility function allowing to get a component from any content
	 * controller.
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param name
	 *            component name
	 * @return the component, null if not found
	 */
	public List<Component> getComponentsByName(final Component parent,
			final String name) {
		// the component to be returned
		final List<Component> matchedComponents = new ArrayList<Component>();

		try {
			final ContentController parentContentController = Fractal
					.getContentController(parent);

			// the subcomponent content controller
			ContentController subContentController = null;

			// List content controller subcomponents
			for (final Component component : parentContentController
					.getFcSubComponents()) {

				// if the component is a composite, search the matching
				// component
				// recursively
				try {
					subContentController = Fractal
							.getContentController(component);

					if (subContentController.getFcSubComponents().length > 0) {
						matchedComponents.addAll(this.getComponentsByName(
								component, name));
					}
				} catch (final NoSuchInterfaceException e1) {
					// do nothing, return null
				}

				try {
					final String componentName = Fractal.getNameController(component)
							.getFcName();
					if ((componentName != null) && (componentName.equals(name))) {
						matchedComponents.add(component);
						break;
					}
				} catch (final NoSuchInterfaceException e) {
					// do nothing, return null
					// matchedComponent = null;
				}
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
			// matchedComponent = null;
		}
		return matchedComponents;
	}

	/**
	 * A utility function allowing to get the first component with the given
	 * name.
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param name
	 *            component name
	 * @return the component, null if not found
	 */
	public Component getFirstComponentByName(final Component parent,
			final String name) {
		final List<Component> comps = this.getComponentsByName(parent, name);

		Component comp = null;
		if ((comps != null) && (comps.size() > 0)) {
			comp = comps.get(0);
		}

		return comp;
	}

	/**
	 * Return the server interface bound to the specified service reference.
	 * This method follows the delegation chain from the service reference to
	 * the client interface to the output interceptor to the bound server
	 * interface.
	 * 
	 * This method is meant to be a bridge from the Fraclet/Julia world to the
	 * Tinfi one. With Fraclet/Julia, when injecting a reference in a @Required
	 * annotated field, the reference of the target bound interface is
	 * injected. With Tinfi, the ServiceReference of the source component is
	 * injected. Since EasyViper assumes the first mode, we provide this method
	 * to facilitate the transition between both modes.
	 * 
	 * @param sr  the service reference
	 * @return    the bound server interface
	 * @author Lionel Seinturier <Lionel.Seinturier@univ-lille1.fr>
	 */
	public Interface getBoundInterface( ServiceReference<?> sr ) {
		ServiceReferenceImpl<?> cri = (ServiceReferenceImpl<?>) sr;
		Object service = cri._getDelegate();
		ComponentInterface ci = (ComponentInterface) service;
		Interceptor intercep = (Interceptor) ci.getFcItfImpl();
		Interface itf = (Interface) intercep.getFcItfDelegate();
		return itf;
	}

	public Component getComponentByInterface(final Component parent,
			final ServiceReference<?> sr, final String interfaceName) {

		Interface itf = getBoundInterface(sr);		
		return getComponentByInterface(parent,itf,interfaceName);
	}

	/**
	 * A utility function allowing to get a component from an interface
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param name
	 *            component name
	 * @return the component, null if not found
	 */
	public Component getComponentByInterface(final Component parent,
			final Interface itf, final String interfaceName) {
		// the component to be returned
		Component matchedComponent = null;
		try {
			final ContentController parentContentController = Fractal
					.getContentController(parent);
			// the subcomponent content controller
			ContentController subContentController = null;

			// List content controller subcomponents
			for (final Component component : parentContentController
					.getFcSubComponents()) {

				// if the component is a composite, search the matching
				// component
				// recursively
				try {
					subContentController = Fractal
							.getContentController(component);
					if (subContentController.getFcSubComponents().length > 0) {
						matchedComponent = this.getComponentByInterface(component,
								itf, interfaceName);
						if (matchedComponent != null) {
							break;
						}
					}
				} catch (final NoSuchInterfaceException e1) {
					// do nothing, return null
					matchedComponent = null;
				}

				try {
					final Interface itfC = (Interface) component
							.getFcInterface(interfaceName);
					if (itfC == itf) {
						matchedComponent = component;
						break;
					}
				} catch (final NoSuchInterfaceException e) {
					// do nothing, return null
					matchedComponent = null;
				}
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
			matchedComponent = null;
		}
		return matchedComponent;
	}

	/**
	 * A utility function allowing to get a component from any content
	 * controller.
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param name
	 *            component name
	 * @return the component, null if not found
	 */
	public List<Component> getComponents(final Component parent) {
		final List<Component> res = new ArrayList<Component>();

		try {
			final ContentController parentContentController = Fractal
					.getContentController(parent);

			// List content controller subcomponents
			for (final Component component : parentContentController
					.getFcSubComponents()) {
				res.add(component);
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
		}
		return res;
	}

	public List<Component> getAllSubComponents(final Component parent) {
		final List<Component> res = new ArrayList<Component>();

		try {
			final ContentController parentContentController = Fractal
					.getContentController(parent);

			// List content controller subcomponents
			for (final Component component : parentContentController
					.getFcSubComponents()) {
				res.add(component);
				res.addAll(getComponents(component));
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
		}
		return res;
	}

	/**
	 * A utility function allowing to get components interface binding to the
	 * client interface of the component controller.
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param name
	 *            component name
	 * @return the component, null if not found
	 */
	public Map<String, Interface> getServerInterfacesLinkedToClientInterfacesOfComponent(
			final Component component) {
		final Map<String, Interface> res = new HashMap<String, Interface>();

		try {
			final BindingController componentBindingController = Fractal
					.getBindingController(component);

			// List content controller subcomponents
			for (final String clientItfName : componentBindingController.listFc()) {
				final Interface itf = (org.objectweb.fractal.api.Interface) componentBindingController
						.lookupFc(clientItfName);
				res.put(clientItfName, itf);
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
		}
		return res;
	}

	/**
	 * A utility function allowing to get components interface binding to the
	 * client interface of the component controller.
	 * 
	 * @param parentContentController
	 *            parentContentController
	 * @param name
	 *            component name
	 * @return the component, null if not found
	 */
	public List<Component> getClientComponentsLinkedToServerInterfacesOfComponent(
			final Component parentComponent, final Interface itfOfComponent) {

		final List<Component> listOfcomponents = new ArrayList<Component>();

		try {
			final ContentController componentContentController = Fractal
					.getContentController(parentComponent);
			// List content controller subcomponents
			for (final Component component : componentContentController
					.getFcSubComponents()) {

				final Map<String, Interface> listOfItf = SCAHelper
						.getSCAHelper()
						.getServerInterfacesLinkedToClientInterfacesOfComponent(
								component);
				for (final Interface itf : listOfItf.values()) {
					if (itf == itfOfComponent) {
						if (!listOfcomponents.contains(component)) {
							listOfcomponents.add(component);
						}
					}
				}
			}
		} catch (final NoSuchInterfaceException e1) {
			// do nothing, return null
		} catch (final SCAException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return listOfcomponents;
	}

	/**
	 * A utility function allowing to get a component LifeCycleController from a
	 * content controller.
	 * 
	 * @param parentContentController
	 *            the parent content controller
	 * @param name
	 *            the name of the component
	 * @return the LifeCycleController of the component, null if not found
	 * @throws SCAException
	 */
	public LifeCycleController getLifeCycleControllerByName(
			final Component parent, final String name) throws SCAException {

		LifeCycleController lifeCycleController = null;

		final List<Component> comps = this.getComponentsByName(parent, name);

		if ((comps != null) && (comps.size() > 1)) {
			throw new SCAException("Several component with the same name");
		}

		Component comp = null;
		comp = comps.get(0);

		if (comp != null) {
			try {
				lifeCycleController = Fractal.getLifeCycleController(comp);
			} catch (final NoSuchInterfaceException e) {
				// do nothing, return null
				lifeCycleController = null;
			}
		}
		return lifeCycleController;
	}

	public boolean isStarted(final Component comp) {
		boolean res = false;
		if (comp != null) {
			try {
				final LifeCycleController lifeCycleController = Fractal
						.getLifeCycleController(comp);
				if (lifeCycleController.getFcState() == LifeCycleController.STARTED) {
					res = true;
				}
			} catch (final NoSuchInterfaceException e) {
				// do nothing, return null
				res = false;
			}
		}
		return res;
	}

	/**
	 * A utility function to start the given fractal component.
	 * 
	 * @param component
	 *            the fractal component
	 * 
	 * @return true if the component was found and stopped, false otherwise
	 * @throws NoSuchInterfaceException :
	 *             impossible to stop the component
	 * @throws IllegalLifeCycleException :
	 *             impossible to stop the component
	 */
	public boolean startComponent(final Component component)
			throws SCAException {
		boolean result = false;
		if (component != null) {
			try {
				final LifeCycleController lifeCycleController = Fractal
						.getLifeCycleController(component);

				if (lifeCycleController != null) {
					if (lifeCycleController.getFcState().equals(
							LifeCycleController.STOPPED)) {
						lifeCycleController.startFc();
						//						List<Component> children = SCAHelper.getSCAHelper().getComponents(component);
						//						for(Component child : children){
						//							startComponent(child);
						//						}
						result = true;
					}
				}
			} catch (final NoSuchInterfaceException e) {
				throw new SCAException("Impossible to start the component",
						e);
			} catch (final IllegalLifeCycleException e) {
				throw new SCAException("Impossible to start the component",
						e);
			}
		}
		return result;
	}

	public boolean startComponentAndSubComponents(final Component component)
			throws SCAException {
		boolean result = false;
		if (component != null) {
			try {
				final LifeCycleController lifeCycleController = Fractal
						.getLifeCycleController(component);

				if (lifeCycleController != null) {
					if (lifeCycleController.getFcState().equals(
							LifeCycleController.STOPPED)) {
						lifeCycleController.startFc();
						result = true;
					}
					List<Component> children = SCAHelper.getSCAHelper().getComponents(component);
					for(Component child : children){
						startComponent(child);
					}
				}
			} catch (final NoSuchInterfaceException e) {
				throw new SCAException("Impossible to start the component",
						e);
			} catch (final IllegalLifeCycleException e) {
				throw new SCAException("Impossible to start the component",
						e);
			}
		}
		return result;
	}

	/**
	 * A utility function to stop the given fractal component.
	 * 
	 * @param component
	 *            the fractal component to stop
	 * 
	 * @return true if the component was found and stopped, false otherwise
	 * @throws NoSuchInterfaceException :
	 *             impossible to stop the component
	 * @throws IllegalLifeCycleException :
	 *             impossible to stop the component
	 */
	public boolean stopComponent(final Component component)
			throws SCAException {

		boolean result = false;
		try {
			final LifeCycleController lifeCycleController = Fractal
					.getLifeCycleController(component);
			if (lifeCycleController.getFcState().equals(
					LifeCycleController.STARTED)) {
				lifeCycleController.stopFc();
				result = true;
			}
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException("Impossible to stop the component", e);
		} catch (final IllegalLifeCycleException e) {
			throw new SCAException("Impossible to stop the component", e);
		}
		return result;
	}

	/**
	 * A utility function to stop the given fractal composite.
	 * 
	 * @param composite
	 * @throws NoSuchInterfaceException
	 * @throws IllegalLifeCycleException
	 * @throws NoSuchInterfaceException
	 */
	public void stopAllSubComponents(final Component composite)
			throws SCAException {
		try {
			// the subcomponent content controller

			ContentController subContentController = null;
			final ContentController parentContentController = Fractal
					.getContentController(composite);

			// List content controller sub components
			for (int i = parentContentController.getFcSubComponents().length; i != 0; i--) {
				final Component component = parentContentController
						.getFcSubComponents()[i - 1];
				// if the component is a composite, stop the composite
				// recursively
				try {
					subContentController = Fractal
							.getContentController(component);
					if (subContentController.getFcSubComponents().length > 0) {
						this.stopAllSubComponents(component);
					}
				} catch (final NoSuchInterfaceException e) {
					// stop the component
					final LifeCycleController lifeCycleController = Fractal
							.getLifeCycleController(component);
					if (lifeCycleController.getFcState().equals(
							LifeCycleController.STARTED)) {
						lifeCycleController.stopFc();
					}
				}
			}
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException("Impossible to stop the component", e);
		} catch (final IllegalLifeCycleException e) {
			throw new SCAException("Impossible to stop the component", e);
		}
	}

	/**
	 * Add a component in a composite component.
	 * 
	 * @param newComponent
	 *            The new component
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public void addComponent(final Component newComponent,
			final Component parentComponent, final List<Binding> listOfBindings)
					throws SCAException {
		try {
			if (parentComponent == null) {
				throw new NullPointerException("The parent component of "
						+ newComponent + " cannot be null");
			}

			// Get the parent controller
			final ContentController parentContentController =  Fractal
					.getContentController(parentComponent);

			//SCAContentController scaCC = ((SCAContentController)parentComponent.getFcInterface("/sca-content-controller"));

			parentContentController.addFcSubComponent(newComponent);

			SCAHelper.getSCAHelper().addBindings(newComponent,
					listOfBindings);

		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to add a new fractal component", e);
		} catch (final IllegalContentException e) {
			throw new SCAException(
					"Impossible to add a new fractal component", e);
		} catch (final IllegalLifeCycleException e) {
			throw new SCAException(
					"Impossible to add a new fractal component", e);
		}
	}

	/**
	 * Delete a component in a composite component.
	 * 
	 * @param oldComponent
	 *            The component to delete
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public synchronized void deleteComponent(final Component oldComponent)
			throws SCAException {
		try {

			final Component parentComponent = this.getParent(oldComponent);

			if (parentComponent != null) {
				final List<Binding> listOfBindings = new ArrayList<Binding>();

				final BindingController bc = Fractal
						.getBindingController(oldComponent);

				for (final String clientItf : bc.listFc()) {
					try {
						bc.unbindFc(clientItf);
					} catch (final IllegalBindingException e) {
						// do nothing => itf not binded
					}
				}

				final List<Component> clientsOfComp = this.getClientComponentsLinkedToServerInterfacesOfComponent(
						parentComponent, (Interface) oldComponent
						.getFcInterface("service"));

				for (final Component clientComponent : clientsOfComp) {
					final BindingController bcClient = Fractal
							.getBindingController(clientComponent);
					for (final String clientItf : bcClient.listFc()) {
						if (bcClient.lookupFc(clientItf) == (Interface) oldComponent
								.getFcInterface("service")) {
							try {
								bcClient.unbindFc(clientItf);
							} catch (final IllegalBindingException e) {
								// do nothing => itf not binded
							}
						}
					}
				}

				this.deleteComponent(oldComponent, parentComponent, listOfBindings);
			}
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException("Impossible to delete component", e);
		} catch (final IllegalLifeCycleException e) {
			throw new SCAException("Impossible to delete component", e);
		}
	}

	/**
	 * Delete a component in a composite component.
	 * 
	 * @param oldComponent
	 *            The component to delete
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public synchronized void deleteComponent(final Component oldComponent,
			final Component parentComponent, final List<Binding> listOfBindings)
					throws SCAException {
		try {
			if (parentComponent == null) {
				throw new NullPointerException("The parent component "
						+ oldComponent + " cannot be null");
			}

			// Get the parent controller
			SCAExtendedContentController parentContentController = (SCAExtendedContentController)
					parentComponent.getFcInterface(SCAContentController.NAME);

			//final ContentController parentContentController = Fractal
			//		.getContentController(parentComponent);


			SCAHelper.getSCAHelper().deleteBindings(oldComponent,
					listOfBindings);

			
			parentContentController.releaseFcContent(oldComponent, true); 
			//removeFcSubComponent(oldComponent);
			
			ContentController contentController = Fractal.getContentController(parentComponent);
			contentController.removeFcSubComponent(oldComponent);
			
//	         if (SCAHelper.getSCAHelper().isStarted(oldComponent)) {
//	                SCAHelper.getSCAHelper().stopComponent(oldComponent);
//	            }


		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to remove the fractal component", e);
		} catch (IllegalContentException e) {
		    throw new SCAException(
                    "Impossible to remove the fractal component", e);
        } catch (IllegalLifeCycleException e) {
            throw new SCAException(
                    "Impossible to remove the fractal component", e);
        } 
	}

	/**
	 * Add a component in a composite component.
	 * 
	 * @param newComponent
	 *            The new component
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public void addBindings(final Component component,
			final List<Binding> listOfBindings) throws SCAException {
		try {

			// Get the binding controller of the new component
			final BindingController cBindingController = Fractal
					.getBindingController(component);

			// Add all the bindings
			if (listOfBindings != null) {
				for (int i = 0; i < listOfBindings.size(); i++) {
					cBindingController.bindFc(listOfBindings.get(i)
							.getClientInterfaceName(), (listOfBindings.get(i))
							.getServerInterface());
				}
			}
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		} catch (final IllegalLifeCycleException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		} catch (final IllegalBindingException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		}
	}

	public boolean isBinded(final Component component,
			final String clientItfName) throws SCAException {
		boolean res = true;
		try {

			// Get the binding controller of the new component
			final BindingController cBindingController = Fractal
					.getBindingController(component);

			// Add all the bindings
			final SCAComponent o = (SCAComponent) cBindingController.lookupFc(clientItfName);

			if (o == null) {
				res = false;
			}
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		}
		return res;
	}

	public boolean isAlreadyBind(final Component component,
			final String clientItfName, final ServiceReference<?> sr)
					throws SCAException {

		Interface itf = getBoundInterface(sr);
		return isAlreadyBind(component,clientItfName,itf);
	}

	public boolean isAlreadyBind(final Component component,
			final String clientItfName, final Interface itf)
					throws SCAException {
		try {

			// Get the binding controller of the new component
			final BindingController cBindingController = Fractal
					.getBindingController(component);

			// Add all the bindings
			final Interface o = (Interface) cBindingController.lookupFc(clientItfName);

			return (o == itf);
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException("Impossible to get the interface", e);
		}
	}

	public Map<String, Object> getListOfBindings(final Component component) throws SCAException{
		Map<String, Object> res = new HashMap<String, Object>();
		// Get the binding controller of the new component
		try {
			final BindingController cBindingController = Fractal
					.getBindingController(component);

			String[] binds = cBindingController.listFc();

			for(int i=0 ; i < binds.length ; i++){
				res.put(binds[i], cBindingController.lookupFc(cBindingController.listFc()[i]));
			}

		} catch (NoSuchInterfaceException e) {
			throw new SCAException(e.getLocalizedMessage());
		}

		return res;

	}


	public List<String> getListOfBinds(final Component component)
			throws SCAException {
		final List<String> res = new ArrayList<String>();
		try {

			// Get the binding controller of the new component
			final BindingController cBindingController = Fractal
					.getBindingController(component);

			// Get all the bindings
			for (int i = 0; i < cBindingController.listFc().length; i++) {
				res.add(cBindingController.listFc()[i]
						+ " link to "
						+ cBindingController.lookupFc(cBindingController
								.listFc()[i]));
			}

		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		}
		return res;
	}

	public Component getParent(final Component childrenComponent)
			throws SCAException {
		Component parent = null;
		try {
			final SuperController superController = Fractal
					.getSuperController(childrenComponent);

			Component parents[] = superController.getFcSuperComponents();
			if (parents.length > 1) {
				throw new SCAException("This component ("
						+ childrenComponent
						+ ") has several parents: It is a shared component");
			}
			if (parents.length == 1) {
				parent = parents[0];
			}
		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(e);
		}
		return parent;
	}

	/**
	 * Delete a component in a composite component.
	 * 
	 * @param newComponent
	 *            The new component
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public void deleteBindings(final Component component,
			final List<Binding> listOfBindings) throws SCAException {
		try {



			// Get the binding controller of the new component
			final BindingController cBindingController = Fractal
					.getBindingController(component);

			// Add all the bindings
			if (listOfBindings != null) {
				for (int i = 0; i < listOfBindings.size(); i++) {
					Binding b = listOfBindings.get(i);

					String cltItfName = b.getClientInterfaceName();

					cBindingController.unbindFc(cltItfName);
				}
			}

		} catch (final NoSuchInterfaceException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		} catch (final IllegalLifeCycleException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		} catch (final IllegalBindingException e) {
			throw new SCAException(
					"Impossible to bind the fractal component", e);
		}
	}

	/**
	 * Delete all client binding of component.
	 * 
	 * @param newComponent
	 *            The new component
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public void deleteAllClientBindingsOfComponent(final Component component)
			throws SCAException {
		final Map<String, Interface> map = this
				.getServerInterfacesLinkedToClientInterfacesOfComponent(component);
		final List<Binding> bindings = new ArrayList<Binding>();
		for (final Entry<String, Interface> entry : map.entrySet()) {
			bindings.add(new Binding(entry.getKey(), entry.getValue()));
		}
		this.deleteBindings(component, bindings);
	}

	/**
	 * Delete all client binding of component.
	 * 
	 * @param newComponent
	 *            The new component
	 * @param parentComponent
	 *            The parent component
	 * @param listOfBindings
	 *            the list of binding to create between this new component and
	 *            the others
	 * @throws NoSuchInterfaceException :
	 *             Impossible to add a component
	 * @throws IllegalLifeCycleException :
	 *             Impossible to add a component
	 * @throws IllegalContentException :
	 *             Impossible to add a component
	 * @throws IllegalBindingException :
	 *             Impossible to add a component
	 */
	public void deleteLinkWithAnItfClientOfComponent(final Component component,
			final String itfCLient) throws SCAException {
		final Map<String, Interface> map = this
				.getServerInterfacesLinkedToClientInterfacesOfComponent(component);

		final List<Binding> bindings = new ArrayList<Binding>();
		for (final Entry<String, Interface> entry : map.entrySet()) {
			if (entry.getKey().equals(itfCLient)) {
				bindings.add(new Binding(entry.getKey(), entry.getValue()));
				break;
			}
		}
		this.deleteBindings(component, bindings);
	}

	/**
	 * Create an explorer for the specified component (the component requires an
	 * explicit interface named fcAppl of type
	 * org.objectweb.fractal.api.Component).
	 * 
	 * @param component
	 *            the component to explore
	 * @throws ADLException :
	 *             occurs when it is impossible to create the explorer
	 * @throws NoSuchInterfaceException :
	 *             occurs when it is impossible to create the explorer
	 * @throws IllegalBindingException :
	 *             occurs when it is impossible to create the explorer
	 * @throws IllegalLifeCycleException :
	 *             occurs when it is impossible to create the explorer
	 * @throws IllegalContentException :
	 *             occurs when it is impossible to create the explorer
	 * @throws SCAException
	 */
	public Component createExplorer(final String fractalName, final String fractalDefinition)
			throws SCAException {
		// do nothing
		return null;
	}

}
