/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2008 OW2 Consortium, http://www.ow2.org/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.easycommons.xml;

import javax.xml.stream.XMLOutputFactory;

import com.ebmwebsourcing.easycommons.pooling.GenericResourcePool;
import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;
import com.ebmwebsourcing.easycommons.pooling.ResourceHandler;

/**
 * This class represent a {@link XMLOutputFactory} pool, which provided {@link XMLOutputFactory}
 * object as resource.
 * 
 * @author noddoux
 */
public class XMLOutputFactoryResourcePool {

    /**
     * {@link XMLOutputFactory} resource handler to manage {@link XMLOutputFactory} life cycle methods
     */
    private class XMLOutputFactoryResourceHandler implements ResourceHandler<XMLOutputFactory> {

        public XMLOutputFactoryResourceHandler() {
        }

        @Override
        public XMLOutputFactory create() {
            return XMLOutputFactory.newInstance();
        }

        @Override
        public void onRelease(XMLOutputFactory xmlOutputFactory) {
        }

        @Override
        public void onTake(XMLOutputFactory xmlOutputFactory) {            
        }
    }
    
    private final GenericResourcePool<XMLOutputFactory> xmlOutputFactoryPool;

    /**
     * Allowing to instantiate a new {@link XMLOutputFactoryResourcePool} containing {@link XMLOutputFactory} resources.
     * 
     * @param minPoolSize
     *            The minimum number of {@link XMLOutputFactory} instances in the pool (created at the
     *            initialization).
     * @param maxPoolSize
     *            the maximum number of {@link XMLOutputFactory} instances in the current pool (limit of the
     *            pool). It must be greater or equals to the specified minSize.
     *            The maximum value is Integer.MAX_VALUE
     * @param poolPolicy
     *            the {@link PoolPolicy} to adopt when the maximum size is reached. it
     *            must not be null.
     */
    public XMLOutputFactoryResourcePool(int minPoolSize, int maxPoolSize,
            PoolPolicy poolPolicy) {
        XMLOutputFactoryResourceHandler xmlOutputFactoryResourceHandler = new XMLOutputFactoryResourceHandler();
        this.xmlOutputFactoryPool = new GenericResourcePool<XMLOutputFactory>(
                xmlOutputFactoryResourceHandler, minPoolSize, maxPoolSize, poolPolicy);
    }

    /**
     * Take one unused {@link XMLOutputFactory} in the current pool. After getting a
     * {@link XMLOutputFactory} from the pool and before returning a
     * {@link XMLOutputFactory}, the method onTake() of the {@link XMLOutputFactory}
     * resource handler is called.
     * 
     * @return one {@link XMLOutputFactory}
     * 
     * @throws PoolException
     *             if the current thread is interrupted for the pool policy WAIT
     *             or if there is no more available resource in the pool for the
     *             pool policy REJECT
     * 
     */
    public XMLOutputFactory take() {
        return this.xmlOutputFactoryPool.take();
    }

    /**
     * Release the specified {@link XMLOutputFactory} After putting back the
     * {@link XMLOutputFactory} in the pool, the method onRelease() of the
     * {@link XMLOutputFactory} resource handler is called.
     * 
     * @param xmlOutputFactory
     *            The {@link XMLOutputFactory} to release
     */
    public final void release(final XMLOutputFactory xmlOutputFactory) {
        this.xmlOutputFactoryPool.release(xmlOutputFactory);
    }
}
