/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2008 OW2 consortium (www.ow2.org)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id: XMLUtil.java 98 2006-02-24 16:18:48Z alouis $
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.easycommons.xml;

import java.io.OutputStream;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;

/**
 * A static pool of {@link XMLOutputFactory} to improve {@link XMLOutputFactory} creation efficiency. 
 * A {@link XMLOutputFactory} is initialized the first time this class is used.
 * An infinite number (integer maximum value) of {@link XMLOutputFactory} may be created. They are never evicted.
 * 
 * @ author noddoux
 */
public final class XMLOutputFactories {

    private static final XMLOutputFactoryResourcePool xmlOutputFactoryPool = new XMLOutputFactoryResourcePool(1, Integer.MAX_VALUE, PoolPolicy.WAIT);

    /**
     * Take a {@link XMLOutputFactory} from the pool
     * 
     * @return one {@link XMLOutputFactory}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link XMLOutputFactory}
     *             of the pool and is interrupted
     */
    public final static XMLOutputFactory takeXMLOutputFactory() throws PoolException {
       return xmlOutputFactoryPool.take();
    }

    /**
     * Release a {@link XMLOutputFactory} to the pool
     * 
     * @param xmlOutputFactory a {@link XMLOutputFactory} to release
     */
    public final static void releaseXMLOutputFactory(XMLOutputFactory xmlOutputFactory) {
        xmlOutputFactoryPool.release(xmlOutputFactory);
    }

    /**
     * Create a {@link XMLStreamWriter} from a specified {@link OutputStream} by
     * using the pool of {@link XMLOutputFactory}
     * 
     * @param outputStream
     *            an {@link OutputStream}
     * 
     * @return the {@link XMLStreamWriter}
     * 
     * @throws XMLStreamException
     *             if an unexpected processing errors occurs
     */
    public final static XMLStreamWriter createXMLStreamWriter(OutputStream outputStream)
            throws XMLStreamException {
        XMLOutputFactory xmlOutputFactory = null;

        try {
            xmlOutputFactory = takeXMLOutputFactory();
        } finally {
            if (xmlOutputFactory != null) {
                takeXMLOutputFactory();
            }
        }
        XMLStreamWriter writer = xmlOutputFactory.createXMLStreamWriter(outputStream);

        return writer;
}
}
