/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.xml;

import javax.xml.transform.Transformer;

import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;

/**
 * A static pool of {@link Transformer} to improve {@link Transformer} creation efficiency. 
 * A {@link Transformer} is initialized the first time this class is used.
 * An infinite number (integer maximum value) of {@link Transformer} may be created. They are never evicted.
 * 
 * @author Nicolas Oddoux - EBM WebSourcing
 */
public final class Transformers {

    private static final TransformerResourcePool transformerPool = new TransformerResourcePool(1, Integer.MAX_VALUE, PoolPolicy.WAIT);

    /**
     * Take a {@link Transformer} from the pool
     * 
     * @return one {@link Transformer}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link Transformer}
     *             of the pool and is interrupted
     */
    public final static Transformer takeTransformer() throws PoolException {
       return transformerPool.take();
    }

    /**
     * Release a {@link Transformer} to the pool
     * 
     * @param transformer a {@link Transformer} to release
     */
    public final static void releaseTransformer(Transformer transformer) {
        transformerPool.release(transformer);
    }
}
