/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.xml;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

import com.ebmwebsourcing.easycommons.stream.InputStreamForker;
import com.ebmwebsourcing.easycommons.stream.ReaderInputStream;

/**
 * 
 * @author Mathieu CARROLLE - EBM WebSourcing
 * 
 */
public class SourceHelper {

    public static void toFile(Source source, File fileToWrite) throws TransformerException, IOException {
        final FileWriter writer = new FileWriter(fileToWrite);
        final Result result = new StreamResult(writer);
        
        Transformer transformer = Transformers.takeTransformer();
        try {
            transformer.transform(source, result);
        } finally {
            Transformers.releaseTransformer(transformer);
            writer.close();
        }
    }
    
    public static String toString(Source source) throws TransformerException {
        final StringWriter writer = new StringWriter();
        final Result result = new StreamResult(writer);
        
        Transformer transformer = Transformers.takeTransformer();
        try {
            transformer.transform(source, result);
        } finally {
            Transformers.releaseTransformer(transformer);
            // It's not needed to close a StringWriter (has no effect)
        }
        
        return writer.toString();
    }

    private static final Source forkStreamSource(final StreamSource source) throws IOException {
        InputStreamForker streamForker;
        if (source.getInputStream() != null) {
            streamForker = new InputStreamForker(source.getInputStream());
        } else {
            streamForker = new InputStreamForker(new ReaderInputStream(source.getReader()));
        }
        source.setInputStream(streamForker.fork());
        return new StreamSource(streamForker.fork());
    }

    /**
     * Fork, if necessary, a {@link Source} so that consuming the forked one
     * does not consume the original one.
     * 
     * @param source
     *            {@link Source} to be forked.
     * @return Forked {@link Source}.
     * @throws <IOException>
     */
    public static Source fork(Source source) throws IOException {
        if (source instanceof DOMSource) {
            return source;
        } else if (source instanceof StreamSource) {
            return forkStreamSource((StreamSource) source);
        } else {
            throw new IllegalArgumentException(
                    "Only DOMSource or StreamSource are supported right now.");
        }
    }

    /**
     * Utility to get the bytes uri
     *
     * @param source the resource to get
     * @param resolvedUri 
     * @param publicId 
     */
    public static InputSource sourceToInputSource(Source source, String publicId, String systemId) {
        InputSource is = null;
        if (source instanceof SAXSource) {
            is = ((SAXSource) source).getInputSource();
        } else if (source instanceof DOMSource) {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            Node node = ((DOMSource) source).getNode();
            if (node instanceof Document) {
                node = ((Document) node).getDocumentElement();
            }
            Element domElement = (Element) node;
            ElementToStream(domElement, baos);
            InputSource isource = new InputSource(source.getSystemId());
            isource.setByteStream(new ByteArrayInputStream(baos.toByteArray()));
            is = isource;
        } else if (source instanceof StreamSource) {
            StreamSource ss = (StreamSource) source;
            InputSource isource = new InputSource(ss.getSystemId());
            isource.setByteStream(ss.getInputStream());
            isource.setCharacterStream(ss.getReader());
            isource.setPublicId(ss.getPublicId());
            is = isource;
        } else {
            is =  new InputSource(source.getSystemId());
        }
        if(is != null) {
            is.setPublicId(publicId);
            is.setSystemId(systemId);
        }
        return is;
    }

    private static void ElementToStream(Element element, OutputStream out) {
        DOMSource source = new DOMSource(element);
        StreamResult result = new StreamResult(out);
        
        Transformer transformer = Transformers.takeTransformer();
        try {
            transformer.transform(source, result);
        } catch (Exception ex) {
        } finally {
            Transformers.releaseTransformer(transformer);
        }
    }
}
