/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easyschema10.api;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPathExpressionException;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easybox.api.XmlObjectXPathEvaluator;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easyschema10.api.element.ComplexType;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Restriction;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easyschema10.api.element.SimpleContent;
import com.ebmwebsourcing.easyschema10.api.element.SimpleType;
import com.ebmwebsourcing.easyschema10.api.type.Type;

public final class SchemaHelper {

	private static final QName XMLSCHEMA_INTEGER_TYPE_QNAME =
			new QName(Constants.XMLSCHEMA_NS_URI, "integer");
    private static final QName XMLSCHEMA_ANYSIMPLETYPE_TYPE_QNAME =
            new QName(Constants.XMLSCHEMA_NS_URI, "anySimpleType");
	
    private SchemaHelper() {
    }


    private static final <X extends XmlObjectNode> X doSelectSingleXmlObjectNode(
            XmlObject xmlObjectContextNode, String xpathStr,
            Class<X> resultInterfaceClass) {
        XmlObjectXPathEvaluator xpathEvaluator = xmlObjectContextNode.getXmlContext()
        .createXPathEvaluator();
        try {
            X result = xpathEvaluator
                    .selectSingleXmlObjectNode(xmlObjectContextNode,
                            String.format(
                                    xpathStr), 
                                    resultInterfaceClass);
            return result;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }        
    }
    
    private static <X extends XmlObjectNode> X[] doSelectXmlObjectNodes(
            XmlObject xmlObjectContextNode, String xpathStr,
            Class<X> resultInterfaceClass) {
        XmlObjectXPathEvaluator xpathEvaluator = xmlObjectContextNode.getXmlContext()
        .createXPathEvaluator();
        try {
            X[] result = xpathEvaluator
                    .selectXmlObjectNodes(xmlObjectContextNode,
                            String.format(
                                    xpathStr), 
                                    resultInterfaceClass);
            return result;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }        
    }
    

    /**
     * Get parent {@link Schema}.
     * 
     * @return Closest {@link Schema} object in ancestors hierarychy, or {@code null}
     *  if no such ancestor.
     */
    public static Schema findParentSchema(XmlObject contextObject) {
        XmlObject current = contextObject;
        while (current != null) {
            if (current instanceof Schema) return (Schema) current;
            current = current.getXmlObjectParent();
        }
        return null;
    }
    
    
    public static Type[] findTypes(XmlObject contextXmlObject) {
        return doSelectXmlObjectNodes(contextXmlObject, 
                String.format(
                "//*[(local-name() = 'simpleType') or (local-name() = 'complexType')][@name]"), 
                Type.class);
    }
    
    public static Type findTypeByQName(XmlObject contextXmlObject, QName typeQName) {
        if (Constants.XMLSCHEMA_NS_URI.equals(typeQName.getNamespaceURI())) {
            Type builtInType = SchemaOfSchemas.getSchema().getSimpleTypeByName(typeQName.getLocalPart());
            if (builtInType == null) {
                // maybe a complex type then ?
                builtInType = SchemaOfSchemas.getSchema().getComplexTypeByName(typeQName.getLocalPart());
            }
            return builtInType;
        }
        
        return doSelectSingleXmlObjectNode(contextXmlObject, 
                String.format(
                "//*[(local-name() = 'simpleType') or (local-name() = 'complexType')][@name = '%s'][ancestor::*/@targetNamespace = '%s']",
                typeQName.getLocalPart(), typeQName.getNamespaceURI()), Type.class);
        
    }

    
    public static Element[] findElements(XmlObject contextXmlObject) {
        return doSelectXmlObjectNodes(contextXmlObject, 
                String.format("//xs:element[@name][parent::xs:schema]"),
                Element.class);
        
    }
    
    
    public static Element findElementByQName(XmlObject contextXmlObject, QName elementQName) {
        return doSelectSingleXmlObjectNode(contextXmlObject, 
                String.format(
                        "//xs:element[@name = '%s']" + 
                        (elementQName.getNamespaceURI().isEmpty() ? "" : "[ancestor::xs:schema/@targetNamespace = '%s']"),
                        elementQName.getLocalPart(), elementQName.getNamespaceURI()), 
                        Element.class);        
    }


    
	public static boolean isTypeDerivedFromInteger(XmlObject contextXmlObject, QName typeQName) {
	    return isTypeDerivedFrom(contextXmlObject, typeQName, XMLSCHEMA_INTEGER_TYPE_QNAME);
	}

    public static boolean isTypeDerivedFromAnySimpleType(XmlObject contextXmlObject, QName typeQName) {
        return isTypeDerivedFrom(contextXmlObject, typeQName, XMLSCHEMA_ANYSIMPLETYPE_TYPE_QNAME);
    }
	
    public static boolean isTypeDerivedFrom(XmlObject contextXmlObject, QName typeQName, QName baseQName) {
        if (baseQName.equals(typeQName)) return true;
        Type type = findTypeByQName(contextXmlObject, typeQName);
        if (type instanceof SimpleType) {
            SimpleType simpleType = (SimpleType) type;
            if (simpleType.hasRestriction()) {
                Restriction restriction = simpleType.getRestriction();
                return isTypeDerivedFromInteger(contextXmlObject, restriction.getBase());
            } else {
                // TODO : check against list or union simple types. not handled right now
                return false;
            }
            
        } else {
            ComplexType complexType = (ComplexType) type;
            if (complexType.hasSimpleContent()) {
                SimpleContent simpleContent = complexType.getSimpleContent();
                if (simpleContent.hasExtension()) {
                    assert simpleContent.getExtension().getBase() != null;
                    return isTypeDerivedFromInteger(contextXmlObject, 
                            simpleContent.getExtension().getBase());
                } else {
                    assert simpleContent.getRestriction().getBase() != null;
                    return isTypeDerivedFromInteger(contextXmlObject, 
                            simpleContent.getRestriction().getBase());
                }
            }
        }
        return false;
    }


        
    
}
