/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Logger;

import org.oasisopen.sca.annotation.PolicySets;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.osoa.sca.annotations.Property;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.expression.Assigner;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalContext;
import com.ebmwebsourcing.easyviper.core.api.model.registry.ProcessKey;
import com.ebmwebsourcing.easyviper.core.api.soa.Partner;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value=Process.class,names="service")
@PolicySets("frascati:scaEasyCompositeWithContent")
public class ProcessImpl extends ScopeImpl implements Process {

	private Logger log = Logger.getLogger(ProcessImpl.class.getSimpleName());

	@Property(name="keys", required=true)
	private List<ProcessKey> keys;

	/**
	 * TODO jlesbegu
	 * A process does not have anymore a unique context
	 * for all endpoints but a Map indexing contexts according to
	 * an endpoint and an operation (local part of Interface wsdl definition)  
	 */
	@Property(name="externalContexts", required=false)
	private Map<Partner, Map<String,ExternalContext>> externalContexts; 

	private Assigner assigner = null;

	private boolean isUnstable;

	private AtomicReference<Execution> execution;
	
	private String executionName;

	public ProcessImpl() throws CoreException {
		super();
		assigner = null;
		externalContexts = Collections.synchronizedMap(new HashMap<Partner, Map<String,ExternalContext>>());
		isUnstable = false;
		execution = new AtomicReference<Execution>();
	}


	public Engine getEngine() throws CoreException {
		Engine res = null;
		try {
			final Component engineComp = SCAHelper.getSCAHelper().getParent(
					this.getComponent());
			res = (Engine) engineComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}
		return res;
	}

	@Override
	public com.ebmwebsourcing.easyviper.core.api.engine.Process getProcess()
			throws CoreException {
		return this;
	}

	public Map<Partner, Map<String,ExternalContext>> getExternalContexts() {
		return this.externalContexts;
	}


	public ExternalContext getExternalContext(Partner serviceendpoint, String operation){
		Map<String, ExternalContext> map = this.externalContexts.get(serviceendpoint);
		if(map != null){
			return map.get(operation);
		}else{
			return null;
		}
	}

	@Override
	public void setLog(final Logger logger) {
		super.setLog(logger);
		this.log = logger;
	}




	@Override
	public void end(boolean clean) throws CoreException {
		this.log.fine("PROCESS ENDED : " + this.getName());
		for(Map<String, ExternalContext> mapOperationContext : externalContexts.values()){
			for(ExternalContext externalContext : mapOperationContext.values()){
				if(externalContext != null && externalContext.getTerminationHandler() != null) {
					externalContext.getTerminationHandler().end(this);
				}
			}
		}
		this.getBehaviour().reset();
		super.end(clean);

	}


	public List<ProcessKey> getProcessKeys() {
		return keys;
	}


	public void setProcessKeys(List<ProcessKey> keys) {
		this.keys = keys;
	}


	public Assigner getAssigner() {
		return assigner;
	}

	public void setAssigner(Assigner assigner) {
		this.assigner = assigner;
	}

	@Override
	public void addExternalContext(Partner partner, String operation,
			ExternalContext context) {
		if (partner == null) return;

		Map<String, ExternalContext> map = this.externalContexts.get(partner);
		if(map == null){
			map = Collections.synchronizedMap(new HashMap<String, ExternalContext>());
		}
		map.put(operation, context);
		this.externalContexts.put(partner, map);
		log.fine("add external context ... "+this.externalContexts);	

	}



	@Override
	public void clearExternalContexts() {
		this.externalContexts.clear();
	}

	public Behaviour findBehaviour(String activityName){
		Behaviour behaviour = null;
		try {
			String nodeName = "node_supporting_"+activityName;
			Node n = this.getNodeByName(nodeName);
			if(n != null){
				Component c = SCAHelper.getSCAHelper().getComponentsByName(this.getComponent(), nodeName).get(0);
				SCAHelper.getSCAHelper().startComponent(c);
				Component cb = SCAHelper.getSCAHelper().getComponentsByName(this.getComponent(), activityName).get(0);
				SCAHelper.getSCAHelper().startComponent(cb);

				return n.getBehaviour();
			}

		} catch (CoreException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}catch (SCAException e) {
			e.printStackTrace();
		}

		return behaviour;
	}


	@Override
	public final boolean isUnstable() {
		return isUnstable;
	}

	@Override
	public void setUnstable(boolean unstable) {
		isUnstable = unstable;

	}	

	@Override
	public boolean hasExecution() {
		return execution.get() != null;
	}


	@Override
	public Execution getExecution() {
		assert hasExecution();
		return execution.get();
	}


	@Override
	public void setExecution(Execution execution) {
		this.execution.set(execution);
	}


	final public static ExternalContext getFirstExternalContext(Map<Partner, Map<String,ExternalContext>> contexts) {
		Map<String, ExternalContext> map = null;
		if(contexts.size() > 0) {
			map = contexts.values().iterator().next();
			if(map.values().size() > 0) {
				return map.values().iterator().next();
			} else {
				return null;
			}
		} else {
			return null;
		}
	}


	@Override
	public String getProcessExecutionName() {
		return this.executionName;
	}


	@Override
	public void setProcessExecutionName(String name) {
		this.executionName = name;
		
	}
}


