/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.jdom.Element;
import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Reference;
import org.oasisopen.sca.annotation.Scope;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.osoa.sca.annotations.Property;
import org.ow2.frascati.tinfi.api.control.SCAContentController;
import org.ow2.frascati.tinfi.control.content.SCAExtendedContentController;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easycommons.uuid.SimpleUUIDGenerator;
import com.ebmwebsourcing.easyviper.core.api.Core;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution.State;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.ReceiverBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.configuration.ConfigurationEngine;
import com.ebmwebsourcing.easyviper.core.api.engine.registry.ProcessInstanceRegistry;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.ServiceManager;
import com.ebmwebsourcing.easyviper.core.api.engine.time.TimerFinishedEvent;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalContext;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalEnvironment;
import com.ebmwebsourcing.easyviper.core.api.model.Model;
import com.ebmwebsourcing.easyviper.core.api.model.registry.ProcessKey;
import com.ebmwebsourcing.easyviper.core.api.model.registry.definition.ProcessDefinition;
import com.ebmwebsourcing.easyviper.core.api.soa.Partner;
import com.ebmwebsourcing.easyviper.core.api.soa.message.Message;
import com.ebmwebsourcing.easyviper.core.api.tools.PoolOfProcessForkers;
import com.ebmwebsourcing.easyviper.core.impl.engine.pattern.CreationPatternFactory;
import com.ebmwebsourcing.easyviper.core.impl.engine.registry.MemoryProcessInstanceRegistryImpl;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.service.ServiceManagerImpl;
import com.ebmwebsourcing.easyviper.core.impl.model.registry.ProcessKeyImpl;
import com.ebmwebsourcing.easyviper.core.impl.services.AutoFlushMessageServiceImpl;
import com.ebmwebsourcing.easyviper.extended.service.autotrash.api.AutoTrashProcessService;
import com.ebmwebsourcing.easyviper.extended.service.autotrash.impl.AutoTrashProcessServiceImpl;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = Engine.class, names = "service")
@PolicySets("frascati:scaEasyCompositeWithContent")
public class EngineImpl extends SCAComponentImpl implements Engine {

	Logger log = Logger.getLogger(EngineImpl.class.getSimpleName());

	private static final SimpleUUIDGenerator uuidGenerator = new SimpleUUIDGenerator();

	@Reference(name = "externalEnv", required = false)
	protected ExternalEnvironment externalEnvironment;

	@Reference(name = "model", required = false)
	private Model model;

	@Property(name = "processInstanceRegistry", required = true)
	private ProcessInstanceRegistry processInstanceRegistry = new MemoryProcessInstanceRegistryImpl(
			this);

	@Property(name = "serviceManager", required = true)
	private ServiceManager serviceManager = new ServiceManagerImpl(this);

	private PoolOfProcessForkers poolOfProcessForkers = null;

	private ConfigurationEngine configuration = null;

	/**
	 * Creates a new instance
	 * 
	 * @throws CoreException
	 */
	public EngineImpl() throws CoreException {
		poolOfProcessForkers = new PoolOfProcessForkers();
	}

	public synchronized Process createNewEmptyProcessInstance(final QName uniqueProcessName,
			ProcessDefinition processDefinition) throws CoreException {
		this.log.fine("start createNewEmptyProcessInstance");

		this.checkModel();

		// create an empty process
		Process process = (Process) CreationPatternFactory.getInstance().createProcessPattern(
				uniqueProcessName.toString(), this);

		final Component processDefinitionComp = process.getComponent();
		try {
			process = (Process) processDefinitionComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			this.log.severe(e.getMessage());
			e.printStackTrace();
			throw new CoreException(e);
		}

		if (this.model == null) {
			this.log.fine("store the process instance in registry in instance registry");
			// store in instance registry
			final ProcessKey key = new ProcessKeyImpl(uniqueProcessName, null, null);
			List<ProcessKey> keys = new ArrayList<ProcessKey>(1);
			keys.add(key);
			process.setProcessKeys(keys);

			this.processInstanceRegistry.storeProcessInstance(key, process);
		} else {
			List<ProcessKey> keys = this.model.getRegistry().createKeys(processDefinition);
			this.log.fine("Number of keys: " + keys.size());
			for (final ProcessKey key : keys) {
				this.log.fine("key put in process instances map: " + key);

				this.getProcessInstanceRegistry().storeProcessInstance(key, process);
			}
			process.setProcessKeys(keys);
		}
		this.log.fine("end of createNewEmptyProcessInstance");
		return process;
	}

	// TODO: Fix bug: unnessary methods
	private void checkModel() throws CoreException {
		try {
			if (this.model == null) {
				final Component coreComp = SCAHelper.getSCAHelper().getParent(this.getComponent());
				final Core core = (Core) coreComp.getFcInterface("service");
				this.model = core.getModel();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
	}

	// TODO: Fix bug: unnessary methods
	private void checkExternalEnvironment() throws CoreException {
		try {
			if (this.externalEnvironment == null) {
				final Component coreComp = SCAHelper.getSCAHelper().getParent(this.getComponent());
				final Core core = (Core) coreComp.getFcInterface("service");
				this.externalEnvironment = core.getExternalEnvironment();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
	}

	public synchronized void accept(final Message internalMessage, final ExternalContext context)
			throws CoreException {

		this.checkModel();

		// load the good instance
		Execution validExecution = this.acceptInternalMessage(internalMessage);

		if (validExecution == null) {
			this.log.fine("no process found => store the received message");
			AutoFlushMessageServiceImpl autoflush = this.serviceManager
					.getService(AutoFlushMessageServiceImpl.class);
			autoflush.addMessagesInRegistry(internalMessage, context);
		} else {
			// set the context
			// TODO Check Reflexion on what is a context (implements equality
			// ...)

			Process process = validExecution.getCurrentTarget().getProcess();

			Partner endpoint = findEndpointFromMessage(internalMessage, process);

			this.log.fine("set the context: " + context + " into process instance "
					+ process.getName());
			process.addExternalContext(endpoint, internalMessage.getOperationName(), context);

			restartExecution(validExecution, context);
		}
	}

	private final Collection<String> getAllActiveThreadNames() {
		Collection<String> result = new HashSet<String>();
		// TODO : not very nice, should be reimplemented with activeCount and
		// enumerate
		// caring about races...
		Map<Thread, StackTraceElement[]> threads = Thread.getAllStackTraces();
		for (Thread t : threads.keySet()) {
			result.add(t.getName());
		}
		return result;
	}

	private void restartProcess(Process process, ExternalContext context) throws CoreException {
		assert process != null;
		// check if there is already a thread running this execution.
		// we know it because we created each execution thread using
		// execution
		// hashCode.
		Execution execution = process.getExecution();
		restartExecution(execution, context);
	}

	@Override
	public void restartExecution(Execution execution, ExternalContext context) {
		Collection<String> threadNames = getAllActiveThreadNames();
		if (!threadNames.contains(String.valueOf(execution.getName()))) {
			log.fine("Spawning (or respawning) thread for execution " + execution.getName());
			ExecutionThread processThread = new ExecutionThread(execution, getCore()
					.getExternalEnvironment(), log);

			if (configuration.getSingleThreadedExecution()) {
				processThread.run();
			} else {
				processThread.start();
			}
		}
	}


	private final Execution findFirstAvailableExecution(final Message internalMessage) throws CoreException {
		assert processInstanceRegistry != null;

		Execution validExecution = null;
		List<Process> processes = new ArrayList<Process>();

		// find the good instances able to handle the message
		this.log.fine("try to find an available instance");
		processes = this.processInstanceRegistry.getProcessInstances(internalMessage);
		assert processes != null;
		this.log.fine("number of potential process: " + processes.size());

		if (processes.isEmpty())
			return null;

		List<Execution> possibleExecution = new ArrayList<Execution>();
		boolean reinitOrSuspended = false;

		Iterator<Process> it = processes.iterator();
		while (it.hasNext()) {
			Process process = it.next();

			reinitOrSuspended = false;

			try {
				log.fine("test process: " + process.getName());

				// reinit all ended process
				if (process.hasExecution()
						&& (process.getExecution().getState() == Execution.State.ENDED)) {
					this.log.fine("reinit and restart an older instance: " + process.getName());
					/* TODO no instance recylcing ; review races on process reuse  */
					//synchronized (process) {
						process.end(true);
						run(process);
					//}
					reinitOrSuspended = true;
				} else {
					reinitOrSuspended = true;
				}
			} catch (CoreException e) {
				// error to renit process
				process.setUnstable(true);
				processes.remove(process);
				it = processes.iterator();
				AutoTrashProcessService autoTrash = this.serviceManager
						.getService(AutoTrashProcessServiceImpl.class);
				autoTrash.addUninstableProcessInstance(process);
			}

			if (reinitOrSuspended) {
				// give the message to each suspended receiver behaviour
				// (either it accept or not)
				try {
					// this does not really feed with the message, this
					// is just to execute matchers to see
					// if the execution is capable of handling internal
					// message.
					final Execution exec = this.injectMessageInProcessInstance(process,
							internalMessage);

					if (exec != null) {
						System.err.println("---10");

						this.log.fine("available instance found => restart the process "
								+ process.getName());
						possibleExecution.add(exec);

						// CURRENT STRATEGY: stop when an execution is
						// found.
						break;
					}
				} catch (final CoreException e) {
					this.log.severe("Error injecting a message in process " + process.getName()
							+ ": " + e.getMessage());
					// do nothing
					e.printStackTrace();
				}
			}

		}

		if (possibleExecution.size() > 0) {
			System.err.println("---12");
			validExecution = possibleExecution.get(0);
		}
		return validExecution;

	}





	@SuppressWarnings("unchecked")
	private Execution acceptInternalMessage(final Message internalMessage) throws CoreException {

		Execution validExecution = findFirstAvailableExecution(internalMessage);

		if (validExecution == null) {
			this.log.fine("no available instance found");

			// no instance process found => create a new instance IF receiver is
			// "createInstance=YES"
			if (this.model != null) {

				final ProcessKey key = new ProcessKeyImpl();
				key.setEndpoint(internalMessage.getEndpoint());
				key.setService(internalMessage.getService());

				// find definition
				final ProcessDefinition definition = this.model.getRegistry().getProcessDefinition(
						key);
				if (definition == null) {
					throw new CoreException(
							"Impossible to find definition corresponding to this key: " + key);
				}

				// Check if create instance is set to yes or not
				if (isCreateInstance(definition, internalMessage)) {

					// compile the definition
					final Process process = this.model.getCompiler().compile(definition);

					if (process != null) {

						// run the instance process
						this.log.fine("run the process...");
						run(process);

						// inject the message in process instance
						this.log.fine("inject the message in process");
						final Execution exec = this.injectMessageInProcessInstance(process,
								internalMessage);

						if (exec != null) {
							validExecution = exec;
						}
					} else {
						this.log.fine("no process definition found concerned by the received message: "
								+ key);
					}
				} else {
					// do nothing, message will be stored
				}
			}
		}
		return validExecution;
	}

	private Execution injectMessageInProcessInstance(final Process process,
			final Message internalMessage) throws CoreException {
		Execution res = null;
		while(process.getExecution() == null) {
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				// do nothing
				log.warning("Wait execution is started");
			}
		}
		final Map<String, Execution> execs = process.getExecution().getSuspendedExecutions();

		this.log.fine("list of execution to try: " + execs);

		for (Execution exec : execs.values()) {
			try {
				this.log.fine("try to inject message in process " + process.getName()
						+ " using execution " + exec.getName());
				if (exec.getCurrentTarget() instanceof Node) {

					final Node node = (Node) exec.getCurrentTarget();
					if (node.getBehaviour() instanceof ReceiverBehaviour) {

						final ReceiverBehaviour receiverBehaviour = (ReceiverBehaviour) node
								.getBehaviour();
						if (receiverBehaviour.getMessage() == null) {

							receiverBehaviour.setMessage(internalMessage);
							this.log.fine("message setted in receive behaviour: "
									+ receiverBehaviour.getName());
							boolean accept = receiverBehaviour.accept(exec, internalMessage);
							if (accept) {
								this.log.fine("accepted by receiver: " + receiverBehaviour.getName());
								res = exec;
								break;
							} else {
								this.log.fine("not accepted by receiver: " + receiverBehaviour.getName());
								receiverBehaviour.setMessage(null);
							}
						} else {

							this.log.fine("the receiver " + receiverBehaviour.getName()
									+ " already contains a message");
						}
					} else {
						throw new CoreException(
								"Error: the node selected is not concerned by the received message "
										+ node.getBehaviour().getName() + "::"
										+ node.getClass().getCanonicalName());
					}
				} else {
					throw new CoreException(
							"Error: the execution selected is not concerned by the received message");
				}
			} catch (CoreException e) {
				this.log.fine("Error in injectMessageInProcessInstance: " + e.getMessage());
			}
		}

		return res;
	}





	public void flushMessagesInRegistry() throws CoreException {
		this.log.fine("flush stored messages");
		AutoFlushMessageServiceImpl autoflush = this.serviceManager
				.getService(AutoFlushMessageServiceImpl.class);
		autoflush.flushMessagesInRegistry();
	}

	public void sendTo(final Message message, String address, Map<Partner, Map<String,ExternalContext>> context, boolean isReply)
			throws CoreException {

		this.checkExternalEnvironment();
		if ((this.externalEnvironment.getSenders() != null)
				&& (this.externalEnvironment.getSenders().size() > 0)) {

			this.externalEnvironment.getSenders().get(0).sendTo(message, address, context, isReply);
		} else {
			throw new CoreException("Senders does not exist!!!");
		}
	}

	public Message sendSyncTo(final Message request, String address, Map<Partner, Map<String,ExternalContext>> context)
			throws CoreException {
		Message res = null;
		this.checkExternalEnvironment();
		if ((this.externalEnvironment.getSenders() != null)
				&& (this.externalEnvironment.getSenders().size() > 0)) {
			res = this.externalEnvironment.getSenders().get(0)
					.sendSyncTo(request, address, context);
		} else {
			throw new CoreException("Senders does not exist!!!");
		}
		return res;
	}

	public ProcessInstanceRegistry getProcessInstanceRegistry() {
		return this.processInstanceRegistry;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	public void deleteProcess(final Process process) throws CoreException {
		try {
			SCAExtendedContentController parentContentController = (SCAExtendedContentController) process
					.getExecution().getComponent().getFcInterface(SCAContentController.NAME);

			parentContentController.releaseFcContent(process.getComponent(), true);

		} catch (NoSuchInterfaceException e) {
			throw new CoreException("Delete process " + process.getName() + "failed. Caused by "
					+ e.getMessage());
		}
	}

	public Core getCore() throws CoreException {
		Core res = null;
		try {
			final Component coreComp = SCAHelper.getSCAHelper().getParent(this.getComponent());
			res = (Core) coreComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public ServiceManager getServiceManager() {
		return this.serviceManager;
	}

	public ClassLoader getClassLoader() {
		return Thread.currentThread().getContextClassLoader();
	}

	private boolean isCreateInstance(ProcessDefinition def, Message mess) throws CoreException {
		return this.model.getRegistry().isCreateInstance(def, mess);

	}

	public PoolOfProcessForkers getPoolOfProcessForkers() {
		return this.poolOfProcessForkers;
	}

	public Partner findEndpointFromMessage(Message mess, Process process) {

		Collection<Partner> partners = process.getPartners().values();
		Iterator<Partner> itEndpoints = partners.iterator();
		Partner endpoint = null;

		while (itEndpoints.hasNext()) {
			Partner current = itEndpoints.next();
			Element key = current.getValue(process.getExecution());
			if(key != null) {
				String endpointName = process.getPartnerEvaluator().getEndpointName(key);
				QName serviceName = process.getPartnerEvaluator().getServiceName(key);

				if (endpointName != null && serviceName != null) {
					if (endpointName.equalsIgnoreCase(mess.getEndpoint())
							&& serviceName.equals(mess.getService())) {
						endpoint = current;
						break;
					}
				}
			}
		}

		return endpoint;
	}

	public Process findProcessFromName(String processInstanceName) {
		Process p = null;

		ProcessKey key = new ProcessKeyImpl();

		QName processDefinitionName = extractProcessDefinitionName(processInstanceName);

		try {
			key = this.getCore().getModel().getRegistry().findProcessKey(processDefinitionName);

			List<Process> processInstances = this.getProcessInstanceRegistry().getProcessInstances(
					key);

			for (Process process : processInstances) {

				if (process.getName().equals(processInstanceName)) {
					p = process;
					break;
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return p;
	}

	private static QName extractProcessDefinitionName(String processInstanceName) {
		int indexOf_ = processInstanceName.lastIndexOf("_");
		String valueOfQName = processInstanceName.substring(0, indexOf_);
		return QName.valueOf(valueOfQName);
	}

	@Override
	public void onTimerFinish(Execution execution, TimerFinishedEvent e) throws CoreException {
		e.getWaitBehaviour().terminateWaiting(execution);
		restartProcess(e.getExecution().getCurrentScope().getProcess(), null);
	}

	private Execution doRun(Process process, boolean isStepByStep) throws CoreException {
		Component executionComp;
		try {
			executionComp = SCAHelper.getSCAHelper().createNewComponent(
					ExecutionImpl.class.getName(), null);
			Execution execution = (Execution) executionComp.getFcInterface("service");
			SCAHelper.getSCAHelper().startComponent(executionComp);
			String executionName = "mainExec_"+uuidGenerator.getNewID();
			process.setProcessExecutionName(executionName);
			SCAHelper.getSCAHelper().changeName(executionComp, executionName);
			execution.setStepByStep(isStepByStep);
			SCAHelper.getSCAHelper().addComponent(executionComp, getComponent(), null);
			SCAHelper.getSCAHelper().startComponent(executionComp);

			// Add binding between execution and process
			execution.setInitialTarget(process);
			process.setExecution(execution);

			restartProcess(process, null);

			return execution;
		} catch (SCAException e) {
			throw new CoreException(e);
		} catch (NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public Execution run(Process process) throws CoreException {
		Execution execution = doRun(process, false);
		if (getConfiguration().getSynchronousRun()) {
			while ((execution.getState() != State.ENDED)
					&& (execution.getState() != State.CANCELLED)) {
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
				}
			}
		}
		return execution;
	}

	public Execution runStepByStep(Process process) throws CoreException {
		return doRun(process, true);
	}

	@Override
	public ConfigurationEngine getConfiguration() {
		return this.configuration;
	}

	@Override
	public void setConfiguration(ConfigurationEngine configuration) {
		this.configuration = configuration;
	}

}
