/****************************************************************************
 *
 * Copyright (c) 2009-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/
 
package com.ebmwebsourcing.easyviper.core.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Scope;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.ow2.frascati.tinfi.api.control.ContentInstantiationException;
import org.ow2.frascati.tinfi.api.control.SCAContentController;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.Binding;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.Core;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.configuration.ConfigurationEngine;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.Service;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalEnvironment;
import com.ebmwebsourcing.easyviper.core.api.model.Model;
import com.ebmwebsourcing.easyviper.core.api.services.AutoFlushMessageService;
import com.ebmwebsourcing.easyviper.core.impl.engine.EngineImpl;
import com.ebmwebsourcing.easyviper.core.impl.env.ExternalEnvironmentImpl;
import com.ebmwebsourcing.easyviper.core.impl.services.AutoFlushMessageServiceImpl;
import com.ebmwebsourcing.easyviper.extended.service.autotrash.api.AutoTrashProcessService;
import com.ebmwebsourcing.easyviper.extended.service.autotrash.impl.AutoTrashProcessServiceImpl;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
@Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value = Core.class, names = "service")
@PolicySets("frascati:scaEasyCompositeWithContent")
public class CoreImpl extends SCAComponentImpl implements Core {

    public static final String DEFAULT_ENGINE_NAME = "WorkflowEngine";

    private Logger log = Logger.getLogger(CoreImpl.class.getSimpleName());

    private Component engineComponent;

    private Component modelComponent;

    private Component externalEnvironmentComponent;

    /**
     * Default Constructor
     * 
     * @throws CoreException
     */
    public CoreImpl() throws CoreException {
        super();
    }

    public Engine createEngine(final ConfigurationEngine conf) throws CoreException {

        if (this.engineComponent != null) {
            throw new CoreException("Error: Workflow engine already exist!!!");
        }

        Engine engine = null;
        try {
            this.engineComponent = SCAHelper.getSCAHelper().createNewComponent(
                    EngineImpl.class.getName(), null);
            SCAHelper.getSCAHelper().addComponent(this.engineComponent, getComponent(), null);

            // engine = (Engine)
            // this.engineComponent.getFcInterface("/content");
            SCAContentController scacc = (SCAContentController) this.engineComponent
                    .getFcInterface(SCAContentController.NAME);
            try {
                engine = (Engine) scacc.getFcContent();
            } catch (ContentInstantiationException e) {
                throw new CoreException(e);
            }

            if (!this.log.getName().equals(CoreImpl.class.getName())) {
                engine.setLog(this.log);
            }

            SCAHelper.getSCAHelper().startComponent(this.engineComponent);
            SCAHelper.getSCAHelper().changeName(this.engineComponent, CoreImpl.DEFAULT_ENGINE_NAME);

            engine = (Engine) this.engineComponent.getFcInterface("service");
            if (conf.getAutoFlushMessageFrequency() != null
                    && conf.getAutoFlushMessageFrequency() > 0) {
                AutoFlushMessageService autoFlush = engine.getServiceManager().getService(
                        AutoFlushMessageServiceImpl.class);
                autoFlush.setRefreshFrequency(conf.getAutoFlushMessageFrequency());

                engine.getServiceManager().activateService(AutoFlushMessageServiceImpl.class);
            }
            if (conf.getAutoTrashProcessFrequency() != null
                    && conf.getAutoTrashProcessFrequency() > 0) {
                AutoTrashProcessService autoTrash = engine.getServiceManager().getService(
                        AutoTrashProcessServiceImpl.class);
                autoTrash.setRefreshFrequency(conf.getAutoTrashProcessFrequency());
                engine.getServiceManager().activateService(AutoTrashProcessServiceImpl.class);
            }
            if (conf.getAdditionnalServices() != null) {
                for (Class<? extends Service> s : conf.getAdditionnalServices()) {
                    engine.getServiceManager().addService(s);
                }
            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        engine.setConfiguration(conf);
        this.log.finest("workflow engine created and started");
        return engine;
    }

    public Engine getEngine() throws CoreException {
        Engine res = null;
        try {
            if (this.engineComponent != null) {
                res = (Engine) this.engineComponent.getFcInterface("service");
            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public Model createModel(final String name, final Class<? extends Model> modelClass)
            throws CoreException {

        if (this.modelComponent != null) {
            throw new CoreException("Error: Model already exist!!!");
        }

        Model model = null;
        try {
            this.modelComponent = SCAHelper.getSCAHelper().createNewComponent(modelClass.getName(),
                    null);
            SCAHelper.getSCAHelper().addComponent(this.modelComponent, getComponent(), null);

            model = (Model) (((SCAContentController) this.modelComponent
                    .getFcInterface("/sca-content-controller")).getFcContent());

            if (!this.log.getName().equals(CoreImpl.class.getName())) {
                model.setLog(this.log);
            }

            SCAHelper.getSCAHelper().startComponent(this.modelComponent);
            SCAHelper.getSCAHelper().changeName(this.modelComponent, name);

            model = (Model) this.modelComponent.getFcInterface("service");

            // link model to engine
            if (model != null) {
                final List<Binding> listOfBindings = new ArrayList<Binding>();
                listOfBindings
                        .add(new Binding("model", (org.objectweb.fractal.api.Interface) model));
                SCAHelper.getSCAHelper().addBindings(this.engineComponent, listOfBindings);

            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        } catch (ContentInstantiationException e) {
            throw new CoreException(e);
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        this.log.finest("model created and started");

        return model;
    }

    public Model getModel() throws CoreException {
        Model res = null;
        try {
            if (this.modelComponent != null) {
                final Object obj = this.modelComponent.getFcInterface("service");
                res = (Model) this.modelComponent.getFcInterface("service");
            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public ExternalEnvironment createExternalEnvironment(final String name) throws CoreException {
        ExternalEnvironment externalEnvironment = null;

        if (this.getEngine() == null) {
            throw new CoreException(
                    "You must create the engine before create an external environment!!!");
        }

        Component fractalExternalEnvironment = null;
        try {
            fractalExternalEnvironment = SCAHelper.getSCAHelper().createNewComponent(
                    ExternalEnvironmentImpl.class.getName(), null);
            SCAHelper.getSCAHelper().addComponent(fractalExternalEnvironment, getComponent(), null);
            this.externalEnvironmentComponent = fractalExternalEnvironment;

            try {
                // link externalEnvironment to engine
                if (this.getEngine() != null) {
                    final List<Binding> listOfBindings = new ArrayList<Binding>();
                    listOfBindings.add(new Binding("engine",
                            (org.objectweb.fractal.api.Interface) this.getEngine()));
                    SCAHelper.getSCAHelper().addBindings(this.externalEnvironmentComponent,
                            listOfBindings);

                }

                // externalEnvironment = (ExternalEnvironment)
                // fractalExternalEnvironment
                // .getFcInterface("/content");
                SCAContentController scacc = (SCAContentController) fractalExternalEnvironment
                        .getFcInterface(SCAContentController.NAME);
                try {
                    externalEnvironment = (ExternalEnvironment) scacc.getFcContent();
                } catch (ContentInstantiationException e) {
                    throw new CoreException(e);
                }
                // externalEnvironment.init(fractalExternalEnvironment);
                if (!this.log.getName().equals(CoreImpl.class.getName())) {
                    externalEnvironment.setLog(this.log);
                }

                SCAHelper.getSCAHelper().startComponent(fractalExternalEnvironment);
                SCAHelper.getSCAHelper().changeName(fractalExternalEnvironment, name);

                externalEnvironment = (ExternalEnvironment) fractalExternalEnvironment
                        .getFcInterface("service");

                // link engine to externalEnvironment
                final List<Binding> listOfBindings = new ArrayList<Binding>();
                listOfBindings.add(new Binding("externalEnv",
                        (org.objectweb.fractal.api.Interface) externalEnvironment));
                SCAHelper.getSCAHelper().addBindings(this.engineComponent, listOfBindings);

            } catch (final NoSuchInterfaceException e) {
                throw new CoreException(e);
            }
            this.log.finest("external Environment created and started");
        } catch (final SCAException e) {
            throw new CoreException(e);
        }
        return externalEnvironment;
    }

    public ExternalEnvironment getExternalEnvironment() throws CoreException {
        ExternalEnvironment res = null;
        try {
            if (this.externalEnvironmentComponent != null) {
                final Object obj = this.externalEnvironmentComponent.getFcInterface("service");
                res = (ExternalEnvironment) this.externalEnvironmentComponent
                        .getFcInterface("service");
            }
        } catch (final NoSuchInterfaceException e) {
            throw new CoreException(e);
        }
        return res;
    }

    public void setLog(final Logger logger) {
        this.log = logger;
    }

}
