/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easysawsdl10.api;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.with.WithAnyXmlObjects;
import com.ebmwebsourcing.easybox.api.with.WithOtherAttributes;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easysawsdl10.api.anonymoustype.AttrExtensions;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.type.Type;

public final class SawsdlHelper {
	
	private final static QName MODEL_REF_QNAME = new QName(Constants.SAWSDL10_NS_URI, "modelReference");
	private final static QName LIFT_REF_QNAME = new QName(Constants.SAWSDL10_NS_URI, "liftingSchemaMapping");
	private final static QName LOWER_REF_QNAME = new QName(Constants.SAWSDL10_NS_URI, "loweringSchemaMapping");
	
	private SawsdlHelper(){
	}

	// ModelReference
	public static void addModelReference(XmlObject contextObject, URI... uris){
		if(contextObject instanceof WithOtherAttributes){
			if(((WithOtherAttributes) contextObject).hasOtherAttribute(MODEL_REF_QNAME)){
				String currentModelRef = ((WithOtherAttributes) contextObject).getOtherAttribute(MODEL_REF_QNAME);
				((WithOtherAttributes) contextObject).addOtherAttribute(MODEL_REF_QNAME, currentModelRef + " " + urisToString(uris));
			} else {
				((WithOtherAttributes) contextObject).addOtherAttribute(MODEL_REF_QNAME, urisToString(uris));
			}
		} else if(contextObject instanceof WithAnyXmlObjects) {
			if(((WithAnyXmlObjects) contextObject).hasAnyXmlObject(AttrExtensions.QNAME)){
				AttrExtensions attr = ((WithAnyXmlObjects) contextObject).getFirstAnyXmlObject(AttrExtensions.class);
				String currentModelRef = attr.getOtherAttribute(MODEL_REF_QNAME);
				attr.addOtherAttribute(MODEL_REF_QNAME, currentModelRef + " " + urisToString(uris));
			} else {
				AttrExtensions attr = contextObject.getXmlContext().getXmlObjectFactory().create(AttrExtensions.class);
				attr.addOtherAttribute(MODEL_REF_QNAME, urisToString(uris));
				((WithAnyXmlObjects) contextObject).addAnyXmlObject(attr);
			}
		} else {
			// Useless for WSDL 1.1 or 2.0 but could be useful in case of extensions to policies
			throw new UncheckedException("Impossible to add semantic annontation on " + contextObject.getXmlObjectQName());
		}
	}
	
	public static URI[] getModelReference(XmlObject contextObject){
		String modelRef = null;
		if(contextObject instanceof WithOtherAttributes){
			modelRef = ((WithOtherAttributes) contextObject).getOtherAttribute(MODEL_REF_QNAME);
		} else if(contextObject instanceof WithAnyXmlObjects){
			AttrExtensions attr= ((WithAnyXmlObjects) contextObject).getFirstAnyXmlObject(AttrExtensions.class);
			if(attr != null){
				modelRef = attr.getOtherAttribute(MODEL_REF_QNAME);
			}
		}
		if(modelRef != null) {
			return stringToUris(modelRef);
		}
		return new URI[0];
	}
	
	public static void removeModelReference(XmlObject contextObject){
		if(contextObject instanceof WithOtherAttributes){
			((WithOtherAttributes) contextObject).removeOtherAttribute(MODEL_REF_QNAME);
		} else if(contextObject instanceof WithAnyXmlObjects){
			AttrExtensions attr= ((WithAnyXmlObjects) contextObject).getFirstAnyXmlObject(AttrExtensions.class);
			if(attr != null){
				((WithAnyXmlObjects) contextObject).removeAnyXmlObject(attr);
			}
		}
	}
	
	
	// Lifting Schema Mapping
	public static void addLiftingSchemaMapping(Element contextObject, URI... uris){
		addSchemaMapping(contextObject, LIFT_REF_QNAME, uris);
	}
	
	public static void addLiftingSchemaMapping(Type contextObject, URI... uris){
		addSchemaMapping(contextObject, LIFT_REF_QNAME, uris);
	}
	
	public static URI[] getLiftingSchemaMapping(Element contextObject){
		return getSchemaMapping(contextObject, LIFT_REF_QNAME);
	}
	
	public static URI[] getLiftingSchemaMapping(Type contextObject){
		return getSchemaMapping(contextObject, LIFT_REF_QNAME);
	}	
	
	public static void removeLiftingSchemaMapping(Element contextObject){
		removeSchemaMapping(contextObject, LIFT_REF_QNAME);
	}
	
	public static void removeLiftingSchemaMapping(Type contextObject){
		removeSchemaMapping(contextObject, LIFT_REF_QNAME);
	}

	
	// Lowering Schema Mapping
	public static void addLoweringSchemaMapping(Element contextObject, URI... uris){
		addSchemaMapping(contextObject, LOWER_REF_QNAME, uris);
	}
	
	public static void addLoweringSchemaMapping(Type contextObject, URI... uris){
		addSchemaMapping(contextObject, LOWER_REF_QNAME, uris);
	}
	
	public static URI[] getLoweringSchemaMapping(Element contextObject){
		return getSchemaMapping(contextObject, LOWER_REF_QNAME);
	}

	public static URI[] getLoweringSchemaMapping(Type contextObject){
		return getSchemaMapping(contextObject, LOWER_REF_QNAME);
	}
	
	public static void removeLoweringSchemaMapping(Element contextObject){
		removeSchemaMapping(contextObject, LOWER_REF_QNAME);
	}

	public static void removeLoweringSchemaMapping(Type contextObject){
		removeSchemaMapping(contextObject, LOWER_REF_QNAME);
	}

	
	//  Schema Mapping
	private static void addSchemaMapping(XmlObject contextObject, QName mappingTypeQName, URI... uris){
		if(contextObject instanceof WithOtherAttributes){
			if(((WithOtherAttributes) contextObject).hasOtherAttribute(mappingTypeQName)){
				String currentModelRef = ((WithOtherAttributes) contextObject).getOtherAttribute(mappingTypeQName);
				((WithOtherAttributes) contextObject).addOtherAttribute(mappingTypeQName, currentModelRef + " " + urisToString(uris));
			} else {
				((WithOtherAttributes) contextObject).addOtherAttribute(mappingTypeQName, urisToString(uris));
			}
		}
	}
	
	private static URI[] getSchemaMapping(XmlObject contextObject, QName mappingTypeQName){
		String schemaMapping = null;
		if(contextObject instanceof WithOtherAttributes){
			schemaMapping = ((WithOtherAttributes) contextObject).getOtherAttribute(mappingTypeQName);
		}
		if(schemaMapping != null) {
			return stringToUris(schemaMapping);
		}
		return new URI[0];
		
	}
	
	private static void removeSchemaMapping(XmlObject contextObject, QName mappingTypeQName){
		if(contextObject instanceof WithOtherAttributes){
			((WithOtherAttributes) contextObject).removeOtherAttribute(mappingTypeQName);
		}
	}
	
	
	// Private methods
	private static final URI[] stringToUris(String str){
		List<URI> uris = new ArrayList<URI>();
		for(String uri : str.split("\\s+")){
			try {
				uris.add(new URI(uri));
			} catch (URISyntaxException e) {
				throw new RuntimeException("Problem in semantic annontation URIs: ",e);
			}
		}
		return uris.toArray(new URI[uris.size()]);
	}
	
	private static final String urisToString(URI[] uris){
		StringBuilder urisStr = new StringBuilder();
		for(URI uri : uris){
			if(urisStr.length() != 0){
				urisStr.append(" ");
			}
			urisStr.append(uri.toString());
		}
		return urisStr.toString();
	}
	
}
