/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */
package com.petalslink.easiersbs.matching.process.util;

import java.net.URI;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.ebmwebsourcing.easybpmn.bpmn20.api.DefinitionsHelper;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowNode;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Process;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Task;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithFlowElements;
import com.ebmwebsourcing.easybpmn.sabpmn20.api.SabpmnHelper;
import com.ebmwebsourcing.easybpmn.sabpmn20.api.element.SemanticDetails;
import com.ebmwebsourcing.easybpmn.sabpmn20.api.element.SemanticElement;
import com.ebmwebsourcing.easybpmn.sabpmn20.api.element.SemanticElements;
import com.petalslink.easiersbs.matching.service.ProfileFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.ProfileFactory;
import com.petalslink.easiersbs.matching.service.api.profile.SearchElement;
import com.petalslink.easiersbs.matching.service.api.profile.SearchPart;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.profile.SearchElementImpl;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class BpmnUtil {

	private BpmnUtil() {

	}

	/**
	 * List all BPMN Tasks in a target BPMN
	 * 
	 * @param bpmn
	 * @return task list
	 */
	public static Set<Task> extractTasks(Definitions bpmn) {
		Set<Task> res = new HashSet<Task>();

		for (Process process : bpmn.getProcesses()) {
			res.addAll(process.getFlowElementsByClass(Task.class));
		}

		return res;
	}

	/**
	 * Extract search profile from BPMN activity in order to compute hybrid
	 * service matchmaking
	 * 
	 * @param Task
	 *            EasyBPMN task
	 * @return SearchProfile containing all semantic concepts embedded in BPMN
	 *         activities and its I/O
	 */
	public static SearchProfile extractSearchProfile(Task task) {
		ProfileFactory factory = ProfileFactoryImpl.getInstance();

		SearchProfile profile = factory.newSearchProfile();

		// Semantic interface
		SearchPart semanticInterface = factory.newSemanticPart();
		profile.setSemanticInterface(semanticInterface);

		// Semantic operation
		List<URI> operationConcepts = Arrays.asList(SabpmnHelper.getModelReference(task));
		SearchPart semanticOperation = factory.newSemanticPart(task.getName(), new HashSet<URI>(operationConcepts));
		SemanticDetails details = SabpmnHelper.getSemanticDetails(task);
		if (details != null && details.getFunctionalDescription() != null) {
			for (URI concept : details.getFunctionalDescription()
					.getModelReference()) {
				semanticOperation.addSemanticConcept(concept);
			}
		}
		profile.setSemanticOperation(semanticOperation);

		// Semantic input
		SearchElement input = extractMainElement(findIncomingSequenceFlow(task));
		profile.setInputSemanticElement(input);

		// Semantic output
		SearchElement output = extractMainElement(findOutgoingSequenceFlow(task));
		profile.setOutputSemanticElement(output);

		return profile;
	}
	
	/**
	 * Extract semantic element from Sequence flows 
	 * (and create a parent if several elements are found)
	 * 
	 * @param flows Set of incoming or outgoing flows
	 * @return SearchElement corresponding to root expected element
	 */
	private static SearchElement extractMainElement(Set<SequenceFlow> flows){
		Set<SearchElement> elems = new HashSet<SearchElement>();
		for(SequenceFlow flow : flows){
			elems.addAll(extractSemanticElements(flow));
		}
		if(elems.isEmpty()){
			return null;
		} else if(elems.size() == 1){
			return elems.iterator().next();
		} else {
			SearchElement res = new SearchElementImpl("", new HashSet<URI>(), true);
			for(SearchElement elem : elems){
				res.addChildElement(elem);
			}
			return res;
		}
	}

	private static Set<SearchElement> extractSemanticElements(SequenceFlow flow) {
		Set<SearchElement> res = new HashSet<SearchElement>();
		ProfileFactory factory = ProfileFactoryImpl.getInstance();

		List<URI> commonConcepts = Arrays.asList(SabpmnHelper
				.getModelReference(flow));
		SemanticElements semElements = SabpmnHelper.getSemanticElements(flow);

		if(semElements != null){
			for (SemanticElement semElement : semElements.getSemanticElements()) {
				SearchElement elem = factory.newSemanticElement();
				elem.setName(semElement.getName());
				elem.setRequired(true);
				for (URI concept : commonConcepts) {
					elem.addSemanticConcept(concept);
				}
				for (URI concept : semElement.getModelReference()) {
					elem.addSemanticConcept(concept);
				}
				res.add(elem);
			}
		}

		return res;
	}

	private static Set<SequenceFlow> findIncomingSequenceFlow(Task task) {
		Set<SequenceFlow> res = new HashSet<SequenceFlow>();

		WithFlowElements process = DefinitionsHelper.getParentContainer(task);

		for (FlowNode node : task.getIncomingFlowNodes()) {
			res.add(getSequenceFlow(process, node.getId(),
					task.getId()));
		}

		return res;
	}

	private static Set<SequenceFlow> findOutgoingSequenceFlow(Task task) {
		Set<SequenceFlow> res = new HashSet<SequenceFlow>();

		WithFlowElements process = DefinitionsHelper.getParentContainer(task);

		for (FlowNode node : task.getOutgoingFlowNodes()) {
			res.add(getSequenceFlow(process, task.getId(),
					node.getId()));
		}

		return res;
	}
	
	// TODO : DefinitionHelper method, delete for 1.0-SNAPSHOT use
    public static SequenceFlow getSequenceFlow(WithFlowElements proc, String sourceId, String targetId) {
        for (SequenceFlow sf : proc.getFlowElementsByClass(SequenceFlow.class)) {
            if (sf.getSourceRef().getId().equals(sourceId)
                    && sf.getTargetRef().getId().equals(targetId)) {
                return sf;
            }
        }
        return null;
    }



}
