/**
 * domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.validation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.collaboration.ILaneSetBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.common.IFlowElementBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.common.IFlowNodeBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.common.IMessageFlowBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.event.definition.IEventDefinitionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.DefinitionsBeanVisitor;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.ReceiveTaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.SendTaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.CatchEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.ThrowEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.definition.MessageEventDefinitionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.ExclusiveGatewayBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.ParallelGatewayBean;

public class BPMNValidator{

	private static BPMNValidator instance;
	
	private BPMNValidator(){}
	
	public static BPMNValidator getInstance(){
		if(instance==null){
			instance = new BPMNValidator();
		}
		return instance;
	}
	
	public void validate(DefinitionsBean defs) throws BPMNValidationException{
		BPMNValidationVisitor visitor = new BPMNValidationVisitor(defs);
		try {
			visitor.visitDefinitionsByPools();
		} catch (Exception e) {
			e.printStackTrace();
			throw new BPMNValidationException(null);
		}
		if(visitor.validationErrors.size()>0){
			throw new BPMNValidationException(visitor.validationErrors);
		}
	}
	
	
	
	private class BPMNValidationVisitor extends DefinitionsBeanVisitor{
		
		private ICollaborationBean currentCollab;
		private IProcessBean currentProcess;
		Map<String, String> validationErrors;
		
		public BPMNValidationVisitor(DefinitionsBean defs){
			super(defs);
			validationErrors = new HashMap<String, String>();
			currentProcess = null;
		}

		/*
		 * OK definitions should have at least one pool
		 * OK end event : one or more incoming sequence flow, 0 outgoing sequence flow
		 * OK start event : one or more outgoing sequence flow, 0 incoming sequence flow
		 * OK gateway has incoming and outgoing sf
		 * OK gateway either diverges or converges
		 * OK lane : not empty
		 * OK pool : not empty 
		 * OK a pool having an end event must have at least one start event
		 * OK sequence flow => source and target in the same pool
		 * OK sendTask and receiveTask => mf or sf to message event
		 * OK message flow source and target are not in the same pool
		 * OK only one outgoing or incoming mf by flow node
		 * OK sf going out of exclusive gateway must be conditional 
		 * OK sf going out of parallel gateway must not be conditional
		 * 
		 * KO conditional flow : ensure condition is set 
		 * 
		 */
		
		@Override
		public void visitCollaboration(ICollaborationBean collab){
			if(collab.getParticipants()==null || collab.getParticipants().isEmpty()){
				validationErrors.put(collab.getId(), BPMNValidationErrorMessage.NO_POOL.getMessage());
			}
			currentCollab = collab;
		}
		
		@Override
		public void visitParticipant(IParticipantBean pool){
			if(pool.getProcess()==null){
				validationErrors.put(pool.getId(), BPMNValidationErrorMessage.EMPTY_POOL.getMessage());
			}
		}
		
		@Override
		public void visitParticipantProcess(IProcessBean proc, IParticipantBean p){
			if(proc.getLaneSets()==null || proc.getLaneSets().isEmpty()){
				validationErrors.put(p.getId(), BPMNValidationErrorMessage.EMPTY_POOL.getMessage());
			}
			else{
				boolean found = false;
				for(ILaneSetBean lsb : proc.getLaneSets()){
					if(lsb.getLanes()!=null && !lsb.getLanes().isEmpty()){
						found = true;
						break;
					}
				}
				if(!found){
					validationErrors.put(p.getId(), BPMNValidationErrorMessage.EMPTY_POOL.getMessage());
				}
				else{
					if(proc.getEndEvents()!=null && !proc.getEndEvents().isEmpty()){
						if(proc.getStartEvents()==null || proc.getStartEvents().isEmpty()){
							validationErrors.put(p.getId(), BPMNValidationErrorMessage.NO_START.getMessage());
						}
					}
				}
			}
		}
		
		@Override
		public void visitProcess(IProcessBean p){
			currentProcess = p;
		}
		
		@Override
		public void visitLane(ILaneBean lane){
			if((lane.getStartEvents()!=null && !lane.getStartEvents().isEmpty())
				||	(lane.getTasks()!=null && !lane.getTasks().isEmpty())
				||	(lane.getGateways()!=null && !lane.getGateways().isEmpty())
				||	(lane.getEndEvents()!=null && !lane.getEndEvents().isEmpty())){
				return;
			}
			else{
				validationErrors.put(lane.getId(), BPMNValidationErrorMessage.EMPTY_LANE.getMessage());
			}
		}
		
		@Override
		public void visitMessageFlow(IMessageFlowBean mfb){
			if(mfb.getSource()==null){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MF_NO_SOURCE.getMessage());
				return;
			}
			if(mfb.getTarget()==null){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MF_NO_TARGET.getMessage());
				return;
			}
			if(mfb.getSource().equals(mfb.getTarget())){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MF_SAME_SOURCE_TARGET.getMessage());
				return;
			}
			
			if(!canBeMFSource(mfb.getSource())){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MF_WRONG_SOURCE.getMessage());
			}
			if(!canBeMFTarget(mfb.getTarget())){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MF_WRONG_TARGET.getMessage());
			}
			
			IParticipantBean p1 = findEnclosingPool(mfb.getSource());
			IParticipantBean p2 = findEnclosingPool(mfb.getTarget());
			if(p1.equals(p2)){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MF_SAME_POOL.getMessage());
			}
			
			List<IMessageFlowBean> sameSource = findMFBySource(mfb.getSource());
			if(sameSource.isEmpty() || sameSource.size()>1){
				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MULTIPLE_OUGOING_MF.getMessage());
			}
			
//			List<MessageFlowBean> sameTarget = findMFByTarget(mfb.getTarget());
//			if(sameTarget.isEmpty() || sameTarget.size()>1){
//				validationErrors.put(mfb.getId(), BPMNValidationErrorMessage.MULTIPLE_INCOMING_MF.getMessage());
//			}
		}
		
		@Override
		public void visitEndEvent(IEndEventBean eeb){
			if(!findSFBySource(eeb).isEmpty()){
				validationErrors.put(eeb.getId(), BPMNValidationErrorMessage.END_OUTGOING.getMessage());
			}
			if(findSFByTarget(eeb).isEmpty()){
				validationErrors.put(eeb.getId(), BPMNValidationErrorMessage.END_INCOMING.getMessage());
			}
		}
		
		@Override
		public void visitStartEvent(IStartEventBean seb){
			if(findSFBySource(seb).isEmpty()){
				validationErrors.put(seb.getId(), BPMNValidationErrorMessage.START_NO_OUTGOING.getMessage());
			}
			if(!findSFByTarget(seb).isEmpty()){
				validationErrors.put(seb.getId(), BPMNValidationErrorMessage.START_NO_INCOMING.getMessage());
			}
		}
		
		@Override
		public void visitGateway(IGatewayBean gb){
			List<ISequenceFlowBean> incoming = findSFByTarget(gb);
			List<ISequenceFlowBean> outgoing = findSFBySource(gb);
			if(outgoing.isEmpty()){
				validationErrors.put(gb.getId(), BPMNValidationErrorMessage.GATEWAY_NO_OUTGOING.getMessage());
				return;
			}
			if(incoming.isEmpty()){
				validationErrors.put(gb.getId(), BPMNValidationErrorMessage.GATEWAY_NO_INCOMING.getMessage());
				return;
			}
			int cptIn = incoming.size();
			int cptOut = outgoing.size();
			
			if(cptOut==1 && cptIn==1){
				validationErrors.put(gb.getId(), BPMNValidationErrorMessage.FORWARDING_GATE.getMessage());
			}
			else if(cptIn>1 && cptOut>1){
				validationErrors.put(gb.getId(), BPMNValidationErrorMessage.MERGING_DIVERGING_GATE.getMessage());
			}
			
			if(gb instanceof ParallelGatewayBean){
				for(ISequenceFlowBean outer : outgoing){
					if(outer.getExpression()!=null){
						validationErrors.put(outer.getId(), BPMNValidationErrorMessage.SF_PARRALLEL_GATE_CONDITION.getMessage());
					}
				}
			}
		}
		
		@Override
		public void visitTask(ITaskBean t){
			boolean found = false;
			if(t instanceof ReceiveTaskBean){
				for(ISequenceFlowBean sf : findSFByTarget(t)){
					if(sf.getSourceNode() instanceof CatchEventBean){
						CatchEventBean ce = (CatchEventBean) sf.getSourceNode();
						if(ce.getTriggers()!=null){
							for(IEventDefinitionBean ed : ce.getTriggers()){
								if(ed instanceof MessageEventDefinitionBean){
									found = true;
									break;
								}
							}
						}
					}
				}
				
				if(!found && !findMFByTarget(t).isEmpty()){
					found = true;
				}
				
				if(!found){
					validationErrors.put(t.getId(), BPMNValidationErrorMessage.NO_MSG_RECEIVE_TASK.getMessage());
				}
			}
			
			else if(t instanceof SendTaskBean){
				for(ISequenceFlowBean sf : findSFBySource(t)){
					if(sf.getTargetNode() instanceof ThrowEventBean){
						ThrowEventBean te = (ThrowEventBean) sf.getTargetNode();
						if(te.getResults()!=null){
							for(IEventDefinitionBean ed : te.getResults()){
								if(ed instanceof MessageEventDefinitionBean){
									found = true;
									break;
								}
							}
						}
					}
				}
				
				if(!found && !findMFBySource(t).isEmpty()){
					found = true;
				}
				
				if(!found){
					validationErrors.put(t.getId(), BPMNValidationErrorMessage.NO_MSG_SEND_TASK.getMessage());
				}
			}
		}
		
		@Override
		public void visitSequenceFlow(ISequenceFlowBean sf){
			if(sf.getSourceNode()==null){
				validationErrors.put(sf.getId(), BPMNValidationErrorMessage.SF_NO_SOURCE.getMessage());
				return;
			}
			if(sf.getTargetNode()==null){
				validationErrors.put(sf.getId(), BPMNValidationErrorMessage.SF_NO_TARGET.getMessage());
				return;
			}
			if(sf.getSourceNode().equals(sf.getTargetNode())){
				validationErrors.put(sf.getId(), BPMNValidationErrorMessage.SF_SAME_SOURCE_TARGET.getMessage());
				return;
			}
			
			List<? extends IFlowElementBean> l = currentProcess.getFlowNodes();
			if(!l.contains(sf.getSourceNode()) || !l.contains(sf.getTargetNode())){
				validationErrors.put(sf.getId(), BPMNValidationErrorMessage.SF_SOURCE_TARGET_WRONG_POOL.getMessage());
			}
			
			if(sf.getSourceNode() instanceof ExclusiveGatewayBean){
				if(sf.getExpression()==null || sf.getExpression().getContent().isEmpty()){
					validationErrors.put(sf.getId(), BPMNValidationErrorMessage.SF_COND_MISSING.getMessage());
				}
			}
		}
		
		
		
		private List<ISequenceFlowBean> findSFBySource(IFlowElementBean source){
			List<ISequenceFlowBean> result = new ArrayList<ISequenceFlowBean>();
			for(ISequenceFlowBean sf : currentProcess.getSequenceFlows()){
				if(sf.getSourceNode()==null){
					continue;
				}
				if(sf.getSourceNode().getId().equals(source.getId())){
					result.add(sf);
				}
			}
			return result;
		}
		
		private List<ISequenceFlowBean> findSFByTarget(IFlowElementBean target){
			List<ISequenceFlowBean> result = new ArrayList<ISequenceFlowBean>();
			for(ISequenceFlowBean sf : currentProcess.getSequenceFlows()){
				if(sf.getTargetNode()==null){
					continue;
				}
				if(sf.getTargetNode().getId().equals(target.getId())){
					result.add(sf);
				}
			}
			return result;
		}
		
		private List<IMessageFlowBean> findMFBySource(IFlowElementBean source){
			List<IMessageFlowBean> result = new ArrayList<IMessageFlowBean>();
			if(currentCollab.getMessageFlows()!=null){
				for(IMessageFlowBean mf : currentCollab.getMessageFlows()){
					if(mf.getSource()==null){
						continue;
					}
					if(mf.getSource().getId().equals(source.getId())){
						result.add(mf);
					}
				}
			}
			return result;
		}
		
		private List<IMessageFlowBean> findMFByTarget(IFlowElementBean target){
			List<IMessageFlowBean> result = new ArrayList<IMessageFlowBean>();
			if(currentCollab.getMessageFlows()!=null){
				for(IMessageFlowBean mf : currentCollab.getMessageFlows()){
					if(mf.getTarget()==null){
						continue;
					}
					if(mf.getTarget().getId().equals(target.getId())){
						result.add(mf);
					}
				}
			}
			return result;
		}
		
		private boolean canBeMFSource(IFlowNodeBean node){
			if(node instanceof ThrowEventBean){
				ThrowEventBean teb = (ThrowEventBean) node;
				if(teb.getResults()!=null){
					for(IEventDefinitionBean edb : teb.getResults()){
						if(edb instanceof MessageEventDefinitionBean){
							return true;
						}
					}
				}
			}
			else if(node instanceof SendTaskBean){
				return true;
			}
			return false;
		}
		
		private boolean canBeMFTarget(IFlowNodeBean node){
			if(node instanceof CatchEventBean){
				CatchEventBean ceb = (CatchEventBean) node;
				if(ceb.getTriggers()!=null){
					for(IEventDefinitionBean edb : ceb.getTriggers()){
						if(edb instanceof MessageEventDefinitionBean){
							return true;
						}
					}
				}
			}
			else if(node instanceof ReceiveTaskBean){
				return true;
			}
			return false;
		}
		
		private IParticipantBean findEnclosingPool(IFlowElementBean fe){
			for(IParticipantBean pool : currentCollab.getParticipants()){
				if(pool.getProcess()!=null && pool.getProcess().getFlowNodes().contains(fe)){
					return pool;
				}
			}
			return null;
		}
		
	}
	
	
	
	public enum BPMNValidationErrorMessage{
		SF_NO_SOURCE("A sequence flow must have a source flow node."),
		SF_NO_TARGET("A sequence flow must have a target flow node."),
		SF_SAME_SOURCE_TARGET("A sequence flow cannot have the same object as source and target."),
		SF_COND_MISSING("A sequence flow going out of an exclusive gateway should have a condition."),
		SF_SOURCE_TARGET_WRONG_POOL("The source and the target of a sequence flow must be in the same pool."),
		SF_PARRALLEL_GATE_CONDITION("A sequence flow going out of a parallel gateway should not be conditional."),
		
		MF_NO_SOURCE("A message flow must have a source flow node."),
		MF_NO_TARGET("A message flow must have a target flow node."),
		MF_SAME_SOURCE_TARGET("A message flow cannot have the same object as source and target."),
		//MULTIPLE_INCOMING_MF("A flow element can only have one incoming message flow."),
		MULTIPLE_OUGOING_MF("A flow element can only have one outgoing message flow."),
		MF_SAME_POOL("A message flow must link two flow nodes that belong to two different pools."),
		MF_WRONG_SOURCE("The source of a message flow should be a message start event or a send task."),
		MF_WRONG_TARGET("The target of a message flow should be a message end event or a receive task."),
		
		NO_MSG_RECEIVE_TASK("A receive task should either be the target of a message flow or be linked to a message event."),
		NO_MSG_SEND_TASK("A send task should either be the source of a message flow or be linked to a message event."),
		
		MERGING_DIVERGING_GATE("A gateway cannot merge and diverge the flow (non normative, please use two sequencial gateways)."),
		FORWARDING_GATE("Each gateway should merge or diverge the flow."),
		GATEWAY_NO_OUTGOING("A gateway should have at least one outgoing sequence flow."),
		GATEWAY_NO_INCOMING("A gateway should have at least one incoming sequence flow."),
		
		NO_START("A process having an end event should have at least one start event."),
		NO_POOL("A collaboration should have at least one pool."),
		EMPTY_POOL("A pool cannot be empty."),
		EMPTY_LANE("A lane cannot be empty."),
		
		END_OUTGOING("An end event should not outgoing sequence flows."),
		END_INCOMING("An end event should have at least one incoming sequence flow."),
		
		START_NO_OUTGOING("A start event should have at least one outgoing sequence flow."),
		START_NO_INCOMING("A start event should not have incoming sequence flows.");
		
		private String msg;
		
		private BPMNValidationErrorMessage(String msg){
			this.msg = msg;
		}
		
		public String getMessage(){
			return msg;
		}
	}
	
}
