/**
 * diagram-interchange - Diagram Interchange Domain - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.AttributeExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.ObjectExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagram;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IEdge;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IStyle;

public abstract class DiagramElement implements IDiagramElement {
	
	public static final String IDSuffix = "_diagram";

	private IStyle style;
	private IDiagramElement owningElement;
	private IModelElement modelElement;
	private LinkedHashSet<IDiagramElement> ownedElements;
	private IDiagram owningDiagram;
	private List<ObjectExtension> objectExtensions;
	private List<AttributeExtension> attributeExtensions;
	private Set<IEdge> incomingEdges;
	private Set<IEdge> outgoingEdges;
	
	public DiagramElement() {
		this.ownedElements = new LinkedHashSet<IDiagramElement>();
		objectExtensions = new ArrayList<ObjectExtension>();
		attributeExtensions = new ArrayList<AttributeExtension>();
		incomingEdges		= new HashSet<IEdge>();
		outgoingEdges		= new HashSet<IEdge>();
	}
	
	@Override
	public Set<IEdge> getIncomingEdges() {
		return incomingEdges;
	}
	
	@Override
	public Set<IEdge> getOutgoingEdges() {
		return outgoingEdges;
	}
	
	@Override
	public void addIncomingEdge(IEdge edge) {
		this.incomingEdges.add(edge);
	}
	
	@Override
	public void removeIncomingEdge(IEdge edge) {
		this.incomingEdges.remove(edge);
	}
	
	@Override
	public void addOutgoingEdge(IEdge edge) {
		this.outgoingEdges.add(edge);
	}
	
	@Override
	public void removeOutgoingEdge(IEdge edge) {
		this.outgoingEdges.remove(edge);
	}
	
	@Override
	public void setIncomingEdges(Set<IEdge> edges) {
		this.incomingEdges = edges;
	}

	@Override
	public void setOutgoingEdges(Set<IEdge> edges) {
		this.outgoingEdges = edges;
	}
	
	@Override
	public void addDiagramElement(IDiagramElement diagramElement) {
		this.addOwnedElement(diagramElement);
		diagramElement.setOwningElement(this);
	}
	
	@Override
	public void removeDiagramElement(IDiagramElement diagramElement){
		this.removeOwnedElement(diagramElement);
	}
	
	protected void addOwnedElement(IDiagramElement diagramElement) {
		this.ownedElements.add(diagramElement);
	}

	protected void removeOwnedElement(IDiagramElement diagramElement) {
		this.ownedElements.remove(diagramElement);
		//also remove the references from linked diagram elements
		for(IEdge edge : getIncomingEdges()) {
			edge.setTarget(null);
		}
		for(IEdge edge : getOutgoingEdges()) {
			edge.setSource(null);
		}
		if(diagramElement instanceof IEdge) {
			IEdge edge = (IEdge) diagramElement;
			if(edge.getSource()!=null) edge.getSource().removeOutgoingEdge(edge);
			if(edge.getTarget()!=null) edge.getTarget().removeIncomingEdge(edge);
		}
	}

	@Override
	public String getId() {
		return getModelElement().getId() + IDSuffix;
	}

	@Override
	public LinkedHashSet<IDiagramElement> getOwnedElements() {
		return ownedElements;
	}
	
	@Override
	public IDiagramElement getOwningElement() {
		return owningElement;
	}

	@Override
	public IModelElement getModelElement() {
		return modelElement;
	}

	@Override
	public IDiagram getOwningDiagram() {
		return owningDiagram;
	}

	@Override
	public IStyle getStyle() {
		return style;
	}

	@Override
	public void setModelElement(IModelElement modelElement) {
		this.modelElement = modelElement;
	}

	public void setStyle(IStyle style) {
		this.style = style;
	}

	@Override
	public void setOwningElement(IDiagramElement owningElement) {
		this.owningElement = owningElement;
	}

	public void setOwnedElements(LinkedHashSet<IDiagramElement> ownedElements) {
		this.ownedElements = ownedElements;
	}

	public void setOwningDiagram(IDiagram owningDiagram) {
		this.owningDiagram = owningDiagram;
	}


	////////////////////////////////////////////
	/////                                  /////
	/////       	EXTENSIONS		       /////
	/////                                  /////
	////////////////////////////////////////////
	@Override
	public void addObjectExtension(ObjectExtension obj){
		this.objectExtensions.add(obj);
	}
	
	@Override
	public void removeObjectExtension(ObjectExtension obj){
		this.objectExtensions.remove(obj);
	}
	
	@Override
	public void addAttributeExtension(AttributeExtension att){
		for(AttributeExtension a : attributeExtensions){
			if(a.hasSameErasure(att)){
				String qname = a.getAttributeQNameNS()+":"+a.getAttributeQNameLocalPart();
				throw new IllegalArgumentException(
						"This element ("+getId()+") already has an attribute with QName "+qname+".");
			}
		}
		this.attributeExtensions.add(att);
	}
	
	@Override
	public void removeAttributeExtension(AttributeExtension att){
		this.attributeExtensions.remove(att);
	}
	
	@Override
	public AttributeExtension getAttributeExtension(String ns, String lp){
		for(AttributeExtension att : attributeExtensions){
			if(att.getAttributeQNameNS().equals(ns) && att.getAttributeQNameLocalPart().equals(lp)){
				return att;
			}
		}
		return null;
	}
	
	@Override
	public List<ObjectExtension> getObjectExtensions() {
		return objectExtensions;
	}

	@Override
	public void setObjectExtensions(List<ObjectExtension> objectExtensions) {
		this.objectExtensions = objectExtensions;
	}

	@Override
	public List<AttributeExtension> getAttributeExtensions() {
		return attributeExtensions;
	}

	@Override
	public void setAttributeExtensions(
			List<AttributeExtension> attributeExtensions) {
		this.attributeExtensions = attributeExtensions;
	}
}
