/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.diagrameditor;


import com.ebmwebsourcing.geasytools.diagrameditor.api.events.builder.IModelBuilderHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.events.loader.IDiagramViewLoaderHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.events.validation.IRuleNotificationHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.events.validation.IValidationHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.modeleditor.IEditorView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.syntax.ISyntaxModelBuilder;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleNotificationRegistry;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleNotificationView;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagram;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IMainModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.builder.BuildDiagramInterchangeCompleteEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.ElementLoadedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadErrorEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadSuccessEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.loader.LoadingStateChangeEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.HideContextualNotificationsEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.HideRulesNotificationWithLevelEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.NotificationSelectedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.RuleResolvedEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ShowContextualNotificationsEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ShowRulesNotificationWithLevelEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationFailureEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationStartEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationSuccessEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.events.validation.ValidationWarningEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.DiagramValidationException;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.RuleNotificationRegistry;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.event.shared.HandlerManager;

public class DiagramController implements  IDiagramController {

	private HandlerManager handlerManager;
	private DiagramInterchangeModelBuilder diBuilder;
	private DiagramEditorModelBuilder editorModelBuilder;
	private ContextBuilder contextBuilder;
	
	private DiagramElementViewStateHandler graphicStateHandler;
	private DiagramValidationHandler diagramViewsValidationHandler;

	
	protected DiagramViewLoader diagramViewLoader;
	
	private IEditorView editorView;
	private IDiagramView diagramView;
	
	private IMainModelElement mainModelElement;
	//private IDiagram diagramModel;
	private IRuleNotificationRegistry ruleNotificationRegistry;
	private IRuleNotificationView ruleNotificationView;
	
	private ISyntaxModelBuilder<?> syntaxModelBuilder;
	
	private UIModeHandlers uiModeHandlers;
	
	
	public DiagramController(IDiagramView diagramUI,IDiagram diagramModel,IMainModelElement mainModel) {
		this.handlerManager 				= new HandlerManager(this);		
		this.diagramView					= diagramUI;
		this.diBuilder		 				= new DiagramInterchangeModelBuilder(diagramUI,this);
		this.mainModelElement				= mainModel;
		
		this.graphicStateHandler 			= new DiagramElementViewStateHandler(this);
		this.diagramViewLoader				= new DiagramViewLoader(this);
		this.uiModeHandlers					= new UIModeHandlers(this);
		
		this.diagramViewsValidationHandler 	= new DiagramValidationHandler(this,diagramUI);
		this.ruleNotificationRegistry		= new RuleNotificationRegistry(this);
		this.contextBuilder					= new ContextBuilder(this); //must be declated before DiagramEditorModelBuilder !		
		
		DiagramViewListener listener = new DiagramViewListener(this);
		listener.listen(diagramUI);
		
		this.loadDiagram(diagramModel, false);
	}
	
	public DiagramController(IDiagramView diagramUI,IDiagram diagramModel,IEditorView editorView,IMainModelElement mainModel) {
		this(diagramUI,diagramModel,mainModel);
		this.editorView 			= editorView;
		this.editorView.setDataProvider(mainModel);

		this.editorModelBuilder 	= new DiagramEditorModelBuilder(diagramUI,this);
	}
	
	public void setSyntaxModelBuilder(ISyntaxModelBuilder<?> syntaxModelBuilder) {
		this.syntaxModelBuilder = syntaxModelBuilder;
		this.syntaxModelBuilder.setMainDiagramElement(diagramView.getDiagram().getRootElement());
	}

	
	protected DiagramElementViewStateHandler getGraphicStateHandler() {
		return graphicStateHandler;
	}
	

	public ContextBuilder getContextBuilder() {
		return contextBuilder;
	}
	
	
	public void setNotificationView(IRuleNotificationView view){
		this.ruleNotificationView = view;
	}
	

	
	public void loadDiagram(IDiagram diagramInterchange,boolean getWithRelativeCoordinates){
		this.uiModeHandlers.deactivate();
		this.diagramViewLoader.loadDiagram(diagramView,diagramInterchange,getWithRelativeCoordinates);
		this.uiModeHandlers.activate();
	}
	
	public IDiagram getDiagramInterchangeModel(boolean getWithRelativeCoordinates){
		IDiagram diagram = this.diBuilder.getDiagramInterchangeModel();
		diagram.getRootElement().setModelElement(getDiagramSyntaxModel());
		this.fireEvent(new BuildDiagramInterchangeCompleteEvent(diagramView));
		return diagram;
	}
	
	/**
	 * Returns actual diagram Syntax Model if validation pass 
	 * otherwise returns null
	 * @return
	 */
	public IModelElement getDiagramSyntaxModel() {
		//bind the diagram syntax model
		return this.syntaxModelBuilder.getSyntaxModel();
	}
	
	public ISyntaxModelBuilder<?> getSyntaxModelBuilder() {
		return syntaxModelBuilder;
	}
	
	public void setEditorView(IEditorView editorView){
		this.editorView = editorView;
	}
	
	public IEditorView getEditorView() {
		return editorView;
	}
	
	
	public IDiagramView getDiagramView() {
		return diagramView;
	}
	
	public void addLoaderHandler(IDiagramViewLoaderHandler handler) {
		handlerManager.addHandler(LoadingStateChangeEvent.TYPE, handler);
		handlerManager.addHandler(LoadSuccessEvent.TYPE, handler);
		handlerManager.addHandler(ElementLoadedEvent.TYPE, handler);
		handlerManager.addHandler(LoadErrorEvent.TYPE, handler);
	}
	
	public void addValidationHandler(IValidationHandler handler) {
		handlerManager.addHandler(ValidationSuccessEvent.TYPE, handler);
		handlerManager.addHandler(ValidationFailureEvent.TYPE, handler);
		handlerManager.addHandler(ValidationWarningEvent.TYPE, handler);
		handlerManager.addHandler(ValidationStartEvent.TYPE, handler);
	}
	
	public void addRuleNotificationHandler(IRuleNotificationHandler handler) {
		handlerManager.addHandler(NotificationSelectedEvent.TYPE, handler);
		handlerManager.addHandler(RuleResolvedEvent.TYPE, handler);
		handlerManager.addHandler(ShowRulesNotificationWithLevelEvent.TYPE, handler);
		handlerManager.addHandler(HideRulesNotificationWithLevelEvent.TYPE, handler);
		handlerManager.addHandler(ShowContextualNotificationsEvent.TYPE, handler);
		handlerManager.addHandler(HideContextualNotificationsEvent.TYPE, handler);
	}
	
	public void addModelBuilderHandler(IModelBuilderHandler handler){
		handlerManager.addHandler(BuildDiagramInterchangeCompleteEvent.TYPE, handler);
	}
	
	public void undo(){
		
	}
	
	public void redo(){
		
	}

	public void validate() throws DiagramValidationException{
		
		this.diagramViewsValidationHandler.validate();
	
		if (this.getNotificationRegistry().getRulesInFailure().size()>0) {
			
			String msg = "";
			
			for(IConformityRule r : this.getNotificationRegistry().getRulesInFailure()){
				msg = msg + "</br> - " + r.getRuleName() + "</br>";
			}
			
			throw new DiagramValidationException(msg);
			
		}
	
	}
	
	@Override
	public void fireEvent(GwtEvent<?> event) {
		handlerManager.fireEvent(event);
	}

	@Override
	public IRuleNotificationRegistry getNotificationRegistry() {
		return ruleNotificationRegistry;
	}

	@Override
	public IRuleNotificationView getRuleNotificationView() {
		return ruleNotificationView;
	}

	@Override
	public IMainModelElement getMainModelElement(){		
		return this.mainModelElement;
	}
	
	protected IMainModelElement getMainModelElementWithoutValidation(){
		return this.mainModelElement;
	}

	@Override
	public void removeSelectedElements() {
		this.diagramView.removeSelectedElements();
	}
	
	
	
}
