/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.geasytools.diagrameditor.helper.designer;

import java.util.ArrayList;
import java.util.HashSet;

import com.ebmwebsourcing.geasytools.diagrameditor.DiagramController;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.test.IDiagramAssert;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IDiagramElementViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IDiagramViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.helper.designer.event.DesignerCompleteEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.helper.designer.event.DesignerStartEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.helper.designer.event.DiagramElementPropertyChangeEvent;
import com.ebmwebsourcing.geasytools.diagrameditor.helper.designer.event.IDiagramDesignerHandler;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.test.DiagramAssert;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.test.DiagramAssertionException;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectable;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnector;
import com.ebmwebsourcing.geasytools.geasyui.api.core.IUIElement;
import com.ebmwebsourcing.geasytools.geasyui.api.draggable.IDraggableElement;
import com.ebmwebsourcing.geasytools.geasyui.api.draggable.IDraggableProxyDefaultHandlers;
import com.ebmwebsourcing.geasytools.geasyui.api.draggable.IHasDragProxy;
import com.ebmwebsourcing.geasytools.geasyui.api.draggable.IHasDragProxyData;
import com.ebmwebsourcing.geasytools.geasyui.api.draggable.events.IDragProxyHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.IUIPanel;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.event.shared.HasHandlers;
import com.google.gwt.user.client.ui.Widget;

public class DiagramDesigner implements IDiagramAssert,HasHandlers{

	private IDiagramView diagramView;
	private int periodMillis = 1000;
	private DesignerEngine reactor;
	protected ArrayList<IAction> actions;
	private HandlerManager handlerManager;
	
	private boolean previousDropAccepted;
	
	private IDiagramAssert testCase;
	
	public DiagramDesigner(DiagramController controller) {
		
		this.diagramView 	= controller.getDiagramView();
		this.actions 	 	= new ArrayList<IAction>();
		
		this.handlerManager = new HandlerManager(this);
		
		this.reactor 		= new DesignerEngine(this);
		
		this.testCase       = new DiagramAssert(controller);
	}
	

	public int getPeriodMillis() {
		return periodMillis;
	}
	
	public DiagramDesigner(DiagramController controller,IDiagramAssert testCase){
		this(controller);
		this.testCase = testCase;
	}
	
	public void play(){
		this.reactor.scheduleRepeating(periodMillis);
		fireEvent(new DesignerStartEvent(this));
	}
	
	public void setPeriodMillis(int periodMillis) {
		this.periodMillis = periodMillis;
	}
	
	public void addAction(IAction action){
		actions.add(action);
		if (periodMillis==0) action.execute(); 
	}
	
	public int getEstimatedExecutionTime(){
		return ((actions.size()*periodMillis));
	}
	
	private void addElement(final IDiagramElementView element,final int x,final int y){
		
		element.hide();
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				diagramView.addUIElement(element);
				element.show();
				element.setRelativeX(x);
				element.setRelativeY(y);
			}
		});

	}
	
	
	
	public DiagramElementProxy dropElement(final Class<? extends IDiagramElementView > clazz,final int x,final int y){

			final DiagramElementProxy proxy = new DiagramElementProxy(this);
		
			addAction(new IAction() {
				
				@Override
				public void execute() {

					if (diagramView.getAcceptedTypes().contains(clazz)){
					
						IDiagramElementView element = (IDiagramElementView) diagramView.getElementFactory().getElement(new FakeDragProxy(clazz));			
						
						proxy.setDiagramElement(element);
						
						diagramView.addUIElement(element);
						
						element.show();
						element.setRelativeX(x);
						element.setRelativeY(y);
						
						previousDropAccepted = true;
						
					}else{
						
						previousDropAccepted = false;
						
					}
					
				}
			});

		
		return proxy;
	}
	
	public void delete(final DiagramElementProxy<?> element){
		this.addAction(new IAction() {
			
			@Override
			public void execute() {
				
				diagramView.removeElement(element.getDiagramElement());
				
			}
		});

	}
	
	
	private void addElement(IDiagramElementView element){
		this.addElement(element,0,0);
	}
	
	public void addHandler(IDiagramDesignerHandler handler){
		handlerManager.addHandler(DesignerStartEvent.TYPE, handler);
		handlerManager.addHandler(DesignerCompleteEvent.TYPE, handler);
		handlerManager.addHandler(DiagramElementPropertyChangeEvent.TYPE, handler);
	}

	
	public void selectElement(IDiagramElementView element){
		
	}
	
	
	public void connectElements(final DiagramElementProxy source,final DiagramElementProxy target,final Class<? extends IDiagramElementView > connectorClazz){
		
		assert source != null : "Source cannot be null";
		assert target !=null : "Target cannot be null";
		

		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				IDiagramElementView connector = (IDiagramElementView) diagramView.getElementFactory().getElement(new FakeDragProxy(connectorClazz));	
				
				assert connector!=null;
				assert (source.getDiagramElement() instanceof IConnectable);
				assert (target.getDiagramElement() instanceof IConnectable);
				
				if (connector instanceof IConnector){
					
					((IConnector) connector).connect((IConnectable)source.getDiagramElement(),(IConnectable)target.getDiagramElement());
					
				}else{
					
					throw new IllegalStateException(connector + " is not instance of "+IConnector.class); 
					
				}
				
			}
		});
		
	}
	

	
	
	
	private class FakeDragProxy implements IHasDragProxy{
		
		private Class<? extends IUIElement> element;
		
		public FakeDragProxy(Class<? extends IUIElement> element) {
			this.element = element;
		}

		@Override
		public void addDragProxyHandler(IDragProxyHandler handler) {
			// TODO Auto-generated method stub
			
		}

		@Override
		public IUIElement getCreatedElement() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public IHasDragProxyData getData() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public IDraggableElement getDraggableProxy() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public IDraggableProxyDefaultHandlers getDraggableProxyDefaultHandlers() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public HashSet<Class<? extends IUIElement>> getDraggedTypes() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public Class<? extends IUIElement> getIUIElementType() {
			return element;
		}

		@Override
		public String getProxyImgUrl() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public IUIPanel getUIPanel() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public void setCreatedElement(IUIElement createdElement) {
			// TODO Auto-generated method stub
			
		}

		@Override
		public void setDraggableProxy(IDraggableElement draggableElement) {
			// TODO Auto-generated method stub
			
		}

		@Override
		public void setUIPanel(IUIPanel panel) {
			// TODO Auto-generated method stub
			
		}


		public Widget asWidget() {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public HandlerRegistration addMouseOutHandler(MouseOutHandler handler) {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public void fireEvent(GwtEvent<?> event) {
			// TODO Auto-generated method stub
			
		}

		@Override
		public HandlerRegistration addMouseDownHandler(MouseDownHandler handler) {
			// TODO Auto-generated method stub
			return null;
		}

		@Override
		public HandlerRegistration addMouseMoveHandler(MouseMoveHandler handler) {
			// TODO Auto-generated method stub
			return null;
		}
		
	}
	
	
	
	private abstract class ConnectionAction implements IAction {
		
		private IDiagramElementView source;
		private IDiagramElementView target;
		
		public ConnectionAction(IDiagramElementView source,IDiagramElementView target) {
			
			this.source = source;
			this.target = target;
			
		}
		
		public IDiagramElementView getSource() {
			return source;
		}
		
		public IDiagramElementView getTarget() {
			return target;
		}
		
	}
	
	

	////////////////////////////////////////////
	//ASSERTIONS
	/////////////////////////

	
	public void assertPreviousDropWasRefused(){
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				if (previousDropAccepted) throw new DiagramAssertionException("Previous drop wasn't refused");
				
			}
		});

			
		
		
	}
	
	public void assertPreviousDropWasAccepted(){
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				if (!previousDropAccepted) throw new DiagramAssertionException("Previous drop wasn't accepted");
				
			}
		});

		
	}
	
	
	public void assertElementIsConformToRule(final DiagramElementProxy<?> element,
			final Class<? extends IDiagramElementViewConformityRule> ruleClazz){
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertElementIsConformToRule(element, ruleClazz);
				
			}
		});
		
	}
	
	public void assertElementIsNotConformToRule(final DiagramElementProxy<?> element,
			final Class<? extends IDiagramElementViewConformityRule> ruleClazz){
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertElementIsNotConformToRule(element, ruleClazz);
				
			}
		});
		
	}
	
	
	@Override
	public void assertValidationFails() {
		
		this.addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertValidationFails();
				
			}
		});
		
	}


	@Override
	public void fireEvent(GwtEvent<?> event) {
		handlerManager.fireEvent(event);
	}
	
	public void execute(){
		
		for(int i=0;i<actions.size();i++){
			
			actions.get(i).execute();
			
		}
	
	}


	@Override
	public void assertValidationDoesNotFail() {
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertValidationDoesNotFail();
				
			}
		});
		
	}


	@Override
	public void assertDiagramIsConformToRule(
			final Class<? extends IDiagramViewConformityRule> ruleClazz) {

		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertDiagramIsConformToRule(ruleClazz);
				
			}
		});
		
	}


	@Override
	public void assertDiagramIsNotConformToRule(
			final Class<? extends IDiagramViewConformityRule> ruleClazz) {

		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertDiagramIsNotConformToRule(ruleClazz);
				
			}
		});
		
	}


	@Override
	public void assertDiagramElementsCount(final int count) {
		
		addAction(new IAction() {
			
			@Override
			public void execute() {
				
				testCase.assertDiagramElementsCount(count);
				
			}
		});
		
	}
	
}
