/**
 * model-manager - Handles models on client side for stuffs like undo/redo, methods observers, uibinding ... - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.reflection;

import java.util.Collection;
import java.util.HashMap;

import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.IHasReflectionHandler;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.IHasReflectionProxy;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.annotation.Reflection;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.event.AdderCalledEvent;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.event.FieldValueChangedEvent;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.event.RemoverCalledEvent;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.reflection.event.SetFieldValueCalledEvent;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder.SetterHandler;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Body;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.ClassTypeHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.ComposerHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Field;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.JClassTypeHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.JTypeHelper;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Method;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.rebind.helper.Visibility;
import com.google.gwt.core.ext.Generator;
import com.google.gwt.core.ext.GeneratorContext;
import com.google.gwt.core.ext.TreeLogger;
import com.google.gwt.core.ext.UnableToCompleteException;
import com.google.gwt.core.ext.typeinfo.JClassType;
import com.google.gwt.core.ext.typeinfo.JField;
import com.google.gwt.core.ext.typeinfo.JPrimitiveType;
import com.google.gwt.event.shared.HandlerManager;

public class ReflectionProxyGenerator extends Generator {

	private static final String handlerManagerFieldName = "handlerManager";
	public static String getHandlerManagerMethodName = "getHandlerManager()"; 
	private static final String suffix = "_proxy";

	private JClassType type;
	private ComposerHelper composerHelper;
	private JClassTypeHelper helper;
	private String simpleClassName;
	private Reflection reflectionAnnotation;

	private Field handlerManagerField;
	private Field fieldsField;


	@Override
	public String generate(TreeLogger logger, GeneratorContext context,
			String typeName) throws UnableToCompleteException {
		simpleClassName 		= type.getSimpleSourceName() + "_ReflectionProxy";

		composerHelper		   	= new ComposerHelper(context, logger, type.getPackage().getName(), simpleClassName);

		composerHelper.setSuperClass(typeName);

		helper = new JClassTypeHelper(type);

		reflectionAnnotation	= type.getAnnotation(Reflection.class);

		composerHelper.addInterface(IHasReflectionProxy.class);

		handlerManagerField = new Field(Visibility.PRIVATE, HandlerManager.class, handlerManagerFieldName);
		composerHelper.addField(handlerManagerField);

		createConstructor();

		createGetHandlerManagerMethod();

		createGetFieldsMethod();

		createSetFieldValue(Object.class);
		createSetFieldValue(Integer.class);
		createSetFieldValue(Long.class);
		createSetFieldValue(Float.class);
		createSetFieldValue(Double.class);
		createSetFieldValue(Boolean.class);

		createCopyMethod();

		overrideAllFields();

		overrideAllSetters();

		overrideAllGetters();

		overrideAllAddersAndRemovers();

		createAddHandlerMethod();

		createGetRealTypeMethod();

		composerHelper.commit();

		return composerHelper.getCreatedClassName();
	}


	private void createGetRealTypeMethod(){
		Method m = new Method(Visibility.PUBLIC,Class.class,"getRealType");
		m.isOverride(true);
		m.getBody().append("return "+type.getQualifiedSourceName()+".class");
		composerHelper.addMethod(m);
	}


	private void createAddHandlerMethod(){
		Method addHandlerMethod = new Method(Visibility.PUBLIC,void.class,"addHandler");
		addHandlerMethod.isOverride(true);
		addHandlerMethod.addParameter(addHandlerMethod.new Parameter(IHasReflectionHandler.class, "handler"));

		composerHelper.addImport(SetFieldValueCalledEvent.class);

		addHandlerMethod.getBody().append(handlerManagerField.getName()+".addHandler("+FieldValueChangedEvent.class.getCanonicalName()+".TYPE,handler)");

		composerHelper.addMethod(addHandlerMethod);
	}

	
	private boolean ignoreField(JField field){
		if(field.isFinal()) {
			return true;
		}

		if (reflectionAnnotation!=null){
			for(String fieldName:reflectionAnnotation.skipFields()){
				if (fieldName.equals(field.getName())){
					return true;
				}
			}
		}
		return false;
	}


	private void createConstructor(){
		fieldsField = new Field(Visibility.PRIVATE,HashMap.class,"fields",String.class,com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder.Field.class);
		composerHelper.addField(fieldsField);

		Method constructor = new Method(Visibility.PUBLIC, simpleClassName);

		constructor.getBody().append("super()");

		constructor.getBody().append(handlerManagerField.getName()+" = "+getHandlerManagerMethodName);

		//populate fields field
		for(JField field:helper.getAllFields()){
			if (ignoreField(field)==false){
				Class<?> fieldClass = JTypeHelper.getClass(field.getType());
				Field tmpField      = new Field(Visibility.PRIVATE,fieldClass,field.getName(),field.getType().isParameterized());

				Method getter		= tmpField.getGetter(false);

				//initialize fields
				//only if they are not null ie they have been initialized in a parent constructor 
				appendIsFieldInitializedIf(field,constructor.getBody());
				if(Collection.class.isAssignableFrom(fieldClass)) {
					constructor.getBody().append("if(super."+getter.getName()+"()!=null) {");
					constructor.getBody().append("this."+field.getName()+suffix+" = new "+ClassTypeHelper.getCollectionSubTypeName(fieldClass)+"(super."+getter.getName()+"())");
					constructor.getBody().append("}");
					constructor.getBody().append("else {");
					constructor.getBody().append("this."+field.getName()+suffix+" = new "+ClassTypeHelper.getCollectionSubTypeName(fieldClass)+"()");
					constructor.getBody().append("}");
				}
				else {
					constructor.getBody().append("this."+field.getName()+suffix+" = super."+getter.getName()+"()");
				}
				constructor.getBody().append("}");
			}
		}

		constructor.getBody().append(fieldsField.getName()+" = getFields()");

		composerHelper.addMethod(constructor);
	}

	
	private void appendIsFieldInitializedIf(JField field, Body methodBody) {
		JPrimitiveType pt = field.getType().isPrimitive();
		if(pt!=null) {
			if(pt==JPrimitiveType.BOOLEAN) {
				methodBody.append("if (this."+field.getName()+suffix+"==false) {");
			}
			else if(JTypeHelper.isNumberJtype(pt)) {
				methodBody.append("if (this."+field.getName()+suffix+"==0) {");
			}
			else {
				//FIXME ?? this case is for char, byte and void fields 
				methodBody.append("if (true) {");
			}
		}
		else {
			methodBody.append("if (this."+field.getName()+suffix+"==null) {");
		}
	}

	
	private void createGetHandlerManagerMethod() {
		Method m = new Method(Visibility.PROTECTED, HandlerManager.class, "getHandlerManager");
		m.getBody().append("if("+handlerManagerField.getName()+"==null) {");
		m.getBody().instantiate(handlerManagerField,"this");
		m.getBody().append("}");
		m.getBody().append("return "+handlerManagerField.getName());
		composerHelper.addMethod(m);
	}


	private void createGetFieldsMethod(){
		Method m = new Method(Visibility.PUBLIC, HashMap.class, "getFields");
		m.isOverride(true);

		m.getBody().append("if("+fieldsField.getName()+"==null){");
		m.getBody().instantiate(fieldsField);

		composerHelper.addImport(com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.uibinder.Field.class);
		composerHelper.addImport(SetterHandler.class);

		for(JField field:helper.getAllFields()){
			if (!ignoreField(field)) {
				Field tmpField      = new Field(Visibility.PRIVATE,JTypeHelper.getClass(field.getType()),field.getName(),field.getType().isParameterized());
				Method getter		= tmpField.getGetter(false);
				Method setter = tmpField.getSetter(false, false);

				String fieldVarName = "field"+field.getName();

				m.getBody().append("Field "+fieldVarName+" = new Field(\""+field.getName()+"\","+field.getType().getQualifiedSourceName()+".class,"+field.getEnclosingType().getQualifiedSourceName()+".class)");

				m.getBody().append(String.format(fieldVarName+".setFieldValue(%s())", getter.getName()));

				m.getBody().append(fieldVarName+".setEnclosingObject(this)");

				m.getBody().append(fieldVarName+".setSetterHandler(new SetterHandler(){ " +
						"public void setValue(Object value){" +
						setter.getName()+"(("+JTypeHelper.getClass(field.getType()).getCanonicalName()+")value);"+
						"}"+
						"} )");

				m.getBody().append(String.format("this.fields.put(\"%s\",%s)", field.getName(),fieldVarName));
			}
		}

		m.getBody().append("}");
		m.getBody().append("return "+fieldsField.getName());

		composerHelper.addMethod(m);
	}


	private void overrideAllFields(){
		for(JField field:helper.getAllFields()){
			if (ignoreField(field)==false){
				Field newField = new Field(Visibility.PRIVATE,JTypeHelper.getClass(field.getType()),field.getName()+suffix,field.getType().isParameterized());
				newField.setJtype(field.getType());

				composerHelper.addField(newField);
			}
		}
	}


	private void overrideAllGetters(){
		for(JField field:helper.getAllFields()){
			if (!ignoreField(field)) {
				Field f = new Field(Visibility.PRIVATE, JTypeHelper.getClass(field.getType()), field.getName());
				f.setJtype(field.getType());

				Method getter = f.getGetter(true,suffix);
				getter.getBody().prepend("super."+getter.getName()+"()");

				composerHelper.addMethod(getter);
			}
		}
	}


	private void overrideAllSetters(){
		for(JField field:helper.getAllFields()) {
			if (!ignoreField(field)) {
				Field f = new Field(Visibility.PRIVATE,JTypeHelper.getClass(field.getType()),field.getName(),field.getType().isParameterized());
				f.setJtype(field.getType());

				Method setter = f.getSetter(true,false,suffix);

				setter.getBody().clear();
				setter.getBody().append("super."+setter.call());
				
				String paramName = setter.getParameters().iterator().next().getName();
				setter.getBody().append(String.format("this.setFieldValue(\"%s\",%s)", f.getName(), paramName));
				
				setter.getBody().append(String.format("Field field = (Field) getFields().get(\"%s\")",f.getName()));
				setter.getBody().append(String.format("field.setFieldValue(%s)", f.getName()));

				composerHelper.addMethod(setter);
			}
		}
	}


	private void overrideAllAddersAndRemovers() {
		for(JField field:helper.getAllFields()) {
			if (!ignoreField(field)) {
				Class<?> fieldClass = JTypeHelper.getClass(field.getType());
				if(Collection.class.isAssignableFrom(fieldClass)) {
					Field f = new Field(Visibility.PRIVATE,JTypeHelper.getClass(field.getType()),field.getName(),field.getType().isParameterized());
					f.setJtype(field.getType());

					Method adder = f.getAdder(true, suffix);
					String adderParam = adder.getParameters().iterator().next().getName();
					adder.getBody().prepend("super."+adder.call());
					adder.getBody().append(String.format("this.%s.fireEvent(new %s(\"%s\",%s))",getHandlerManagerMethodName,AdderCalledEvent.class.getCanonicalName(),field.getName(),adderParam));
					composerHelper.addMethod(adder);
					
					Method remover = f.getRemover(true, suffix);
					String removerParam = remover.getParameters().iterator().next().getName();
					remover.getBody().prepend("super."+remover.call());
					remover.getBody().append(String.format("this.%s.fireEvent(new %s(\"%s\",%s))",getHandlerManagerMethodName,RemoverCalledEvent.class.getCanonicalName(),field.getName(),removerParam));
					composerHelper.addMethod(remover);
				}
			}
		}
	}


	private void createCopyMethod(){
		Method m = new Method(Visibility.PUBLIC,void.class ,"copy");
		m.addParameter(m.new Parameter(IHasReflectionProxy.class, "p"));

		composerHelper.addImport(Collection.class);
		//TODO: check field type before assigning value
		m.getBody().append("Collection<Field> fs = (Collection<Field>) this.fields.values()");
		m.getBody().append(
				"for(Field f1:fs){\n" +

					"for(Field f2:p.getFields().values()){\n"+

						"if (f1.getName().equals(f2.getName()) ){" +

							" f1.setValue(f2.getValue());\n"+

						"}"+
						"}\n"+

				"}");

		composerHelper.addMethod(m);
	}


	private void createSetFieldValue(Class<?> clazz){
		Method setFieldValueMethod = new Method(Visibility.PUBLIC,void.class, "setFieldValue");
		setFieldValueMethod.isOverride(true);
		setFieldValueMethod.addParameter(setFieldValueMethod.new Parameter(String.class, "name"));
		setFieldValueMethod.addParameter(setFieldValueMethod.new Parameter(clazz, "value"));

		for(JField field:helper.getAllFields()) {
			if (ignoreField(field)==false){
				Field f = new Field(Visibility.PRIVATE,JTypeHelper.getClass(field.getType()),field.getName(),field.getType().isParameterized());

				Class<?> fieldType = JTypeHelper.getClass(field.getType());

				if (ClassTypeHelper.isOfSameNumberType(clazz, fieldType)){
					setFieldValueMethod.getBody().append(String.format("if (name.equals(\"%s\")) this.%s = (%s) value", f.getName(),f.getName()+suffix,field.getType().getQualifiedSourceName()));
					setFieldValueMethod.getBody().append(String.format("if (name.equals(\"%s\")) this.%s.fireEvent(new %s(%s,%s))", f.getName(),getHandlerManagerMethodName,SetFieldValueCalledEvent.class.getCanonicalName(),"name","value"));
				}else{
					if (fieldType==clazz){
						setFieldValueMethod.getBody().append(String.format("if (name.equals(\"%s\")) this.%s = (%s) value", f.getName(),f.getName()+suffix,f.getType().getCanonicalName()));
						setFieldValueMethod.getBody().append(String.format("if (name.equals(\"%s\")) this.%s.fireEvent(new %s(%s,%s))", f.getName(),getHandlerManagerMethodName,SetFieldValueCalledEvent.class.getCanonicalName(),"name","value"));
					}
					if (clazz==Object.class && ClassTypeHelper.isPrimitive(fieldType)==false){
						setFieldValueMethod.getBody().append(String.format("if (name.equals(\"%s\")) this.%s = (%s) value", f.getName(),f.getName()+suffix,f.getType().getCanonicalName()));						
						setFieldValueMethod.getBody().append(String.format("if (name.equals(\"%s\")) this.%s.fireEvent(new %s(%s,%s))", f.getName(),getHandlerManagerMethodName,SetFieldValueCalledEvent.class.getCanonicalName(),"name","value"));					
					}
				}
			}
		}

		composerHelper.addMethod(setFieldValueMethod);
	}


	public void setType(JClassType type) {
		this.type = type;
	}

}
