/**
 * webeditor-file-repository - Petalslink is an open source editor providing tools for Service Oriented Architecture. - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasywebeditor.server.file.repository;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpSession;

import com.ebmwebsourcing.geasywebeditor.client.file.repository.exception.FileRepositoryInquiryException;
import com.ebmwebsourcing.geasywebeditor.client.file.repository.service.FileRepositoryService;
import com.ebmwebsourcing.geasywebeditor.client.file.repository.to.SearchFormData;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstance;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceFormat;
import com.ebmwebsourcing.webeditor.api.domain.project.IProjectInstanceMetaData;
import com.ebmwebsourcing.webeditor.impl.domain.exception.ServiceException;
import com.ebmwebsourcing.webeditor.server.impl.service.project.ProjectServiceImpl;
import com.ebmwebsourcing.webeditor.server.impl.service.upload.UploadService;
import com.ebmwebsourcing.webeditor.server.impl.service.user.UserServiceImpl;
import com.google.gwt.user.server.rpc.RemoteServiceServlet;



public class FileRepositoryServiceImpl extends RemoteServiceServlet implements FileRepositoryService{
    
    private static final String NOT_REPO_MSG = "The URL does not designate a compliant file repository.";

	public static final String UPLOAD_DIR = System.getProperty("java.io.tmpdir")+File.separator+".webeditor"+File.separator+"fileRepository"+File.separator;
	
	private Map<String, String> repoAddresses = Collections.synchronizedMap(new HashMap<String, String>());
	
	private Map<String, FileRepositoryInquiryService> repoDescriptions = Collections.synchronizedMap(new HashMap<String, FileRepositoryInquiryService>());




	@Override
	public Map<String, String> searchFile(SearchFormData data) throws ServiceException {
	    String userId = UserServiceImpl.getInstance().getLoggedUser().getId();
		try{
			FileRepositoryInquiryService s = repoDescriptions.get(userId);
			return s.searchFile(repoAddresses.get(userId), data);
		}
		catch (Throwable e) {
			e.printStackTrace();
			throw new ServiceException(e.getMessage());
		}
	}



	@Override
	public boolean testURL(String repoAddress, String fileRepositoryInquiryServiceName) throws ServiceException {
	    String userId = UserServiceImpl.getInstance().getLoggedUser().getId();
		if(repoAddress.endsWith(File.separator)){
			repoAddress.substring(0, repoAddress.length()-2);
		}
		try{
			//we check the server works and provides the wanted operations
			FileRepositoryInquiryService s = FileRepositoryInquiryServiceFactory.getInquiryService(fileRepositoryInquiryServiceName);
			s.testServerCompliance(repoAddress);

			//everything ok
			repoAddresses.put(userId, repoAddress);
			repoDescriptions.put(userId, s);
			return true;
		} catch(FileRepositoryInquiryException frie){
			throw new ServiceException(NOT_REPO_MSG);
		} catch (Throwable t){
			t.printStackTrace();
			throw new ServiceException(t.getMessage());
		}
	}



	@Override
	public List<IProjectInstanceMetaData> attachFilesFromRepoDescription(Map<String, String> files,
	        IProjectInstance instance, IProjectInstanceFormat metaDataFormat) throws ServiceException {
	    List<IProjectInstanceMetaData> result = new ArrayList<IProjectInstanceMetaData>();
	    ProjectServiceImpl projectService = ProjectServiceImpl.getInstance();
		for(String name : files.keySet()){
			try{
			    String userId = UserServiceImpl.getInstance().getLoggedUser().getId();
				File tempFile = createTemporaryFile(userId, name, files.get(name));
				String key = generateKey(userId);
				
				HttpSession session = this.getThreadLocalRequest().getSession();
				UploadService.getInstance(session).registerFile(key, tempFile);
				
				projectService.setSession(session);
				result.add(projectService.addMetaData(key, instance, metaDataFormat));
			}
			catch (Throwable e) {
				e.printStackTrace();
				throw new ServiceException(e.getMessage());
			}
		}
		return result;
	}

	
    private File createTemporaryFile(String userId, String fileName, String fileId) throws IOException, FileRepositoryInquiryException {
        //prepare the temporary dir
        String dirPath = UPLOAD_DIR+userId;
        File dir = new File(dirPath);
        if(!dir.exists()){
            buildUploadDirectories(userId);
        }
        
        //prepare the temporary file
        File file = new File(dirPath+File.separator+fileName);
        if(!file.exists()){
            file.createNewFile();
        }
        file.setWritable(true);
        
        //write the content
        String content = getFileAsStringFromRepository(userId, fileId);
        FileOutputStream fos = new FileOutputStream(file);
        fos.write(content.getBytes());
        fos.close();
        
        return file;
	}


	protected String getFileAsStringFromRepository(String userId, String fileId) throws FileRepositoryInquiryException {
		FileRepositoryInquiryService s = repoDescriptions.get(userId);
		return s.getFileContentAsString(repoAddresses.get(userId), fileId);
	}


	
	
	private void buildUploadDirectories(String userId){
		File f1 = new File(UPLOAD_DIR);
		if(!f1.exists()){
			f1.mkdirs();
		}
		
		File f2 = new File(UPLOAD_DIR+userId);
		if(!f2.exists()){
			f2.mkdir();
		}
	}
	
	
	private static int cpt = 0;
	
	private synchronized static String generateKey(String userId) {
	    cpt++;
        return userId+cpt;
    }
}
