/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasySWS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasySWS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.reasoner.impl.ontology;

import java.io.File;
import java.net.URI;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.semanticweb.owlapi.apibinding.OWLManager;
import org.semanticweb.owlapi.io.RDFXMLOntologyFormat;
import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLDatatype;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyAlreadyExistsException;
import org.semanticweb.owlapi.model.OWLOntologyCreationException;
import org.semanticweb.owlapi.model.OWLOntologyDocumentAlreadyExistsException;
import org.semanticweb.owlapi.model.OWLOntologyManager;
import org.semanticweb.owlapi.model.OWLOntologyStorageException;
import org.semanticweb.owlapi.util.OWLOntologyMerger;

import com.petalslink.easiersbs.reasoner.api.ReasonerException;
import com.petalslink.easiersbs.reasoner.api.ontology.OntologyManager;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class OntologyManagerImpl implements OntologyManager {

	Logger logger = Logger.getLogger(this.getClass().toString());
	
	private OWLOntologyManager owlManager = OWLManager.createOWLOntologyManager();
	
	// Map of ontologies'URI (precising if ontology is considers as technical)
	private Map<URI, Boolean> ontologyUris = new HashMap<URI, Boolean>();
	
	// Merged ontology IRI
	private final static IRI EASIERSBS_ONTO_URI = IRI.create("http://www.ebmwebsourcing.com/ontologies/easiersbs.owl");
	
	// Change flag
	private boolean hasNewOnto = false;

	
	public void clearOntology() {
		Set<OWLOntology> ontos = owlManager.getOntologies();
		Iterator<OWLOntology> it = ontos.iterator();
		while(it.hasNext()){
			OWLOntology onto = it.next();
			IRI ontoIRI = owlManager.getOntologyDocumentIRI(onto);
			owlManager.removeOntology(onto);
			logger.finest("Ontology " + ontoIRI + " deleted.");
		}
		ontologyUris.clear();
		logger.finest("All ontologies deleted!");
	}
	
	public void importOntology(URI uri, boolean technical) throws ReasonerException {
		try {
			owlManager.setSilentMissingImportsHandling(true);
			OWLOntology onto = owlManager.loadOntology(IRI.create(uri));
			ontologyUris.put(uri, technical);
			
			// Fix Fact++ bug (unimplemented method tellDatatypeDeclarations)
			for(OWLDatatype datatype : onto.getDatatypesInSignature()){
				owlManager.removeAxioms(onto, datatype.getReferencingAxioms(onto));
			}
			
			hasNewOnto = true;
			logger.fine("Ontology " + uri.toString() + " succefully added.");
		} catch (OWLOntologyAlreadyExistsException e){
			logger.warning("Ontology already imported (URI: " + uri +")");
		} catch (OWLOntologyDocumentAlreadyExistsException e){
			logger.warning("Ontology already imported (URI: " + uri +")");
		} catch (OWLOntologyCreationException e) {
			throw new ReasonerException("Problem during ontology import", e);
		}
	}
	
	public void importOntology(URI uri) throws ReasonerException{
		this.importOntology(uri, false);
	}

	public void importOntology(File file, boolean technical) throws ReasonerException {
		this.importOntology(file.toURI(), technical);
	}
	
	public void importOntology(File file) throws ReasonerException {
		this.importOntology(file, false);
	}
	
	public void exportOntology(URI target) throws ReasonerException {
		try {
			OWLOntology onto = this.getMergedOntology();
			owlManager.saveOntology(onto, new RDFXMLOntologyFormat(), IRI.create(target));
		} catch (OWLOntologyStorageException e) {
			throw new ReasonerException("Problem during global ontology export in " + target.toString(), e);
		}
	}
	
	public OWLOntology getMergedOntology() throws ReasonerException{
		OWLOntologyMerger merger = new OWLOntologyMerger(owlManager);
		OWLOntology mergedOnto = owlManager.getOntology(EASIERSBS_ONTO_URI);

		// If there is change, we recreate the merged ontology
		if(hasNewOnto || (mergedOnto == null)){			
			if(mergedOnto != null){
				owlManager.removeOntology(mergedOnto);
			}
			try {
				mergedOnto = merger.createMergedOntology(owlManager, EASIERSBS_ONTO_URI);
			} catch (OWLOntologyCreationException e) {
				throw new ReasonerException("Problem during ontology merge", e);
			}
			hasNewOnto = false;
		} // Else we return the last one
		return mergedOnto;
	}
	
	public OWLOntologyManager getOntologyManager(){
		return owlManager;
	}
	
	public Map<URI, Integer> getOntologiesInfo(){
		Map<URI, Integer> res = new HashMap<URI, Integer>();
		for(OWLOntology onto : owlManager.getOntologies()){
			res.put(onto.getOntologyID().getOntologyIRI().toURI(), onto.getAxiomCount());
		}
		return res;
	}

	public Set<URI> getNamespaceUris() {
		return ontologyUris.keySet();
	}

	public boolean isTechnicalOntology(URI namespace) {
		if(ontologyUris.containsKey(namespace)){			
			return ontologyUris.get(namespace);
		} else {
			return false;
		}
	}
	
	

}
