/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;


import com.petalslink.easiersbs.matching.service.SBSFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.CompositionException;
import com.petalslink.easiersbs.matching.service.api.EasierSBSException;
import com.petalslink.easiersbs.matching.service.api.SBSFactory;
import com.petalslink.easiersbs.matching.service.api.matcher.HybridMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SemanticMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.SyntacticMatcher;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.api.profile.rated.RatedSemanticProfile;
import com.petalslink.easiersbs.matching.service.profile.rated.RatedSemanticProfileImpl;
import com.petalslink.easiersbs.matching.service.util.ProfileUtil;
import com.petalslink.easiersbs.reasoner.api.ReasonerException;
import com.petalslink.easiersbs.reasoner.api.ReasonerFactory;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.Operation;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class HybridMatcherImpl extends AbstractMatcherImpl implements HybridMatcher {

	private static Logger logger = Logger.getLogger("HybridMatcherImpl");
	
	private SBSFactory factory = SBSFactoryImpl.getInstance();
	
	private SemanticMatcher semanticMatcher;
	private SyntacticMatcher syntacticMatcher;
	
	private MatcherProperties props;
	
	
	public HybridMatcherImpl(ReasonerFactory reasoner, SemanticRegistryManager registry, MatcherProperties props) throws ReasonerException, EasierSBSException{
		this.props = props;
		this.registry = registry;
		semanticMatcher = factory.newSemanticMatcher(reasoner, registry, props);
		syntacticMatcher = factory.newSyntacticMatcher(registry, props);
	}
	
	public HybridMatcherImpl(ReasonerFactory reasoner, SemanticRegistryManager registry) throws ReasonerException, EasierSBSException{
		this(reasoner, registry, new MatcherPropertiesImpl());
	}
	
	@Override
	public void setMatcherProperties(MatcherProperties props) {
		this.props = props;
		semanticMatcher.setMatcherProperties(props);
		syntacticMatcher.setMatcherProperties(props);
	}
	
	@Override
	public SyntacticMatcher getSyntacticMatcher() {
		return syntacticMatcher;
	}

	@Override
	public SemanticMatcher getSemanticMatcher() {
		return semanticMatcher;
	}

	@Override
	public MatchingResult findServices(SearchProfile profile,
			Set<SemanticProfile> serviceList) {
		
		logger.fine("Hybrid matchmaking - begin");
		
		// Service selection (1-1 matchmaking)
		MatchingResult res = this.serviceSelection(profile, serviceList);
		
		// Service composition
		if(this.props.getCompositionServiceLimit() > 1 
				&& res.getBestProfile() != null
				&& res.getBestProfile().getProfileRate() < this.props.getGeneralValidationRate()) {
			MatchingResult compoRes = this.serviceComposition(res, serviceList, new HashSet<Set<Operation>>());
			for(RatedSemanticProfile compo : compoRes.getResultProfiles()){
				res.addResultProfile(compo);
			}
		}
		
		logger.fine("Hybrid matchmaking - end - " + res.getResultProfiles().size() + " results found");
		
		return res;
	}
	
	private MatchingResult serviceSelection(SearchProfile profile,
			Set<SemanticProfile> serviceList) {
		logger.fine("   Service selection - begin");
		MatchingResult hybridRes = null;
		
		if(props.getSemanticWeight() == 0.0) {
			hybridRes = syntacticMatcher.findServices(profile, serviceList);
		} else if(props.getSyntacticWeigth() == 0.0) {
			hybridRes = semanticMatcher.findServices(profile, serviceList);
		} else {
			// Hybrid matching
			hybridRes = new MatchingResultImpl(profile);
			MatchingResult semanticRes = semanticMatcher.findServices(profile, serviceList);
			MatchingResult syntacticRes = syntacticMatcher.findServices(profile, ProfileUtil.getSemanticProfiles(semanticRes.getResultProfiles()));
			
			for(RatedSemanticProfile synProfile : syntacticRes.getResultProfiles()){
				RatedSemanticProfile hybridProfile = new RatedSemanticProfileImpl(synProfile.getProfile());
				
				RatedSemanticProfile semProfile = semanticRes.findProfile(synProfile.getProfile());
				
				hybridProfile.setProfileRate(this.getHybridRate(semProfile.getProfileRate(), synProfile.getProfileRate()));
				hybridProfile.setOperationRate(this.getHybridRate(semProfile.getOperationRate(), synProfile.getOperationRate()));
				hybridProfile.setInputCoverage(semProfile.getInputCoverage());
				hybridProfile.setInputRate(this.getHybridRate(semProfile.getInputRate(), synProfile.getInputRate()));
				hybridProfile.setOutputCoverage(semProfile.getOutputCoverage());
				hybridProfile.setOutputRate(this.getHybridRate(semProfile.getOutputRate(), synProfile.getOutputRate()));
				
				hybridRes.addResultProfile(hybridProfile);
			}
		}
		
		MatchingResult res = new MatchingResultImpl(profile);
		for(RatedSemanticProfile rated : hybridRes.getResultProfiles()){
			if(rated.getProfileRate() >= props.getHybridThreshold()){
				res.addResultProfile(rated);
			}
		}
		
		logger.fine("   Service selection - end - " + res.getResultProfiles().size() + " results found");
		System.out.println("   Service selection - end - " + res.getResultProfiles().size() + " results found");
		return res;
	}
	
	private MatchingResult serviceComposition(MatchingResult prevResult,
			Set<SemanticProfile> serviceList, Set<Set<Operation>> studiedOperationSets) {
		logger.fine("   Service composition - begin");
		MatchingResult res = new MatchingResultImpl(prevResult.getQuery()); // Check if it works
		
		for(RatedSemanticProfile baseProfile : prevResult.getResultProfiles()){
			// We create complementary profile 
			// (research profile without concepts covered by base profile)
			SearchProfile compoProfile = ProfileUtil.computeComplementaryProfile(prevResult.getQuery(), 
					baseProfile.getProfile(), semanticMatcher.getReasoner(), props);
			//System.out.println("Complementary profile : " + compoProfile.toString());
			
			// We search for a new service
			// TODO: remove baseProfile and others from service list
			Set<SemanticProfile> newServiceList = new HashSet<SemanticProfile>(serviceList);
			newServiceList.remove(baseProfile);
			MatchingResult compoResults = this.serviceSelection(compoProfile, newServiceList);
			//System.out.println("Service selection on complementary profile: " + compoResults.getResultProfiles().size());
			
			for(RatedSemanticProfile compoResult : compoResults.getResultProfiles()){
				Set<Operation> involvedOps = new HashSet<Operation>(baseProfile.getProfile().getOperations());
				involvedOps.addAll(compoResult.getProfile().getOperations());
				if(studiedOperationSets.contains(involvedOps)){
					continue;
				} else {
					studiedOperationSets.add(involvedOps);
				}
				
				// We search equivalent profile
				Set<SemanticProfile> eqList = new HashSet<SemanticProfile>();
				try {
					eqList.add(ProfileUtil.computeEquivalentProfile(involvedOps, 
												registry, semanticMatcher.getReasoner(), props));
					//System.out.println("Eq profile: " + eqList.iterator().next().toString());
				} catch (CompositionException e) {
					logger.warning("   Service composition - Loop detected with operations " + involvedOps);
					continue;
				}
				
				// We compute the real rate of this composition
				MatchingResult eqRes = this.findServices(prevResult.getQuery(), eqList);
				//System.out.println("Service selection on eq profile: " + eqRes.getResultProfiles().size());
				
				// If we improved the rate, we store the new result and 
				if(eqRes.hasCorrectProfile(baseProfile.getProfileRate())){
					res.addResultProfile(eqRes.getBestProfile());
				}
			}
			
			// If there is no improvement with this baseprofile, we don't study others
			if(res.getResultProfiles().size() == 0){
				break;
			} else if(res.getBestProfile().getProfile().getOperations().size() < props.getCompositionServiceLimit()){
				logger.fine("   Service composition - level " + (res.getBestProfile().getProfile().getOperations().size() + 1));
				for(RatedSemanticProfile profile : this.serviceComposition(res, newServiceList, studiedOperationSets).getResultProfiles()){
					res.addResultProfile(profile);
				}
			}
		}
		
		logger.fine("   Service composition - end - " + res.getResultProfiles().size() + " results found");
		return res;
	}
	
	private double getHybridRate(double semanticRate, double syntacticRate) {
		// Weighted Harmonic Mean
		// return (1 / ((props.getSemanticWeight() / semanticRate) + (props.getSyntacticWeigth() / syntacticRate)));
		// Average
		return ((semanticRate * props.getSemanticWeight()) + (syntacticRate * props.getSyntacticWeigth()));
	}

}
