package org.petalslink.easiestdemo.wsoui.plugin.bpel;

/*
 * Copyright 2001-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.apache.cxf.Bus;
import org.apache.cxf.BusFactory;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.petalslink.abslayer.service.api.Interface;
import org.petalslink.easiestdemo.wsoui.core.WSOUIException;
import org.petalslink.easiestdemo.wsoui.core.bpel.AssemblyFileCopier;
import org.petalslink.easiestdemo.wsoui.core.bpel.BPELJUnitTestGenerator;
import org.petalslink.easiestdemo.wsoui.core.bpel.BPELProcessGenerator;
import org.petalslink.easiestdemo.wsoui.core.bpel.LoggerFileCopier;
import org.petalslink.easiestdemo.wsoui.core.bpel.SCACompositeDiagramGenerator;
import org.petalslink.easiestdemo.wsoui.core.bpel.SCACompositeGenerator;
import org.petalslink.easiestdemo.wsoui.core.bpel.UnixScriptsGenerator;
import org.petalslink.easiestdemo.wsoui.core.bpel.WindowsScriptsGenerator;
import org.petalslink.easiestdemo.wsoui.plugin.wsdl.WSDL2JavaGenerateSourcesMojo;
import org.petalslink.easiestdemo.wsoui.plugin.wsdl.WsdlOption;

import com.ebmwebsourcing.easybpel.model.bpel.api.BPELException;
import com.ebmwebsourcing.easybpel.model.bpel.tools.validator.AnalysorResult;
import com.ebmwebsourcing.easybpel.model.bpel.tools.validator.BPELAnalyser;
import com.ebmwebsourcing.easybpel.model.bpel.tools.validator.BPELValidator;
import com.ebmwebsourcing.easybpel.model.bpel.tools.validator.ValidatorResult;
import com.ebmwebsourcing.easybpel.model.bpel.tools.validator.ValidatorResultPrinter;
import com.ebmwebsourcing.easycommons.research.util.io.FileUtil;


/**
 * @goal bpel2java
 * @phase generate-sources
 * @description WSOUI BPEL To Java Tool
 * @requiresDependencyResolution test
 */
public class BPEL2JavaGenerateSourcesMojo extends AbstractMojo {
	
	private final Logger LOG = Logger.getLogger(BPEL2JavaGenerateSourcesMojo.class
			.getName());


	/**
	 * Path where the sources should be placed
	 * @parameter
	 */
	String sourceRoot = "/src/main/java";

	/**
	 * Path where the test sources should be placed
	 * @parameter
	 */
	String testsourceRoot = "/src/test/java";

	/**
	 * Path where the test sources should be placed
	 */
	String mainresourceRoot = "/src/main/resources";


	/**
	 * 
	 */
	String generatedTestSourceRoot = "/target/generated-test-sources/wsoui";


	/**
	 * Path where the generated sources should be placed
	 * 
	 */
	String generatedSourceRoot = "/target/generated-sources/wsoui";



	URI basedir = new File(".").toURI();

	/**
	 * @parameter expression="${project}"
	 * @required
	 */
	MavenProject project;


	/**
	 * @parameter
	 */
	BpelOption bpelOptions[];



	/**
	 * @parameter
	 */
	boolean generateEndpoint = true;


	BPELValidator validator = null;
	BPELAnalyser analyzer = null;

	File sourceRootFile = null;
	File testsourceRootFile = null;
	File mainresourceRootFile = null;
	File generatedTestSourceRootFile = null;
	File generatedSourceRootFile = null;
	File targetFile = null;

	private boolean compileClass = true; 

	public void execute() throws MojoExecutionException {

		if(project != null) {
			basedir = project.getBasedir().toURI();
			sourceRootFile = new File(FileUtil.getURI(basedir, sourceRoot));
			testsourceRootFile = new File(FileUtil.getURI(basedir, testsourceRoot));
			mainresourceRootFile = new File(FileUtil.getURI(basedir, mainresourceRoot));
			generatedTestSourceRootFile = new File(FileUtil.getURI(basedir, generatedTestSourceRoot));
			generatedSourceRootFile = new File(FileUtil.getURI(basedir, generatedSourceRoot));
			targetFile = new File(FileUtil.getURI(basedir, "/target"));
		} else {
			sourceRootFile = new File(URI.create(basedir.toString() + sourceRoot));
			testsourceRootFile = new File(URI.create(basedir.toString() + testsourceRoot));
			mainresourceRootFile = new File(URI.create(basedir.toString() + mainresourceRoot));
			generatedTestSourceRootFile = new File(URI.create(basedir.toString() + generatedTestSourceRoot));
			generatedSourceRootFile = new File(URI.create(basedir.toString() + generatedSourceRoot));
			targetFile = new File(URI.create(basedir.toString() + "/target"));
		}


		List<BpelOption> effectiveBpelOptions = Arrays.asList(this.bpelOptions);

		if (effectiveBpelOptions.size() == 0) {
			getLog().info("Nothing to generate");
			return;
		}

		try {

			validator = new BPELValidator();
			analyzer = new BPELAnalyser();


			for (BpelOption o : effectiveBpelOptions) {
				callBpel2Java(o);     
			}

		} catch (Exception e) {
			e.printStackTrace();
			throw new MojoExecutionException(e.getMessage(), e);
		} finally {
			// cleanup as much as we can.
			Bus bus = BusFactory.getDefaultBus(false);
			if (bus != null) {
				bus.shutdown(true);
			}
			org.apache.cxf.tools.wsdlto.core.PluginLoader.unload();
		}

		if(this.isCompileClass()) { 
			// add all sources to the compilation process
			if (project != null && generatedSourceRootFile != null && generatedSourceRootFile.exists()) {
				project.addCompileSourceRoot(generatedSourceRootFile.getAbsolutePath());
			}
			if (project != null && generatedTestSourceRootFile != null && generatedTestSourceRootFile.exists()) {
				project.addTestCompileSourceRoot(generatedTestSourceRootFile.getAbsolutePath());
			}
			System.gc();
		}
	}



	private void callBpel2Java(BpelOption bpelOption) throws MojoExecutionException {
		File outputDirFile = this.generatedSourceRootFile;
		outputDirFile.mkdirs();

		if(project != null) {
			basedir = project.getBasedir().toURI();
		}
		try {
			ValidatorResult report = validator.validate(bpelOption.getBpelURI(basedir).toURL());
			if(report.getErrors().size() > 0) {
				throw new MojoExecutionException(ValidatorResultPrinter.getInstance().printAll(report));
			} else {
				System.out.println(ValidatorResultPrinter.getInstance().printAll(report));
			}
			AnalysorResult result = this.analyzer.analyze(report.getProcess());

			System.out.println("List of imports in process " + report.getProcess().getName() + " : " + result.getImports());

			// find excludes wsdl
			List<URI> excludesWsdl = new ArrayList<URI>();
			for(Interface itf: report.getProcess().getProcessInterfaces()) {
				URI bpelUri = itf.getParentDescription().getDocumentBaseURI();
				excludesWsdl.add(bpelUri);
			}

			if(bpelOption.getIgnoreWsdls() != null) {
				for(String wsdl: bpelOption.getIgnoreWsdls()) {
					File cleanwsdl = new File(wsdl.replace("/", "\\"));
					excludesWsdl.add(cleanwsdl.toURI());
				}
			}

			// create wsdl options from urls in bpel
			List<WsdlOption> wsdlOptions = new ArrayList<WsdlOption>();
			boolean generate = true;
			for(URI wsdlUri: result.getImports()) {
				generate = true;

				if(!wsdlUri.toString().endsWith(".xsd")) {
					LOG.finest("*************************************************** excludesWsdl: " + excludesWsdl);
					System.out.println("*************************************************** wsdlUri: " + wsdlUri);
					if(!isContainInList(excludesWsdl, wsdlUri) && generate) {
						LOG.finest("*************************************************** accepted: " + true);
						WsdlOption wo = new WsdlOption();
						wo.setWsdl(createWsdlUrl(wsdlUri.toString(), bpelOption.getBpelURI(basedir).toString(), basedir.toString()));
						wo.setProvider(true);
						wo.setOverrideOlderBusinessClass(bpelOption.isOverrideOlderBusinessClass());
						wo.setGenerateBusinessClass(bpelOption.isGenerateBusinessClass());

						wsdlOptions.add(wo);
					} else {
						LOG.finest("*************************************************** accepted: " + false);
					}

				}
			}

			// run wsdl2java
			WSDL2JavaGenerateSourcesMojo wsdlMojo = new WSDL2JavaGenerateSourcesMojo();
			wsdlMojo.setProject(this.project);
			wsdlMojo.setCompileClass(false);
			wsdlMojo.setGeneratedSourceRoot(this.generatedSourceRoot);
			wsdlMojo.setSourceRoot(this.sourceRoot);
			wsdlMojo.setTestsourceRoot(this.testsourceRoot);
			wsdlMojo.setGeneratedTestSourceRoot(this.generatedTestSourceRoot);

			WsdlOption[] opts = wsdlOptions.toArray(new WsdlOption[wsdlOptions.size()]);
			wsdlMojo.setWsdlOptions(opts);
			wsdlMojo.execute();

			getLog().info("Calling bpel2java with process: " + new QName(report.getProcess().getTargetNamespace(), report.getProcess().getName()));


			// create BPELProcess main file
			if(bpelOption.isGenerateBusinessClass() == true) {
				BPELProcessGenerator bpg = new BPELProcessGenerator(report.getProcess(), wsdlMojo.getAllWSOUIServer(), bpelOption.isOverrideOlderBusinessClass());
				bpg.generate(this.findShortNameOfBPELFile(bpelOption.getBpelURI(basedir).toURL().toString(), basedir.toString()), sourceRootFile.toString(), bpelOption.getEsbFactoryCanonicalName());

				SCACompositeGenerator scg = new SCACompositeGenerator(report.getProcess(), wsdlMojo.getAllMetainfs(), bpelOption.isOverrideOlderBusinessClass());
				scg.generate(this.findShortNameOfBPELFile(bpelOption.getBpelURI(basedir).toURL().toString(), basedir.toString()), sourceRootFile.toString());

				SCACompositeDiagramGenerator scdg = new SCACompositeDiagramGenerator(report.getProcess(), wsdlMojo.getAllMetainfs(), bpelOption.isOverrideOlderBusinessClass());
				scdg.generate(this.findShortNameOfBPELFile(bpelOption.getBpelURI(basedir).toURL().toString(), basedir.toString()), sourceRootFile.toString());
			}

			// create BPELProcess test file
			BPELJUnitTestGenerator bjutg = new BPELJUnitTestGenerator(report.getProcess(), wsdlMojo.getAllWSOUIServer());
			bjutg.generate(this.findShortNameOfBPELFile(bpelOption.getBpelURI(basedir).toURL().toString(), basedir.toString()), testsourceRootFile.toString());


			// copy assembly
			AssemblyFileCopier assemblyCopy = new AssemblyFileCopier(this.targetFile);
			assemblyCopy.copyAssemblyFile();

			// copy logger
			LoggerFileCopier loggerCopyTarget = new LoggerFileCopier(this.targetFile);
			loggerCopyTarget.copyLoggerFile();
			LoggerFileCopier loggerCopySrc = new LoggerFileCopier(this.mainresourceRootFile);
			loggerCopySrc.copyLoggerFile();

			// generate windows scripts
			String projectName = null;
			String version = null;
			if(project != null) {
				projectName = project.getArtifactId();
				version = project.getVersion();
			}
			WindowsScriptsGenerator wsg = new WindowsScriptsGenerator(projectName, version, report.getProcess(), wsdlMojo.getAllServerMain(), bpelOption.isOverrideOlderBusinessClass());
			wsg.generateAllScripts(targetFile.toString());

			UnixScriptsGenerator usg = new UnixScriptsGenerator(projectName, version, report.getProcess(), wsdlMojo.getAllServerMain(), bpelOption.isOverrideOlderBusinessClass());
			usg.generateAllScripts(targetFile.toString());

		} catch (MalformedURLException e) {
			e.printStackTrace();
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (BPELException e) {
			e.printStackTrace();
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (WSOUIException e) {
			e.printStackTrace();
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (IOException e) {
			e.printStackTrace();
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (URISyntaxException e) {
			e.printStackTrace();
			throw new MojoExecutionException(e.getMessage(), e);
		} 
	}









	private boolean isContainInList(List<URI> excludesWsdl, URI wsdlUri) {
		boolean res = false;
		String wsdlShort = wsdlUri.toString();
		if(wsdlUri.toString().lastIndexOf("/") > 0) {
			wsdlShort = wsdlUri.toString().substring(wsdlUri.toString().lastIndexOf("/") + 1, wsdlUri.toString().length());
		}
		if(wsdlUri.toString().lastIndexOf("\\") > 0) {
			wsdlShort = wsdlUri.toString().substring(wsdlUri.toString().lastIndexOf("\\") + 1, wsdlUri.toString().length());
		}
		for(URI uri: excludesWsdl) {
			if(uri.getPath().toString().endsWith(wsdlShort)) {
				res = true;
				break;
			}
		}
		return res;
	}



	private String createWsdlUrl(String wsdlUri, String bpelUri, String basedir) {

		if(basedir.endsWith("./")) {
			basedir.substring(0, basedir.length() - "./".length());
		}
		String relativePath = bpelUri.replace(basedir, "");
		String path = relativePath.substring(0, relativePath.lastIndexOf("/") + 1) + wsdlUri;
		return path;
	}

	private String findShortNameOfBPELFile(String bpelUri, String basedir) {


		if(basedir.endsWith("./")) {
			basedir = basedir.substring(0, basedir.length() - "./".length());
		}
		String path = bpelUri.replace(basedir, "");

		path = path.replace("src/main/resources/", "");
		path = path.replace("src/test/resources/", "");
		path = path.replace("target/test-classes/", "");
		path = path.replace("target/classes/", "");
		return path;
	}


	public BpelOption[] getBpelOptions() {
		return bpelOptions;
	}


	public void setBpelOptions(BpelOption[] bpelOptions) {
		this.bpelOptions = bpelOptions;
	}


	public String getSourceRoot() {
		return sourceRoot;
	}


	public void setSourceRoot(String sourceRoot) {
		this.sourceRoot = sourceRoot;
	}


	public String getTestsourceRoot() {
		return testsourceRoot;
	}


	public void setTestsourceRoot(String testsourceRoot) {
		this.testsourceRoot = testsourceRoot;
	}


	public String getGeneratedTestSourceRoot() {
		return generatedTestSourceRoot;
	}


	public void setGeneratedTestSourceRoot(String generatedTestSourceRoot) {
		this.generatedTestSourceRoot = generatedTestSourceRoot;
	}


	public String getGeneratedSourceRoot() {
		return generatedSourceRoot;
	}


	public void setGeneratedSourceRoot(String generatedSourceRoot) {
		this.generatedSourceRoot = generatedSourceRoot;
	}


	public boolean isGenerateEndpoint() {
		return generateEndpoint;
	}


	public void setGenerateEndpoint(boolean generateEndpoint) {
		this.generateEndpoint = generateEndpoint;
	}


	public boolean isCompileClass() {
		return compileClass;
	}


	public void setCompileClass(boolean compileClass) {
		this.compileClass = compileClass;
	}

}
