
package com.ebmwebsourcing.easycommons.io;

import java.io.File;
import java.io.IOException;
import java.util.Random;

public final class FileSystemHelper {

    private static final Random random = new Random();

    private FileSystemHelper() {
    }

    private static final File getSystemTempDir() {
        String tmpDirPath = System.getProperty("java.io.tmpdir");
        assert tmpDirPath != null;
        File tmpDir = new File(tmpDirPath);
        assert tmpDir != null;
        return tmpDir;
    }

    public static File createTempDir() throws IOException {
        return createTempDir(getSystemTempDir(), "tmp");
    }

    public static File createTempDir(String prefix) throws IOException {
        return createTempDir(getSystemTempDir(), prefix);
    }

    public static File createTempDir(File parentDir, String prefix) throws IOException {
        File tempDir = null;
        while (true) {
            tempDir = new File(parentDir, prefix + random.nextLong());
            if (!tempDir.exists())
                break;
        }
        if (!tempDir.mkdir()) {
            throw new IOException(String.format("Impossible to create temp dir '%s'",
                    tempDir.getAbsolutePath()));
        }
        tempDir.deleteOnExit();
        return tempDir;
    }

    /**
     * Cleans a directory without deleting it.
     * 
     * @param directory
     *            directory to clean
     * @throws IOException
     *             in case cleaning is unsuccessful
     */
    public static void cleanDirectory(File directory) throws IOException {
        assert directory != null;
        if (!directory.exists()) {
            String message = directory + " does not exist";
            throw new IOException(message);
        }
        if (!directory.isDirectory()) {
            String message = directory + " is not a directory";
            throw new IOException(message);
        }

        File[] files = directory.listFiles();
        if (files == null) { // null if security restricted
            throw new IOException("Failed to list contents of " + directory);
        }

        IOException exception = null;
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            try {
                forceDelete(file);
            } catch (IOException ioe) {
                exception = ioe;
            }
        }

        if (null != exception) {
            throw exception;
        }
    }

    /**
     * Deletes a file. If file is a directory, delete it and all
     * sub-directories.
     * <p>
     * The difference between File.delete() and this method are:
     * <ul>
     * <li>A directory to be deleted does not have to be empty.</li>
     * <li>You get exceptions when a file or directory cannot be deleted.
     * (java.io.File methods returns a boolean)</li>
     * </ul>
     * 
     * @param file
     *            file or directory to delete, must not be <code>null</code>
     * @throws NullPointerException
     *             if the directory is <code>null</code>
     * @throws IOException
     *             in case deletion is unsuccessful
     */
    public static void forceDelete(File file) throws IOException {
        assert file != null;
        if (file.isDirectory()) {
            deleteDirectory(file);
        } else {
            if (!file.exists()) {
                return;
            }
            if (!file.delete()) {
                String message = "Unable to delete file: " + file;
                throw new IOException(message);
            }
        }
    }

    /**
     * Deletes a directory recursively.
     * 
     * @param directory
     *            directory to delete
     * @throws IOException
     *             in case deletion is unsuccessful
     */
    private static void deleteDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }

        cleanDirectory(directory);
        if (!directory.delete()) {
            String message = "Unable to delete directory " + directory + ".";
            throw new IOException(message);
        }
    }

}
