
package com.ebmwebsourcing.easycommons.logger;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;

import com.ebmwebsourcing.easycommons.lang.reflect.ReflectionHelper;

/**
 * This formatter is aimed at using first {@link LogRecord} parameter as
 * "just in time evaluated" extra log information.<br/>
 * Note that localization is still possible (parameters are shifted left if
 * necessary).<br/>
 * <br/>
 * In other words :
 * <ul>
 * <li>if first parameter is an instance of {@link LogData}, this data will be
 * appended in log message,</li>
 * <li>otherwise, first parameter is kept for localization duty, as usual.</li>.
 * 
 * @author mjambert
 * 
 */
public class LogDataFormatter extends Formatter {

    private static final String LOGDATA_CANONICAL_CLASSNAME = "com.ebmwebsourcing.easycommons.logger.LogData";

    public static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss,SSS 'GMT'Z",
            Locale.getDefault());

    private final Deque<LogDataAppender> dataAppenders;

    private String startingDelimiter = " {{";

    private String endingDelimiter = "}}";

    private String prefix = null;
    
    public void setExtraParametersDelimiter(String startingDelimiter, String endingDelimiter) {
        this.startingDelimiter = startingDelimiter;
        this.endingDelimiter = endingDelimiter;
    }

    public LogDataFormatter() {
        dataAppenders = new LinkedList<LogDataAppender>();
    }

    public final void addLogDataAppender(LogDataAppender logDataAppender) {
        assert logDataAppender != null;
        dataAppenders.add(logDataAppender);
    }

    final LogDataAppender[] getLogDataAppenders() {
        return dataAppenders.toArray(new LogDataAppender[dataAppenders.size()]);
    }
    
    public final void setPrefix(String prefix) {
    	this.prefix = prefix;
    }

    protected void formatExtraParameters(StringBuffer outBuffer, Map<String, Object> extraParameters) {
        Iterator<String> extraParametersKeySetIterator = extraParameters.keySet().iterator();
        if (extraParametersKeySetIterator.hasNext() && !dataAppenders.isEmpty()) {
            outBuffer.append(this.startingDelimiter);
            while (extraParametersKeySetIterator.hasNext()) {
                String extraParameterKey = extraParametersKeySetIterator.next();
                Iterator<LogDataAppender> itAppender = dataAppenders.descendingIterator();
                while (itAppender.hasNext()) {
                    LogDataAppender appender = (LogDataAppender) itAppender.next();
                    if (!appender.canAppend(extraParameters, extraParameterKey)) {
                        continue;
                    }
                    appender.append(outBuffer, extraParameters, extraParameterKey);
                    if (extraParametersKeySetIterator.hasNext()) {
                        outBuffer.append(", ");
                    }
                    break;
                }
            }
            outBuffer.append(this.endingDelimiter);
        }
    }

    private boolean isThereLogDataParameter(Object[] logParameters) {
        return (logParameters != null)
                && (logParameters.length != 0)
                && logParameters[0] != null
                && ReflectionHelper.isOrInheritedFrom(logParameters[0].getClass(),
                        LOGDATA_CANONICAL_CLASSNAME);
    }

    @SuppressWarnings("unchecked")
    protected void populateExtraParameters(LinkedHashMap<String, Object> extraParameters,
            LogRecord logRecord) {
        Object[] logParameters = logRecord.getParameters();
        if (isThereLogDataParameter(logParameters)) {
            extraParameters.putAll((LinkedHashMap<String, Object>) logParameters[0]);
        }
    }

    protected StringBuffer buildPattern(LogRecord record) {
        StringBuffer sb = new StringBuffer();
        String formattedLog;
        if(this.prefix == null) {
        	formattedLog = String.format("%s %s [%s] : %s", DATE_FORMAT.format(record.getMillis()),
                record.getLevel(), record.getLoggerName(), record.getMessage());
        } else {
        	formattedLog = String.format("%s %s %s [%s] : %s", prefix, DATE_FORMAT.format(record.getMillis()),
                    record.getLevel(), record.getLoggerName(), record.getMessage());
        }
        sb.append(formattedLog);
        return sb;
    }

    @Override
    public String format(LogRecord record) {
        LinkedHashMap<String, Object> extraParameters = new LinkedHashMap<String, Object>();
        populateExtraParameters(extraParameters, record);

        StringBuffer sb = buildPattern(record);
        formatExtraParameters(sb, extraParameters);
        sb.append("\n");

        if (record.getThrown() != null) {
            final StringWriter sw = new StringWriter();
            record.getThrown().printStackTrace(new PrintWriter(sw));
            sw.flush();
            sb.append(sw.toString());
        }
        return sb.toString();
    }

    @Override
    public synchronized String formatMessage(LogRecord record) {
        Object[] extraParameters = record.getParameters();
        String result = null;
        if (isThereLogDataParameter(extraParameters)) {
            Object[] extraParametersWithoutLogData = Arrays.copyOfRange(extraParameters, 1,
                    extraParameters.length);
            record.setParameters(extraParametersWithoutLogData);
            result = super.formatMessage(record);
            record.setParameters(extraParameters);
        } else {
            result = super.formatMessage(record);
        }
        return result;
    }
}
