/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2008 OW2 consortium (www.ow2.org)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id: XMLUtil.java 98 2006-02-24 16:18:48Z alouis $
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.easycommons.xml;

import javax.xml.parsers.DocumentBuilder;

import org.w3c.dom.Document;

import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;

/**
 * A static pool of {@link DocumentBuilder} to improve {@link DocumentBuilder} creation efficiency. 
 * A {@link DocumentBuilder} is initialized the first time this class is used.
 * An infinite number (integer maximum value) of {@link DocumentBuilder} may be created. They are never evicted.
 * 
 * @ author noddoux
 */
public final class DocumentBuilders {

    private static final DocumentBuilderResourcePool documentBuilderPool = new DocumentBuilderResourcePool(1, Integer.MAX_VALUE, PoolPolicy.WAIT);

    /**
     * Take a {@link DocumentBuilder} from the pool
     * 
     * @return one {@link DocumentBuilder}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link DocumentBuilder}
     *             of the pool and is interrupted
     */
    public final static DocumentBuilder takeDocumentBuilder() throws PoolException {
       return documentBuilderPool.take();
    }

    /**
     * Release a {@link DocumentBuilder} to the pool
     * 
     * @param documentBuilder a {@link DocumentBuilder} to release
     */
    public final static void releaseDocumentBuilder(DocumentBuilder documentBuilder) {
        documentBuilderPool.release(documentBuilder);
    }

    /**
     * Create a new {@link Document} (with a {@link DocumentBuilder} of the
     * pool)
     * 
     * @return a new {@link Document}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link DocumentBuilder} of
     *             the pool and is interrupted when the {@link PoolPolicy} is
     *             WAIT or if there is no more available {@link DocumentBuilder}
     *             in the {@link DocumentBuilder} pool when the
     *             {@link PoolPolicy} is REJECT
     */
    public final static Document newDocument() {
        DocumentBuilder documentBuilder = takeDocumentBuilder();
        Document document = documentBuilder.newDocument();
        releaseDocumentBuilder(documentBuilder);
        
        return document;
    }
}
