/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.xml;

import javax.xml.stream.XMLInputFactory;

import com.ebmwebsourcing.easycommons.pooling.GenericResourcePool;
import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;
import com.ebmwebsourcing.easycommons.pooling.ResourceHandler;

/**
 * This class represent a {@link XMLInputFactory} pool, which provided
 * {@link XMLInputFactory} object as resource.
 * 
 * @author Christophe DENEUX - EBMWebSourcing
 */
public class XMLInputFactoryResourcePool {

    /**
     * {@link XMLInputFactory} resource handler to manage
     * {@link XMLInputFactory} life cycle methods
     */
    private class XMLInputFactoryResourceHandler implements ResourceHandler<XMLInputFactory> {

        public XMLInputFactoryResourceHandler() {
        }

        @Override
        public XMLInputFactory create() {
            return XMLInputFactory.newInstance();
        }

        @Override
        public void onRelease(XMLInputFactory xmlInputFactory) {
        }

        @Override
        public void onTake(XMLInputFactory xmlInputFactory) {
        }
    }
    
    private final GenericResourcePool<XMLInputFactory> xmlInputFactoryPool;

    /**
     * Allowing to instantiate a new {@link XMLInputFactoryResourcePool}
     * containing {@link XMLInputFactory} resources.
     * 
     * @param minPoolSize
     *            The minimum number of {@link XMLInputFactory} instances in the
     *            pool (created at the initialization).
     * @param maxPoolSize
     *            the maximum number of {@link XMLInputFactory} instances in the
     *            current pool (limit of the pool). It must be greater or equals
     *            to the specified minSize. The maximum value is
     *            Integer.MAX_VALUE
     * @param poolPolicy
     *            the {@link PoolPolicy} to adopt when the maximum size is
     *            reached. it must not be null.
     */
    public XMLInputFactoryResourcePool(int minPoolSize, int maxPoolSize,
            PoolPolicy poolPolicy) {
        XMLInputFactoryResourceHandler xmlInputFactoryResourceHandler = new XMLInputFactoryResourceHandler();
        this.xmlInputFactoryPool = new GenericResourcePool<XMLInputFactory>(
                xmlInputFactoryResourceHandler, minPoolSize, maxPoolSize, poolPolicy);
    }

    /**
     * Take one unused {@link XMLInputFactory} in the current pool. After
     * getting a {@link XMLInputFactory} from the pool and before returning a
     * {@link XMLInputFactory}, the method onTake() of the
     * {@link XMLInputFactory} resource handler is called.
     * 
     * @return one {@link XMLInputFactory}
     * 
     * @throws PoolException
     *             if the current thread is interrupted for the pool policy WAIT
     *             or if there is no more available resource in the pool for the
     *             pool policy REJECT
     * 
     */
    public XMLInputFactory take() {
        return this.xmlInputFactoryPool.take();
    }

    /**
     * Release the specified {@link XMLInputFactory} After putting back the
     * {@link XMLInputFactory} in the pool, the method onRelease() of the
     * {@link XMLInputFactory} resource handler is called.
     * 
     * @param xmlInputFactory
     *            The {@link XMLInputFactory} to release
     */
    public final void release(final XMLInputFactory xmlInputFactory) {
        this.xmlInputFactoryPool.release(xmlInputFactory);
    }
}
