/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.xml;

import java.io.InputStream;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;

/**
 * A static pool of {@link XMLInputFactory} to improve {@link XMLInputFactory}
 * creation efficiency. A {@link XMLInputFactory} is initialized the first time
 * this class is used. An infinite number (integer maximum value) of
 * {@link XMLInputFactory} may be created. They are never evicted.
 * 
 * @author Christophe DENEUX - EBM WebSourcing
 */
public final class XMLInputFactories {

    private static final XMLInputFactoryResourcePool xmlInputFactoryPool = new XMLInputFactoryResourcePool(
            1, Integer.MAX_VALUE, PoolPolicy.WAIT);

    /**
     * Take a {@link XMLInputFactory} from the pool
     * 
     * @return one {@link XMLInputFactory}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link XMLInputFactory} of
     *             the pool and is interrupted
     */
    public final static XMLInputFactory takeXMLInputFactory() throws PoolException {
        return xmlInputFactoryPool.take();
    }

    /**
     * Release a {@link XMLInputFactory} to the pool
     * 
     * @param xmlInputFactory
     *            a {@link XMLInputFactory} to release
     */
    public final static void releaseXMLInputFactory(final XMLInputFactory xmlInputFactory) {
        xmlInputFactoryPool.release(xmlInputFactory);
    }

    /**
     * Create a {@link XMLStreamReader} from a specified {@link InputStream} by
     * using the pool of {@link XMLInputFactory}
     * 
     * @param InputStream
     *            an {@link InputStream}
     * 
     * @return the {@link XMLStreamReader}
     * 
     * @throws XMLStreamException
     *             if an unexpected processing errors occurs
     */
    public final static XMLStreamReader createXMLStreamReader(final InputStream InputStream)
            throws XMLStreamException {
        XMLInputFactory xmlInputFactory = null;

        try {
            xmlInputFactory = takeXMLInputFactory();
            return xmlInputFactory.createXMLStreamReader(InputStream);
        } finally {
            if(xmlInputFactory != null) {
                releaseXMLInputFactory(xmlInputFactory);
            }
        }
    }
}
