/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.xml;

import java.io.OutputStream;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import com.ebmwebsourcing.easycommons.pooling.PoolException;
import com.ebmwebsourcing.easycommons.pooling.PoolPolicy;

/**
 * A static pool of {@link XMLOutputFactory} to improve {@link XMLOutputFactory} creation efficiency. 
 * A {@link XMLOutputFactory} is initialized the first time this class is used.
 * An infinite number (integer maximum value) of {@link XMLOutputFactory} may be created. They are never evicted.
 * 
 * @author Nicolas Oddoux - EBM WebSourcing
 */
public final class XMLOutputFactories {

    private static final XMLOutputFactoryResourcePool xmlOutputFactoryPool = new XMLOutputFactoryResourcePool(1, Integer.MAX_VALUE, PoolPolicy.WAIT);

    /**
     * Take a {@link XMLOutputFactory} from the pool
     * 
     * @return one {@link XMLOutputFactory}
     * 
     * @throws PoolException
     *             if the current thread waits for a {@link XMLOutputFactory}
     *             of the pool and is interrupted
     */
    public final static XMLOutputFactory takeXMLOutputFactory() throws PoolException {
       return xmlOutputFactoryPool.take();
    }

    /**
     * Release a {@link XMLOutputFactory} to the pool
     * 
     * @param xmlOutputFactory a {@link XMLOutputFactory} to release
     */
    public final static void releaseXMLOutputFactory(XMLOutputFactory xmlOutputFactory) {
        xmlOutputFactoryPool.release(xmlOutputFactory);
    }

    /**
     * Create a {@link XMLStreamWriter} from a specified {@link OutputStream} by
     * using the pool of {@link XMLOutputFactory}
     * 
     * @param outputStream
     *            an {@link OutputStream}
     * 
     * @return the {@link XMLStreamWriter}
     * 
     * @throws XMLStreamException
     *             if an unexpected processing errors occurs
     */
    public final static XMLStreamWriter createXMLStreamWriter(OutputStream outputStream)
            throws XMLStreamException {
        XMLOutputFactory xmlOutputFactory = null;

        try {
            xmlOutputFactory = takeXMLOutputFactory();
            return xmlOutputFactory.createXMLStreamWriter(outputStream);
        } finally {
            if (xmlOutputFactory != null) {
                releaseXMLOutputFactory(xmlOutputFactory);
            }
        }
    }
}
