/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
package com.ebmwebsourcing.easyesb.soa.impl.transport;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.w3c.dom.Document;

import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectValidationException;
import com.ebmwebsourcing.easycommons.research.util.easybox.SOAUtil;
import com.ebmwebsourcing.easyesb.constant.EasyESBFramework;
import com.ebmwebsourcing.easyesb.exchange10.api.element.Exchange;
import com.ebmwebsourcing.easyesb.exchange10.api.type.PatternType;
import com.ebmwebsourcing.easyesb.exchange10.api.type.RoleType;
import com.ebmwebsourcing.easyesb.exchange10.api.type.StatusType;
import com.ebmwebsourcing.easyesb.exchange10.api.util.Util;
import com.ebmwebsourcing.easyesb.soa.api.ESBException;
import com.ebmwebsourcing.easyesb.soa.api.component.Component;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.Endpoint;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.ProviderEndpoint;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.behaviour.EndpointBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.interceptors.ProviderEndpointInvocationInterceptor;
import com.ebmwebsourcing.easyesb.soa.api.node.NodeBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.registry.RegistryEndpointBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.registry.RegistryServiceBehaviour;
import com.ebmwebsourcing.easyesb.soa.api.service.Service;
import com.ebmwebsourcing.easyesb.soa.api.transport.Skeleton;
import com.ebmwebsourcing.easyesb.soa.api.transport.TransportersManager;
import com.ebmwebsourcing.easyesb.soa.api.transport.WakeUpKey;
import com.ebmwebsourcing.easyesb.soa.api.util.MessageUtil;
import com.ebmwebsourcing.easyesb.soa10.api.Constants;
import com.ebmwebsourcing.easyesb.soa10.api.element.SourceNodeInformations;
import com.ebmwebsourcing.easyesb.soa10.api.type.EndpointType;
import com.ebmwebsourcing.easyesb.transporter.api.transport.TransportException;


public class SkeletonImpl implements Skeleton {

	private static Logger log = Logger.getLogger(SkeletonImpl.class.getName());


	private Endpoint<? extends EndpointType> providerEndpoint;


	private boolean takeToSendResponseInCharge = false;

	private List<ProviderEndpointInvocationInterceptor> invocationInterceptors = new ArrayList<ProviderEndpointInvocationInterceptor>();


	public SkeletonImpl(Endpoint<? extends EndpointType> providerEndpoint) {
		this.providerEndpoint = providerEndpoint;
	}

	@SuppressWarnings("unchecked")
	public TransportersManager getTransportersManager() throws ESBException {
		TransportersManager res = null;
		res = this.providerEndpoint.getNode().getTransportersManager();
		return res;
	}

	@SuppressWarnings("unchecked")
	public void accept(Exchange exchange) throws TransportException {
		log.finest("message accepted by endpoint provider " + this.providerEndpoint.getName() + " - Echange id: " + exchange.getUuid());


		//log.finest("exchange:\n" + exchange);
		// Change role
		if(exchange.getRole() == RoleType.CONSUMER) {
			exchange.setRole(RoleType.PROVIDER);
		} else {
			exchange.setRole(RoleType.CONSUMER);
		}

		if((exchange.getStatus().equals(StatusType.DONE)) || (exchange.getStatus().equals(StatusType.FAULT))) {
			log.finest("exchange finished: Status = " + exchange.getStatus() + " - Role = " + exchange.getRole());

			boolean found = false;
			if((exchange.getMessageError() != null && exchange.getMessageError().getBody() != null && exchange.getMessageError().getBody().getPayload() != null) || (exchange.getMessageOut() != null && exchange.getMessageOut().getBody() != null && exchange.getMessageOut().getBody().getPayload() != null))  {
				// try to wake up the stub if it is wait in sendSync
				try {
					Map<UUID, WakeUpKey> stub2awake = this.providerEndpoint.getNode().getTransportersManager().getStub2awake();
					while(stub2awake.containsKey(UUID.fromString(exchange.getUuid()))) {
						WakeUpKey locked = stub2awake.remove(UUID.fromString(exchange.getUuid()));
						synchronized(locked) {
							log.finest("wake thread => notify");
							locked.setExchange(exchange);
							locked.notifyAll();
							found = true;
						}
					}
				} catch (ESBException e) {
					throw new TransportException(e);
				}
				if(! found) {
					// rechange role
					if(exchange.getRole() == RoleType.CONSUMER) {
						exchange.setRole(RoleType.PROVIDER);
					} else {
						exchange.setRole(RoleType.CONSUMER);
					}

					try {
						this.providerEndpoint.getNode().getTransportersManager().push(exchange, this.providerEndpoint.getNode().getQName());
					} catch (ESBException e) {
						throw new TransportException(e);
					}
				}
			}
		} else {

			if(providerEndpoint instanceof ProviderEndpoint || 
					providerEndpoint instanceof Service<?> ||
					providerEndpoint instanceof Component<?>) {

				// realize pre-processing before receiving message (t2)
				for(ProviderEndpointInvocationInterceptor peii: this.invocationInterceptors) {
					peii.processingExchangeAfterReceiving(exchange);
				}

				if(exchange.getPattern().equals(PatternType.IN_ONLY)) {

					// set status if not already done
					exchange.setStatus(StatusType.DONE);



					// Send the status
					try {
						QName sourceNodeEndpointName = getSourceNodeEndpointName(exchange);
						this.providerEndpoint.getNode().getTransportersManager().push(exchange, sourceNodeEndpointName);
					} catch (ESBException e) {
						throw new TransportException(e);
					}
				} 

				try {
					boolean found = false;

					// execute behaviour of endpoint
					if(this.providerEndpoint.getBehaviourClasses() != null) {
						log.finest("execute behaviours : " + this.providerEndpoint.getBehaviourClasses());
						for(EndpointBehaviour behaviour: this.providerEndpoint.getBehaviours()) {
							behaviour.execute(exchange);
							if(hasBeenProcessed(exchange)) {
								found = true;
								break;
							}
						}
					} else {
						log.finest("execute empty behaviour");
					}

					if(!found && !exchange.getPattern().equals(PatternType.IN_ONLY)) {
						throw new ESBException("Impossible to find operation: " + exchange.getOperation() + " on endpoint " + this.providerEndpoint.getQName());
					}

				} catch (IllegalArgumentException e) {
					throw new TransportException(e);
				} catch (ESBException e) {
					throw new TransportException(e);
				} catch (ClassNotFoundException e) {
					throw new TransportException(e);
				} 

				// realize post-processing after receiving message (t3)
				for(ProviderEndpointInvocationInterceptor peii: this.invocationInterceptors) {
					peii.processingExchangeBeforeSending(exchange);
				}
			}

			if(exchange.getPattern().equals(PatternType.IN_OUT) && exchange.getRole() == RoleType.PROVIDER && this.takeToSendResponseInCharge == false) {	
				sendResponseToClient(exchange);
			}
		}
	}

	private boolean hasBeenProcessed(Exchange exchange) {
		if((exchange.getMessageOut() != null && exchange.getMessageOut().getBody() != null) || (exchange.getMessageError() != null && exchange.getMessageError().getBody() != null)
				|| (exchange.getMessageIn().getHeader() != null && exchange.getMessageIn().getHeader().getProperty(MessageUtil.EXCHANGE_ACCEPTED_BY_PROVIDER_PROPERTY) != null)) {
			exchange.getMessageIn().getHeader().removeProperty(MessageUtil.EXCHANGE_ACCEPTED_BY_PROVIDER_PROPERTY);
			return true;
		}
		return false;
	}

	public synchronized void sendResponseToClient(Exchange exchange)
	throws TransportException {

		// realize pre-processing before sending message
//		for(ProviderEndpointInvocationInterceptor peii: this.invocationInterceptors) {
//			peii.preProcessingExchangeBeforeSending(exchange);
//		}

		QName sourceNodeEndpointName = getSourceNodeEndpointName(exchange);
		if(sourceNodeEndpointName == null) {
		    throw new TransportException("Impossible to find source node from exchange: " + exchange);
		}


		// Send the response
		log.finest("*** send response by the provider " + exchange.getDestination() + " to source " + sourceNodeEndpointName);

		if(exchange.getPattern().equals(PatternType.IN_OUT)) {

			// set status if not already done
			exchange.setStatus(StatusType.DONE);
			if(exchange.getMessageError() != null && exchange.getMessageError().getBody() != null && exchange.getMessageError().getBody().getPayload() != null) {
				exchange.setStatus(StatusType.FAULT);
			}

			try {
				if(this.providerEndpoint == null) {
				    throw new TransportException("the provider endpoint cannot be null!!!");
				}
				if(this.providerEndpoint.getNode() == null) {
                    throw new TransportException("Impossible to find node of endpoint: " + this.providerEndpoint.getName());
                }
				if(this.providerEndpoint.getNode().getTransportersManager() == null) {
                    throw new TransportException("Impossible to find transporterManager on node: " + this.providerEndpoint.getNode().getName());
                }
			    this.providerEndpoint.getNode().getTransportersManager().push(exchange, sourceNodeEndpointName);
			} catch (ESBException e) {
				throw new TransportException(e);
			}
		}

		// realize post-processing after sending message
//		for(ProviderEndpointInvocationInterceptor peii: this.invocationInterceptors) {
//			peii.postProcessingExchangeAfterSending(exchange);
//		}
	}

	private QName getSourceNodeEndpointName(Exchange exchange)
	throws TransportException {
		// find node of endpoint
		QName sourceNodeEndpointName = null;
		SourceNodeInformations sourceNodeInfos = null;

		//// find in property of exchange
		Document sourceNodeDoc = null;
		if(exchange.getMessageIn().getHeader() != null) {
			sourceNodeDoc = exchange.getMessageIn().getHeader().getProperty(new QName(Constants.SOA_DATAMODEL_NS_URI, "sourceNodeInformations"));
			if(sourceNodeDoc != null) {
				try {
					InputStream is = Util.convertDocumentToInputStream(sourceNodeDoc);
					sourceNodeInfos = SOAUtil.getInstance().getReader(EasyESBFramework.getInstance()).get().readFragment(is, SourceNodeInformations.class);
					is.close();
				} catch (XmlObjectValidationException e) {
					throw new TransportException(e);
				} catch (XmlObjectReadException e) {
					throw new TransportException(e);
				} catch (IOException e) {
					throw new TransportException(e);
				}
			} 
			if(sourceNodeInfos != null) {
				sourceNodeEndpointName = sourceNodeInfos.getNodeName();
			}
		} else {
			// find in registry
			try {
				EndpointType ep = ((RegistryEndpointBehaviour)((RegistryServiceBehaviour)((NodeBehaviour)this.providerEndpoint.getNode().findBehaviour(NodeBehaviour.class)).getRegistryService().findBehaviour(RegistryServiceBehaviour.class)).getRegistryEndpoint().findBehaviour(RegistryEndpointBehaviour.class)).getEndpoint(exchange.getSource());
				if(ep != null) {
					sourceNodeEndpointName = ep.getNode();
				}
			} catch (ESBException e) {
				throw new TransportException("Impossible to find the node corresponding to this source: " + exchange.getSource(), e);
			}
			if(sourceNodeEndpointName == null) {
				throw new TransportException("Impossible to find the node corresponding to this source: " + exchange.getSource());
			}
		}
		return sourceNodeEndpointName;
	}




	public boolean getTakeToSendResponseInCharge() {
		return takeToSendResponseInCharge;
	}

	public void setTakeToSendResponseInCharge(boolean takeToSendResponseInCharge) {
		this.takeToSendResponseInCharge = takeToSendResponseInCharge;
	}

	public void addProviderEndpointInvocationInterceptor(
			ProviderEndpointInvocationInterceptor i) {
		this.invocationInterceptors.add(i);
	}

	public ProviderEndpointInvocationInterceptor removeProviderEndpointInvocationInterceptor(
			ProviderEndpointInvocationInterceptor i) {
		return (this.invocationInterceptors.remove(i) ? i: null);
	}

	public List<ProviderEndpointInvocationInterceptor> getProviderEndpointInvocationInterceptor() {
		return this.invocationInterceptors;
	}



}
