package com.ebmwebsourcing.easyesb.cli.impl;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.StringTokenizer;
import java.util.TreeMap;

import com.ebmwebsourcing.easyesb.cli.api.Command;
import com.ebmwebsourcing.easyesb.soa.api.ESBException;



public abstract class AbstractCommandManager {

	protected File currentPath;

	protected SortedMap<String, Command> map = new TreeMap<String, Command>();

	protected String connectionAddress;

	private List<String> history;

	/**
	 * 
	 * @param node 
	 * @param petalsService
	 * @param petalsServer
	 * @throws ESBException 
	 * @throws ConnectionErrorException
	 * @throws PetalsAdminServiceErrorException
	 * @throws PetalsAdminDoesNotExistException
	 */
	public AbstractCommandManager() throws ESBException {

		this.history = new ArrayList<String>();

		init(null);

	}

	/**
	 * 
	 * @param command
	 */
	protected void addCommand(Command command) {
		map.put(command.getShortcut(), command);
	}

	public Command findCommand(String shortcut) {
		return map.get(shortcut);
	}

	/**
	 * Read command line. This method ends with an "exit" command line
	 * 
	 */
	public void read() {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));


		int code = 1;
		while (code > 0) {
			try {
				System.out.println();
				System.out.print("cli:/> ");
				code = processCommandLine(br.readLine());
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Process the command, and return a code if "exit" command, return -1
	 * 
	 * @param command
	 * @return
	 */
	protected int processCommandLine(String command) {
		int result = AbstractCommand.OK_CODE;
		if (command != null && command.length() > 0) {
			StringTokenizer tokenizer = new StringTokenizer(command, " ");
			List<String> args = new ArrayList<String>();
			while (tokenizer.hasMoreTokens()) {
				args.add(tokenizer.nextToken());
			}

			String key = args.get(0);

			// get the key for the requested task
			Command task = getTask(key);
			if (task == null) {
				System.out.println("Unrecognized command : " + command + " (tape '-h' for help)");
			} else {

				// get the command arguments
				List<String> arguments = null;
				if (args.size() > 1) {
					arguments = args.subList(1, args.size());
				} else {
					arguments = new ArrayList<String>(0);
				}
				result = task.process(arguments);
				if (result == AbstractCommand.INVALID_ARGS) {
					System.out.println("Bad arguments : " + arguments + " (tape '-h' for help)");
				}else if(result == AbstractCommand.NO_CONNECTION){
					System.out.println("No connection ... Connect first to a esb node with Connect (-c) command.");
				}else if(result == AbstractCommand.ERROR_CODE){
					System.out.println("Error while executing command "+ command);
				}else{
					this.getHistory().add(command);

				}
				result = AbstractCommand.OK_CODE;
			}

		}
		return result;
	}


	public List<String> getHistory() {
		return this.history;
	}
	public void resetHistory(){
		this.history.clear();
	}


	/**
	 * Print the available commands and their options
	 * 
	 */
	public void printUsage() {
		Iterator<String> iter = map.keySet().iterator();
		while (iter.hasNext()) {
			Command command = map.get(iter.next());
			System.out.println(" -" + command.toString());
		}
	}

	/**
	 * Return the task from its shortcut or name.
	 * 
	 * @param command
	 * @return null if no task has been found
	 */
	private Command getTask(String command) {
		Command task = map.get(command);

		if (task == null) {
			// try to get shortcut from command
			String shortcut = getShortcutFromCommand(command);
			if (shortcut != null) {
				task = map.get(shortcut);
			}
		}
		return task;
	}

	/**
	 * 
	 * @param command
	 * @return
	 */
	private String getShortcutFromCommand(String command) {
		String result = null;
		boolean found = false;

		Iterator<String> iter = map.keySet().iterator();
		while (iter.hasNext() && !found) {
			Command task = map.get(iter.next());
			if (command.equals(task.getName())) {
				result = task.getShortcut();
				found = true;
			}
		}
		return result;
	}

	public String getConnectionAddress() {
		return connectionAddress;
	}

	public abstract void init(String host) throws ESBException;


	public List<String> detectCommand(List<String> command) {
		List<String> res = new ArrayList<String>();
		String buff = "";
		for(String c: command) {
			if(c != null) {
				buff = buff + " " + c;
			}
		}
		List<String> tp = Arrays.asList(buff.split(" -"));

		// clean
		for(String c: tp) {
			if(c != null && c.trim().length() > 0) {
				res.add(c);
			}
		}

		System.out.println("commands : "+res);
		return res;
	}

}
