/****************************************************************************
 *
 * Copyright (c) 2010-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package org.petalslink.abslayer.service.impl.wsdl11;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;

import org.petalslink.abslayer.Factory;
import org.petalslink.abslayer.service.api.BindingOperation;
import org.petalslink.abslayer.service.api.Description;
import org.petalslink.abslayer.service.api.Endpoint;
import org.petalslink.abslayer.service.api.Interface;
import org.petalslink.abslayer.service.api.Message;
import org.petalslink.abslayer.service.api.Property;
import org.petalslink.abslayer.service.api.PropertyAlias;
import org.petalslink.abslayer.service.api.Types;
import org.w3c.dom.Document;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectWriteException;
import com.ebmwebsourcing.easybox.api.XmlObjectWriter;
import com.ebmwebsourcing.easybox.api.XmlObjectXPathEvaluator;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easyschema10.api.SchemaHelper;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easywsdl11.api.element.Binding;
import com.ebmwebsourcing.easywsdl11.api.element.Definitions;
import com.ebmwebsourcing.easywsdl11.api.element.Import;
import com.ebmwebsourcing.easywsdl11.api.element.Port;
import com.ebmwebsourcing.easywsdl11.api.element.PortType;
import com.ebmwebsourcing.easywsdl11.api.element.Service;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
public class DescriptionImpl implements Description {

    private DocumentBuilderFactory domBuilder = null;
    private static XmlContext xmlContext = new XmlContextFactory().newContext();


    private ThreadLocal<XmlObjectWriter> xmlwriter = null;

    private final Definitions model;

    private final Map<String, Schema> importedSchemaCache = new HashMap<String, Schema>();
    private final Map<String, Description> importedDescriptionCache = new HashMap<String, Description>();
    private final Map<QName, Message> messageCache = new HashMap<QName, Message>();
    private final Map<QName, Interface> interfaceCache = new HashMap<QName, Interface>();
    private final Map<QName, Property> propertyCache = new HashMap<QName, Property>();
    private final Map<QName, List<Endpoint>> endpointsCache = new HashMap<QName, List<Endpoint>>();
    private final Map<QName, List<Element>> elementsCache = new HashMap<QName, List<Element>>();
    private final Map<QName, List<PropertyAlias>> propertyAliasesCache = new HashMap<QName, List<PropertyAlias>>();


    private DescriptionImpl(Definitions model) {
        this.model = model;

        domBuilder = DocumentBuilderFactory.newInstance();
        domBuilder.setNamespaceAware(true);
        xmlwriter = new ThreadLocal<XmlObjectWriter>() {
            protected XmlObjectWriter initialValue() {
                return xmlContext.createWriter();
            }
        };
    }

    @Override
    public Map<String, String> getNamespaces() {
        return model.getXmlObjectInScopeNamespaces();
    }

    @Override
    public URI getDocumentBaseURI() {
        return model.getXmlObjectBaseURI();
    }

    @Override
    public String getTargetNamespace() {
        return model.getTargetNamespace();
    }


    @Override
    public Types getTypes() {
        return (org.petalslink.abslayer.service.api.Types) Factory.getInstance().wrap(model.getTypes());
    }



    @Override
    public org.petalslink.abslayer.service.api.PartnerLinkType getPartnerLinkType(
            QName qname) {
        if (!model.getTargetNamespace().equals(qname.getNamespaceURI()))
            return null;
        for (com.ebmwebsourcing.easyplnk20.api.element.PartnerLinkType plt : model
                .getAnyXmlObjects(com.ebmwebsourcing.easyplnk20.api.element.PartnerLinkType.class)) {
            if (qname.getLocalPart().equals(plt.getName()))
                return (org.petalslink.abslayer.service.api.PartnerLinkType) Factory.getInstance().wrap(plt);
        }
        return null;
    }


    @Override
    public Message findMessage(QName qname) {
        assert qname != null;
        if (messageCache.containsKey(qname)) return messageCache.get(qname);

        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            com.ebmwebsourcing.easywsdl11.api.element.Message result = xpathEvaluator
                    .selectSingleXmlObjectNode(
                            model,
                            String.format(
                                    "//wsdl11:message[@name='%s'][parent::wsdl11:definitions/@targetNamespace='%s']",
                                    qname.getLocalPart(),
                                    qname.getNamespaceURI()),
                                    com.ebmwebsourcing.easywsdl11.api.element.Message.class);
            Message message = (result == null) ? null :
                (org.petalslink.abslayer.service.api.Message) Factory.getInstance().wrap(result);
            messageCache.put(qname, message);
            return message;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }



    @Override
    public Interface findInterface(QName qname) {
        assert qname != null;
        if (interfaceCache.containsKey(qname)) return interfaceCache.get(qname);

        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            PortType result = xpathEvaluator
                    .selectSingleXmlObjectNode(
                            model,
                            String.format(
                                    "//wsdl11:portType[@name='%s'][parent::wsdl11:definitions/@targetNamespace='%s']",
                                    qname.getLocalPart(),
                                    qname.getNamespaceURI()), PortType.class);
            Interface interfac = (result == null) ? null : 
                (org.petalslink.abslayer.service.api.Interface) Factory.getInstance().wrap(result);
            interfaceCache.put(qname, interfac);
            return interfac;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }

    @Override
    public List<org.petalslink.abslayer.service.api.Service> findServicesImplementingInterface(Interface itf) {
        List<org.petalslink.abslayer.service.api.Service> services = new ArrayList<org.petalslink.abslayer.service.api.Service>();
        for(Endpoint ep: findEndpointsImplementingInterface(itf)) {
            if(!services.contains(ep.getService())) {
                services.add(ep.getService());
            }
        }

        return services;
    }

    @Override
    public List<Endpoint> findEndpointsImplementingInterface(Interface itf) {
        assert itf != null;
        QName itfQname = itf.getQName();
        if (endpointsCache.containsKey(itfQname)) return endpointsCache.get(itfQname);

        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        List<Endpoint> endpoints = new ArrayList<Endpoint>();
        try {
            String queryStr = String.format(
                    "//wsdl11:binding[@type = QName('%s', '%s')]", itf
                    .getQName().getNamespaceURI(), itf.getQName()
                    .getLocalPart());
            Binding[] bindings = xpathEvaluator.selectXmlObjectNodes(model,
                    queryStr, Binding.class);

            for (Binding binding : bindings) {
                queryStr = String.format(
                        "//wsdl11:port[@binding = QName('%s', '%s')]",
                        ((Definitions) binding.getXmlObjectBaseRoot())
                        .getTargetNamespace(), binding.getName());
                Port[] ports = xpathEvaluator.selectXmlObjectNodes(model,
                        queryStr, Port.class);
                if (ports == null)
                    continue;
                for (Port p : ports) {
                    endpoints.add((org.petalslink.abslayer.service.api.Endpoint) Factory.getInstance().wrap(p));
                }
            }
            endpointsCache.put(itfQname, endpoints);
            return endpoints;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }

    }



    @Override
    public Endpoint findEndpoint(String endpointName) {
        assert endpointName != null;

        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            Port result = xpathEvaluator.selectSingleXmlObjectNode(model,
                    String.format("//wsdl11:port[@name='%s']", endpointName),
                    Port.class);
            if (result == null)
                return null;
            return (org.petalslink.abslayer.service.api.Endpoint) Factory.getInstance().wrap(result);
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }



    @Override
    public org.petalslink.abslayer.service.api.Service findService(QName qname) {
        assert qname != null;
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            Service result = xpathEvaluator
                    .selectSingleXmlObjectNode(
                            model,
                            String.format(
                                    "//wsdl11:service[@name='%s'][parent::wsdl11:definitions/@targetNamespace='%s']",
                                    qname.getLocalPart(),
                                    qname.getNamespaceURI()), Service.class);
            if (result == null)
                return null;
            return (org.petalslink.abslayer.service.api.Service) Factory.getInstance().wrap(result);
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }

    @Override
    public org.petalslink.abslayer.service.api.Binding findBinding(QName qname) {
        assert qname != null;
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            Binding result = xpathEvaluator
                    .selectSingleXmlObjectNode(
                            model,
                            String.format(
                                    "//wsdl11:binding[@name='%s'][parent::wsdl11:definitions/@targetNamespace='%s']",
                                    qname.getLocalPart(),
                                    qname.getNamespaceURI()), Binding.class);
            if (result == null)
                return null;
            return (org.petalslink.abslayer.service.api.Binding) Factory.getInstance().wrap(result);
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }

    @Override
    public XmlObject getModel() {
        return model;
    }



    @Override
    public List<Element> findElementsInAllSchema(QName qname) {
        if (elementsCache.containsKey(qname)) return elementsCache.get(qname);
        List<Element> elementsList = new ArrayList<Element>();
        assert qname != null;
        Element element = SchemaHelper.findElementByQName(model, qname);
        if (element != null) {
            elementsList.add(element);
        }
        elementsCache.put(qname, elementsList);
        return elementsList;
    }



    @Override
    public Property findProperty(QName qname) {
        assert qname != null;
        if (propertyCache.containsKey(qname)) return propertyCache.get(qname);

        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            com.ebmwebsourcing.easyvprop20.api.anonymoustype.Property result = xpathEvaluator
                    .selectSingleXmlObjectNode(
                            model,
                            String.format(
                                    "//vprop:property[@name='%s'][parent::wsdl11:definitions/@targetNamespace='%s']",
                                    qname.getLocalPart(),
                                    qname.getNamespaceURI()),
                                    com.ebmwebsourcing.easyvprop20.api.anonymoustype.Property.class);
            Property property = (result == null) ? null : 
                (org.petalslink.abslayer.service.api.Property) Factory.getInstance().wrap(result);
            propertyCache.put(qname, property);
            return property;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }



    @Override
    public Collection<? extends PropertyAlias> getPropertyAliases4ThisProperty(
            QName propName) {
        assert propName != null;
        if (propertyAliasesCache.containsKey(propName)) return propertyAliasesCache.get(propName);
        List<PropertyAlias> propertyAliases = new ArrayList<PropertyAlias>();
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            com.ebmwebsourcing.easyvprop20.api.anonymoustype.PropertyAlias[] result = xpathEvaluator
                    .selectXmlObjectNodes(
                            model,
                            String.format(
                                    "//vprop:propertyAlias[namespace-uri-from-QName(@propertyName) = '%s'][local-name-from-QName(@propertyName) = '%s']",
                                    propName.getNamespaceURI(),
                                    propName.getLocalPart()),
                                    com.ebmwebsourcing.easyvprop20.api.anonymoustype.PropertyAlias.class);
            if (result != null) {
                for (com.ebmwebsourcing.easyvprop20.api.anonymoustype.PropertyAlias propertyAlias : result) {
                    propertyAliases
                    .add((org.petalslink.abslayer.service.api.PropertyAlias) Factory.getInstance().wrap(propertyAlias));
                }
            }
            propertyAliasesCache.put(propName, propertyAliases);
            return propertyAliases;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }


    @Override
    public Collection<? extends PropertyAlias> getPropertyAliases() {
        List<PropertyAlias> propertyAliases = new ArrayList<PropertyAlias>();
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            com.ebmwebsourcing.easyvprop20.api.anonymoustype.PropertyAlias[] result = xpathEvaluator
                    .selectXmlObjectNodes(
                            model,
                            String.format(
                                    "//vprop:propertyAlias"),
                                    com.ebmwebsourcing.easyvprop20.api.anonymoustype.PropertyAlias.class);
            if (result != null) {
                for (com.ebmwebsourcing.easyvprop20.api.anonymoustype.PropertyAlias propertyAlias : result) {
                    propertyAliases
                    .add((org.petalslink.abslayer.service.api.PropertyAlias) Factory.getInstance().wrap(propertyAlias));
                }
            }
            return propertyAliases;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }
    
    @Override
    public List<Interface> getInterfaces() {
        List<org.petalslink.abslayer.service.api.Interface> interfaces = new ArrayList<org.petalslink.abslayer.service.api.Interface>();
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            PortType[] result = xpathEvaluator.selectXmlObjectNodes(model,
                    "//wsdl11:portType", PortType.class);

            if (result == null)
                return interfaces;
            for (PortType binding : result) {
                interfaces.add((org.petalslink.abslayer.service.api.Interface) Factory.getInstance().wrap(binding));
            }
            return interfaces;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }

    @Override
    public List<org.petalslink.abslayer.service.api.Binding> getBindings() {

        List<org.petalslink.abslayer.service.api.Binding> bindings = new ArrayList<org.petalslink.abslayer.service.api.Binding>();
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            Binding[] result = xpathEvaluator.selectXmlObjectNodes(model,
                    "//wsdl11:binding", Binding.class);

            if (result == null)
                return bindings;
            for (Binding binding : result) {
                bindings.add((org.petalslink.abslayer.service.api.Binding) Factory.getInstance().wrap(binding));
            }
            return bindings;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }


    @Override
    public List<org.petalslink.abslayer.service.api.Service> getServices() {
        List<org.petalslink.abslayer.service.api.Service> services = new ArrayList<org.petalslink.abslayer.service.api.Service>();
        for (Service service : model.getServices()) {
            services.add((org.petalslink.abslayer.service.api.Service) Factory.getInstance().wrap(service));
        }
        return services;
    }

    @Override
    public List<org.petalslink.abslayer.service.api.Import> getImports() {
        List<org.petalslink.abslayer.service.api.Import> imports = new ArrayList<org.petalslink.abslayer.service.api.Import>();
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            com.ebmwebsourcing.easywsdl11.api.element.Import[] result = xpathEvaluator.selectXmlObjectNodes(model,
                    "//wsdl11:import", com.ebmwebsourcing.easywsdl11.api.element.Import.class);

            if (result == null)
                return imports;
            for (com.ebmwebsourcing.easywsdl11.api.element.Import impt : result) {
                imports.add((org.petalslink.abslayer.service.api.Import) Factory.getInstance().wrap(impt));
            }
            return imports;
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }
    }

    @Override
    public Document write() throws XmlObjectWriteException {
        Document doc;
        try {
            doc = this.domBuilder.newDocumentBuilder().newDocument();
            this.xmlwriter.get().writeDocument(this.model, doc);
        } catch (ParserConfigurationException e) {
            throw new UncheckedException(e.getMessage(), e);
        } 
        return doc;
    }
    
    public static XmlContext getXmlContext() {
        return xmlContext;
    }

    public static void setXmlContext(XmlContext xmlContext) {
        DescriptionImpl.xmlContext = xmlContext;
    }

    @Override
    public String toString() {
        return "DescriptionImpl [model=" + model + "]";
    }

    @Override
    public BindingOperation findBindingOperation(QName qname) {
        assert qname != null;
        XmlObjectXPathEvaluator xpathEvaluator = model.getXmlContext()
                .createXPathEvaluator();
        try {
            com.ebmwebsourcing.easywsdl11.api.element.BindingOperation result = xpathEvaluator
                    .selectSingleXmlObjectNode(
                            model,
                            String.format(
                                    "//wsdl11:binding[parent::wsdl11:definitions/@targetNamespace='%s']/wsdl11:operation[@name='%s']",
                                    qname.getNamespaceURI(), 
                                    qname.getLocalPart()),
                                    com.ebmwebsourcing.easywsdl11.api.element.BindingOperation.class);
            if (result == null)
                return null;
            return (org.petalslink.abslayer.service.api.BindingOperation) Factory.getInstance().wrap(result);
        } catch (XPathExpressionException xee) {
            throw new UncheckedException(xee);
        }    }

}
