/****************************************************************************
 *
 * Copyright (c) 2010-2012, EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA 
 *
 *****************************************************************************/

package org.petalslink.abslayer;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.ServiceLoader;

import com.ebmwebsourcing.easybox.api.ClassMetadataConstants;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author Nicolas Salatge - EBM WebSourcing
 */
public final class Factory {

    private static final Factory INSTANCE;
    
    private final Map<Class<?>, Constructor<?>> constructorsFromModelMap; // model class => constructor
    
    
    static {
        ServiceLoader<FactoryProvider> sl = ServiceLoader.load(FactoryProvider.class);
        INSTANCE = new Factory(sl);
    }
    
    Factory(Iterable<FactoryProvider> providers) {
        this.constructorsFromModelMap = new HashMap<Class<?>, Constructor<?>>();
        Iterator<FactoryProvider> providersIt = providers.iterator();
        while (providersIt.hasNext()) {
            FactoryProvider provider = providersIt.next();
            constructorsFromModelMap.putAll(provider.getConstructorsFromModelMap());
        }
    }
    
    public static Factory getInstance()  {
        return INSTANCE;
    }
    
    
    public Object wrap(XmlObject obj) {
        if (obj.hasUserData()) {
            return obj.getUserData();
        }
        Class<?> xmlObjectInterfaceClass = 
            obj.getXmlContext().getClassMetadata().get(obj.getClass(), ClassMetadataConstants.IMPLEMENTATION_CLASS_INTERFACE_CLASS);
        assert xmlObjectInterfaceClass != null;
        Constructor<?> constructor = constructorsFromModelMap.get(xmlObjectInterfaceClass);
        assert constructor != null : String.format("Cannot wrap instances of class '%s'.", xmlObjectInterfaceClass);
        try {
            Object o = constructor.newInstance(obj);
            obj.setUserData(o);
            return o;
        } catch (IllegalArgumentException e) {
            throw new UncheckedException(e);
        } catch (InstantiationException e) {
            throw new UncheckedException(e);
        } catch (IllegalAccessException e) {
            throw new UncheckedException(e);
        } catch (InvocationTargetException e) {
            throw new UncheckedException(e);
        }
    }
    
}
