/**
 * Raphael Diagram - A crossbrowser SVG/VML library creating diagrams - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.gwt.raphael.client.diagram.element;



import com.ebmwebsourcing.gwt.raphael.client.core.SVGElement;
import com.ebmwebsourcing.gwt.raphael.client.diagram.element.DiagramElementChildData;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementMouseListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementSortableListener;



/**
 * Sortable elements acts as complex element, the main difference is that <br>
 * the elements inside a sortable element can be dragged only inside the main container <br>
 * in order to sort them. <br> <br> 
 * 
 * <b>NOTE:</b> Elements inside a sortable element shouldn't <br> 
 * be draggable otherwise it could lead to some conflicts. 
 * 
 * @author nfleury
 *
 */
public class DiagramSortableElement extends DiagramComplexElement{

	public DiagramSortableElement(SVGElement svgElement) {
		super(svgElement);
		
		this.isSortable(true);
		
	}
	
	/**
	 * Adds a child to the child list without <br>
	 * adding the svg and html views as it is automatically done <br> 
	 * by the sortable mechanism. <br>
	 * 
	 * In other words the child already exists and have already been <br>
	 * positioned in the parent element. 
	 * 
	 */
	private void addChild(DiagramElement diagramElement){
		
		DiagramElementChildData data = new DiagramElementChildData(diagramElement,diagramElement.getAbsoluteLeft(),diagramElement.getAbsoluteTop(),true);
		
		this.getDiagramElementChildren().add(data);

	
	}
	
	
	
	/**
	 * Adds a  child to current element and automatically <br>
	 * positions the element according to the children already in current element <br> 
	 * 
	 * @param diagramElement
	 */
	public void addDiagramElement(DiagramElement diagramElement){
		
		super.addDiagramElement(diagramElement,0,getChildrenTotalHeight(),true);
		
		if (getDiagramPanel()!=null){
			diagramElement.getJqueryObject().css("top","0");
			diagramElement.refreshSVGPosition();
		}
		
	}

	
	public int getChildrenTotalHeight(){
		
		int result = 0;
		
		for(DiagramElementChildData data:this.getDiagramElementChildren()){
			
		
			
			result = result + data.getDiagramElement().getOffsetHeight();
			
		}

		return result;		
	}
	
	
	/**
	 * Connect current sortable element with 
	 * another sorted list
	 */
	public void connectWith(String selector){
		this.getJqueryObject().sortableConnectWith(selector);
	}
	
	
	@Override
	protected void attachDefaultListeners() {
		super.attachDefaultListeners();
		
		
	this.addSortableListerner(new DiagramElementSortableListener(){

		public void onActivate(DiagramElement sortableElement) {
				
			refreshSVGPosition();
			
		}

		public void onBeforeStop(DiagramElement sortableElement) {
			
			refreshSVGPosition();
			
		}

		public void onChange(DiagramElement sortableElement) {
			
			refreshSVGPosition();
			
		}

		public void onDeactivate(DiagramElement sortableElement) {
			
			refreshSVGPosition();
			
		}

		public void onOut(DiagramElement sortableElement, DiagramElement outElement) {
			
			refreshSVGPosition();
			
		}

		public void onOver(DiagramElement sortableElement, DiagramElement overElement) {
			
			refreshSVGPosition();
			
		}

		public void onReceive(DiagramElement sortableElement, DiagramElement receivedElement) {
			
			refreshSVGPosition();
			
			//Add the child to the sortable
			final DiagramSortableElement sortable = (DiagramSortableElement) sortableElement;
			
			sortable.addChild(receivedElement);
			
			//Prevent bubbling on new parent
			receivedElement.addMouseListener(new DiagramElementMouseListenerAdapter(){
				@Override
				public void onMouseEnter() {
					
					sortable.getMouseListeners().remove(sortable.getDefaultMouseListener());
					
				}
				
				@Override
				public void onMouseLeave() {
					
					sortable.addMouseListener(sortable.getDefaultMouseListener());
				
				}
				
				
				
			});

			
		}

		public void onRemove(DiagramElement sortableElement, DiagramElement removedElement) {

			refreshSVGPosition();
			
			//Remove the child from the element
			DiagramSortableElement sortable = (DiagramSortableElement) sortableElement;
			
			sortable.removeChild(removedElement);
			


			
		}

		public void onSort(DiagramElement sortableElement) {
		
			refreshSVGPosition();
			

			
			
		}

		public void onStart(DiagramElement sortableElement) {
			
			refreshSVGPosition();
			
		}

		public void onStop(DiagramElement sortableElement) {
			
			refreshSVGPosition();
			
		}

		public void onUpdate(DiagramElement sortableElement) {
			
			refreshSVGPosition();
			

			
		}
			

		
			
			
		});
		
		
	}
	

}
