/**
 * BPMN Editor Collaborative client - A collaboration platform client for the BPMN Editor - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmneditor.collaboration.comet.server;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.ChangeDrawingPanelEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.ClientLoginEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.CloseDrawingPanelEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.InitializationEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.LockEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.ModificationEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.NewCollaborationEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.NewDrawingPanelEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.ServiceEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.UnlockEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.UserJoinedCollaborationEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.event.UserLeftCollaborationEvent;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.service.DeltaService;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.to.CollaborationStatus;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.to.cometLogEntry.AddElementCometLogEntry;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.to.cometLogEntry.CometLogEntry;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.user.Collaboration;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.user.User;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.user.UserGroup;
import com.ebmwebsourcing.bpmneditor.collaboration.comet.client.user.mock.MockUserDB;
import com.ebmwebsourcing.bpmneditor.presentation.gwt.client.bpmneditor.bpmn2.renderer.DefinitionsSyntaxModel;

import de.novanic.eventservice.client.event.domain.Domain;
import de.novanic.eventservice.client.event.domain.DomainFactory;
import de.novanic.eventservice.service.RemoteEventServiceServlet;
import de.novanic.eventservice.service.registry.user.UserManagerFactory;

public class DeltaServiceImpl extends RemoteEventServiceServlet implements DeltaService {
	
	private Map<User,Collaboration> presentUsers;
	private List<Collaboration> collaborations;
	private MockUserDB userDB;
	
	private Domain clientManagementDomain;
	
	
	public DeltaServiceImpl(){
		presentUsers = new HashMap<User, Collaboration>();
		collaborations =  new ArrayList<Collaboration>();
		userDB = new MockUserDB();
		
		clientManagementDomain = DomainFactory.getDomain(ServiceEvent.clientManagementDomain);
	}
	
	
	
	public void joinCollaboration(User user, String name){
		Collaboration c = findCollaborationByName(name);
		c.addParticipant(user);
		presentUsers.put(user, c);
		
		//event to inform everybody that a new participant has arrived and the id of the actual master on this domain 
		addEvent(DomainFactory.getDomain(c.getDomainName())
				, new UserJoinedCollaborationEvent(c, user));
		
		addEvent(clientManagementDomain,new UserJoinedCollaborationEvent(c, user));
	}
	
	

	public void pushCurrentState(User master, ArrayList<DefinitionsSyntaxModel> state) {
		Collaboration c = findCollaborationForUser(master);
		
		System.out.println(c);
		
		addEvent(DomainFactory.getDomain(c.getDomainName()),
				new InitializationEvent(master.getId(), state));
		
		System.out.println("Initialization Event sent by "+master);
	}
	
	
	public void leaveCollaboration(User user) {
		Collaboration c = findCollaborationForUser(user);
		c.removeParticipant(user);
		presentUsers.remove(user);

		addEvent(DomainFactory.getDomain(c.getDomainName()),
				new UserLeftCollaborationEvent(c,user));

		addEvent(clientManagementDomain, new UserLeftCollaborationEvent(c,user));
		
		//TODO find a better way to do that
		userDB.disconnect(user.getId());
	}
	
	
	
	public void changeDrawingPanel(User user, String defsId) {
		addEvent(DomainFactory.getDomain(findCollaborationForUser(user).getDomainName())
				, new ChangeDrawingPanelEvent(user.getId(), defsId));
		
		//TODO remove trace
		System.out.println("------------------------------------------------------------------------------------");
		System.out.println("changed drawing panel    by "+user);
		System.out.println("domain "+findCollaborationForUser(user).getDomainName()+" : "+findCollaborationForUser(user).getPresentParticipants().getUsers());
		System.out.println(UserManagerFactory.getInstance().getUserManager().getUsers());
		System.out.println("------------------------------------------------------------------------------------");
	}
	
	
	public void closeDrawingPanel(User user, String defsId) {
		addEvent(DomainFactory.getDomain(findCollaborationForUser(user).getDomainName()),
				new CloseDrawingPanelEvent(user.getId(), defsId));
		
		//TODO remove trace
		System.out.println("------------------------------------------------------------------------------------");
		System.out.println("drawing panel closed   by "+user);
		System.out.println("domain "+findCollaborationForUser(user).getDomainName()+" : "+findCollaborationForUser(user).getPresentParticipants().getUsers());
		System.out.println(UserManagerFactory.getInstance().getUserManager().getUsers());
		System.out.println("------------------------------------------------------------------------------------");
	}

	
	
	public void newDrawingPanel(User user, DefinitionsSyntaxModel defs) {
		addEvent(DomainFactory.getDomain(findCollaborationForUser(user).getDomainName())
				, new NewDrawingPanelEvent(user.getId(),defs));
		
		//TODO remove trace
		System.out.println("------------------------------------------------------------------------------------");
		System.out.println("new drawing panel    by "+user);
		System.out.println("domain "+findCollaborationForUser(user).getDomainName()+" : "+findCollaborationForUser(user).getPresentParticipants().getUsers());
		System.out.println(UserManagerFactory.getInstance().getUserManager().getUsers());
		System.out.println("------------------------------------------------------------------------------------");
	}

	
	
	public void submitModification(User user, CometLogEntry entry) {
		addEvent(DomainFactory.getDomain(findCollaborationForUser(user).getDomainName())
				, new ModificationEvent(user.getId(),entry));
		
		//TODO remove trace
		System.out.println("------------------------------------------------------------------------------------");
		System.out.println("modification    by "+user);
		System.out.println("domain "+findCollaborationForUser(user).getDomainName()+" : "+findCollaborationForUser(user).getPresentParticipants().getUsers());
		System.out.println(UserManagerFactory.getInstance().getUserManager().getUsers());
		if(entry instanceof AddElementCometLogEntry){
			System.out.println(user+" added "+((AddElementCometLogEntry)entry).getBean().getId());
		}
		System.out.println("------------------------------------------------------------------------------------");
	}
	
	
	
	public boolean lock(User user){
		Collaboration collab = findCollaborationForUser(user);
		
		if(collab.getMaster().equals(user)){
			addEvent(DomainFactory.getDomain(collab.getDomainName()), new LockEvent(user.getId()));
			return true;
		}
		
		return false;
	}
	
	
	
	public boolean unlock(User user){
		Collaboration collab = findCollaborationForUser(user);
		
		if(collab.getMaster().equals(user)){
			addEvent(DomainFactory.getDomain(collab.getDomainName()), new UnlockEvent(user.getId()));
			return true;
		}
		
		return false;
	}
	
	
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	
	public CollaborationStatus connnect(String editorID) {
		if(userDB.connect(editorID)){
			addEvent(clientManagementDomain, new ClientLoginEvent(editorID));
			return new CollaborationStatus(userDB.getUsers(),collaborations);
		}
		return null;
	}



	public boolean disconnect(String editorID) {
		for(User user : presentUsers.keySet()){
			if(user.getId().equals(editorID)){
				leaveCollaboration(user);
				break;
			}
		}
		
		userDB.disconnect(editorID);
		return false;
	}



	public Collaboration createCollaboration(String editorId, String collaborationName,
			UserGroup coworkers) {
		
		Collaboration c = findCollaborationByName(collaborationName);
		
		if(c==null){
			c = new Collaboration(collaborationName, generateDomainName(collaborationName), coworkers);
			collaborations.add(c);
			
			addEvent(clientManagementDomain, new NewCollaborationEvent(editorId, c));
			
			return c;
		}
		else{
			return null;
		}
	}


	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	private String generateDomainName(String collaborationName){
		return collaborationName + System.currentTimeMillis();
	}
	
	
	private Collaboration findCollaborationByName(String name){
		for(Collaboration c : collaborations){
			if(c.getName().equals(name)){
				return c;
			}
		}
		return null;
	}
	
	private Collaboration findCollaborationForUser(User user){
		for(User u : presentUsers.keySet()){
			if(u.getId().equals(user.getId())){
				return presentUsers.get(u);
			}
		}
		return null;
	}





}
