package com.ebm_ws.infra.bricks.components.base.binding.expressions;

import java.io.StringReader;
import java.util.ArrayList;

import com.ebm_ws.infra.bricks.components.base.binding.expressions.ICompilationContext.UnresolvedObjectError;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.ArrayObj;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.Attribute;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.Braces;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.GlobalVar;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.Method;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.ParseError;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.ParseHelper;
import com.ebm_ws.infra.bricks.components.base.binding.expressions.impl.StaticObj;

public class ExpressionCompiler
{
	private static IEvaluable[] EVAL_ARRAY_TYPE = {};
//	private static Class[] CLASS_ARRAY_TYPE = {};
	
//	protected static String COMMENT_LINE_TAG = "//";
//	private static String BLANK_CHARSET = " \t";
//	private static String NEWLINE_CHARSET = "\n\r";
	private static String BLANK_CHARS = " \t\n\r";
	private static String SEPARATOR_CHARS = "'\".,;(){}[]=+-*&|/#<>";
	
	private ParseHelper parser;
	
	public ExpressionCompiler(String expression)
	{
		parser = new ParseHelper(new StringReader(expression));
	}
	public IEvaluable compile(ICompilationContext iContext) throws ParseError, UnresolvedObjectError
	{
		parser.curChar();
		try
        {
	        return parseLeadingExpression(iContext, null);
        }
        catch(CompilationError e)
        {
        	parser.rethrow("Compilation error", e);
        	return null;
        }
	}
	private IEvaluable parseLeadingExpression(ICompilationContext iContext, String iExitOn) throws ParseError, CompilationError, UnresolvedObjectError
	{
		// --- leading expression can be:
		// Boolean: true / false
		// Integer: [0-9]+
		// String: '...' or "..."
		// object: <name>
		
		parser.skipChars(BLANK_CHARS);
		
		// --- determine expression
		IEvaluable leadingExpression = null;
		
		if(parser.consumeChar('\'', null))
		{
			// --- String value
			String string = parser.readUntil("\'");
			if(!parser.consumeChar('\'', null))
				parser.error(ParseError.RC_SYNTAX_ERROR, "Closing ' expected.");
			leadingExpression = new StaticObj(iContext, string);
		}
		else if(parser.consumeChar('\"', null))
		{
			// --- String value
			String string = parser.readUntil("\"");
			if(!parser.consumeChar('\"', null))
				parser.error(ParseError.RC_SYNTAX_ERROR, "Closing \" expected.");
			leadingExpression = new StaticObj(iContext, string);
		}
		else if(parser.consumeChar('[', null))
		{
			// --- Array
			// --- now read all array elements
			ArrayList<IEvaluable> elts = new ArrayList<IEvaluable>();
			while(!parser.consumeChar(']', BLANK_CHARS))
			{
				if(elts.size() > 0)
				{
					// --- read comma
					if(!parser.consumeChar(',', BLANK_CHARS))
						parser.error(ParseError.RC_SYNTAX_ERROR, "',' expected");
				}
				parser.skipChars(BLANK_CHARS);
				IEvaluable elt = parseLeadingExpression(iContext, ",]");
				elts.add(elt);
			}
			// --- arguments read
			leadingExpression = new ArrayObj(iContext, elts.toArray(EVAL_ARRAY_TYPE));
		}
		else
		{
			String word = parser.readUntil(BLANK_CHARS+SEPARATOR_CHARS);
			if(word == null || word.length() == 0)
				parser.error(ParseError.RC_SYNTAX_ERROR, "Expression expected.");
			
			if(word.equals("true"))
			{
				leadingExpression = new StaticObj(iContext, Boolean.TRUE);
			}
			else if(word.equals("false"))
			{
				leadingExpression = new StaticObj(iContext, Boolean.FALSE);
			}
			else
			{
				// --- is this an integer?
				try
				{
					Integer val = new Integer(word);
					leadingExpression = new StaticObj(iContext, val);
				}
				catch(NumberFormatException e)
				{
					// --- that was not an integer: must be a global variable
					leadingExpression = new GlobalVar(iContext, word);
				}
			}
		}
		
		// --- then parse next expressions
		return parseObjectExpression(iContext, leadingExpression, iExitOn);
	}
	private IEvaluable parseObjectExpression(ICompilationContext iContext, IEvaluable iObject, String iExitOn) throws ParseError, CompilationError, UnresolvedObjectError
	{
		if(parser.curChar() < 0 || (iExitOn != null && iExitOn.indexOf(parser.curChar()) >= 0))
		{
			// --- end of expression
			return iObject;
		}
		else if(parser.consumeChar('.', null))
		{
			// --- attribute or method access
			String name = parser.readUntil(BLANK_CHARS+SEPARATOR_CHARS);
			if(name == null || name.length() == 0)
				parser.error(ParseError.RC_SYNTAX_ERROR, "Attribute or method name expected.");
			if(parser.consumeChar('(', BLANK_CHARS))
			{
				// --- read a method invocation
				// --- now read all arguments
				ArrayList<IEvaluable> arguments = new ArrayList<IEvaluable>();
				while(!parser.consumeChar(')', BLANK_CHARS))
				{
					if(arguments.size() > 0)
					{
						// --- read comma
						if(!parser.consumeChar(',', BLANK_CHARS))
							parser.error(ParseError.RC_SYNTAX_ERROR, "',' expected");
					}
					parser.skipChars(BLANK_CHARS);
					IEvaluable arg = parseLeadingExpression(iContext, ",)");
					arguments.add(arg);
				}
				// --- arguments read
				return parseObjectExpression(iContext, new Method(iContext, name, iObject, arguments.toArray(EVAL_ARRAY_TYPE)), iExitOn);
			}
			else
			{
				// --- this is an attribute access
				return parseObjectExpression(iContext, new Attribute(iContext, name, iObject), iExitOn);
			}
		}
		else if(parser.consumeChar('[', null))
		{
			// --- brace access
			IEvaluable braces = new Braces(iContext, iObject, parseLeadingExpression(iContext, "]"));
			if(!parser.consumeChar(']', null))
				parser.error(ParseError.RC_SYNTAX_ERROR, "Closing brace ']' expected.");
			return parseObjectExpression(iContext, braces, iExitOn);
		}
		else
		{
			// --- unexpected char
			parser.error(ParseError.RC_SYNTAX_ERROR, "Unexpected char: '"+((char)parser.curChar())+"'.");
			return null;// never called
		}
	}

}
