package com.ebm_ws.infra.bricks.components.base.error;

import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.ResourceBundle;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.ebm_ws.infra.bricks.error.IError;
import com.ebm_ws.infra.bricks.session.BricksSession;
import com.ebm_ws.infra.bricks.util.BricksMessages;
import com.ebm_ws.infra.bricks.util.BricksUrlBuilder;
import com.ebm_ws.infra.bricks.util.HtmlUtils;

public class DefaultErrorHandler extends ErrorHandler
{
	@Override
	public String getTitle(HttpServletRequest iRequest) throws Exception
	{
		BricksSession session = BricksSession.getSession(iRequest);
		return getLocalizedErrorTitle(getError(iRequest), session.getLocale(), iRequest);
	}
	public void preRender(HttpServletRequest iRequest) throws Exception
	{
		Throwable error = getError(iRequest);
		short type = getErrorType(error, iRequest);
		if(type == IError.TECHNICAL_ERROR)
		{
			logger.warn("Technical Error Occured", error);
		}
	}
	public short getErrorType(Throwable iError, HttpServletRequest iRequest)
	{
		if(iError instanceof IError)
			return ((IError)iError).getType();
		return IError.TECHNICAL_ERROR;
	}
	public String getLocalizedErrorTitle(Throwable iError, Locale iLocale, HttpServletRequest iRequest)
	{
		if(iError instanceof IError)
		{
			IError de = (IError)iError;
			return de.getTitle(iRequest);
		}
		Object[] args = new Object[]{iError.getMessage(), iError.getClass().getName(), iError.getClass().getSimpleName()};
//		ResourceBundle msgs = ResourceBundle.getBundle(_bundle, session.getLocale());
		ResourceBundle msgs = BricksMessages.getBundle(iLocale);
		return MessageFormat.format(msgs.getString("error.generic.title"), args);
	}
	public String getLocalizedErrorMessage(Throwable iError, Locale iLocale, HttpServletRequest iRequest)
	{
		if(iError instanceof IError)
		{
			IError de = (IError)iError;
			return de.getMessage(iRequest);
		}
		ResourceBundle msgs = BricksMessages.getBundle(iLocale);
		Object[] args = new Object[]{iError.getMessage(), iError.getClass().getName(), iError.getClass().getSimpleName()};
		return MessageFormat.format(msgs.getString("error.generic.message"), args);
	}
	public String getNextUrl(Throwable iError, HttpServletRequest iRequest)
	{
		String forcedUrl = getNextUrl(iRequest);
		if(forcedUrl != null)
			return forcedUrl;
		
		if(iError instanceof IError)
		{
			IError de = (IError)iError;
			if(de.getNextUrl() != null)
				return de.getNextUrl();
		}
		
		// --- get back (if possible) to the last stable page
		String referrerUrl = iRequest.getHeader("referer");
		if(referrerUrl != null)
			return referrerUrl;
		
		// --- retry
//		BricksUrlBuilder url = BricksUrlBuilder.copyFromRequest(iRequest, true);
//		return url.toUrl(iRequest.getCharacterEncoding(), true);
		
		// --- go back to homepage
		BricksUrlBuilder url = BricksUrlBuilder.copyFromRequest(iRequest, false);
		return url.getServlet();
	}
	public void render(HttpServletRequest iRequest, HttpServletResponse iResponse) throws Exception
	{
		PrintWriter writer = iResponse.getWriter();
		Throwable error = getError(iRequest);
		String nextUrl = getNextUrl(error, iRequest);
		short type = getErrorType(error, iRequest);
		String message = getLocalizedErrorMessage(error, iResponse.getLocale(), iRequest);
		
		writer.println("<div id='errorbox'>");
		writer.println("<p class='message'>");
		if(type == IError.FUNCTIONAL_ERROR)
		{
			// --- for a functional error, only display the message
			writer.print(HtmlUtils.encode2HTML(message));
		}
		else
		{
			// --- for a technical error, display the message, the cause and an advice
			ResourceBundle msgs = BricksMessages.getBundle(iResponse.getLocale());
			String cause = error.getCause() == null ? null : getLocalizedErrorMessage(error.getCause(), iResponse.getLocale(), iRequest);
			writer.println("<dl>");
			
			// --- message (the action that failed)
			writer.print("<dt>");
			writer.print("<strong>");
			writer.print(msgs.getString("error.message.label"));
			writer.print("</strong>");
			writer.println("</dt>");
			writer.print("<dd>");
			writer.print(HtmlUtils.encode2HTML(message));
			writer.println("</dd>");
			
			// --- cause
			if(cause != null)
			{
				writer.print("<dt>");
				writer.print("<strong>");
				writer.print(msgs.getString("error.cause.label"));
				writer.print("</strong>");
				writer.println("</dt>");
				writer.print("<dd>");
				writer.print(HtmlUtils.encode2HTML(cause));
				writer.println("</dd>");
			}
			//	--- advise
			/*
			if(advise != null)
			{
				writer.print("<dt>");
				writer.print("<strong>");
				writer.print(msgs.getString("error.advise.label"));
				writer.print("</strong>");
				writer.println("</dt>");
				writer.print("<dd>");
				writer.print(HtmlUtils.encode2HTML(advise, false));
				writer.println("</dd>");
			}
			*/
			writer.println("</dl>");
		}

		writer.println("</p>");

//		--- 2: OK button
//		----------------
		writer.print("<div class='Buttons'>");
		writer.print("<ul>");
		if(error != null && type != IError.FUNCTIONAL_ERROR)
		{
			writer.print("<li>");
			writer.print("<a href=\"javascript:void(stack.style.display=stack.style.display == 'none' ? 'block' : 'none')\">");
			writer.print(HtmlUtils.encode2HTML(BricksMessages.getMessage(iResponse.getLocale(), "error.show_stack")));
			writer.print("</a>");
			writer.println("</li>");
		}
		
		writer.print("<li>");
		writer.print("<a href='");
		writer.print(nextUrl);
		writer.print("'>");
		writer.print(HtmlUtils.encode2HTML(BricksMessages.getMessage(iResponse.getLocale(), "error.ok")));
		writer.print("</a>");
		writer.println("</li>");
		
		writer.print("</ul>");
		writer.println("</div>");

//		--- 3: dtails
//		--------------
		if(error != null && type != IError.FUNCTIONAL_ERROR)
		{
			writer.println("<div id='stack' style=\"display: none\">");
			writer.println("<dl>");
			for(int i=0; i<10 && error != null; i++)
			{
				if(error instanceof InvocationTargetException)
					error = ((InvocationTargetException)error).getTargetException();
				
				writer.print("<dt>");
				writer.print("<strong>");
//				writer.print("["+i+"]: ");
				writer.print(error.getClass().getName());
				writer.print(": ");
//				writer.print(getLocalizedErrorMessage(error, session.getLocale(), iRequest));
				writer.print(HtmlUtils.encode2HTML(error.getMessage()));
				writer.print("</strong>");
				writer.println("</dt>");
				writer.print("<dd>");
				writer.print("<pre>");
//				error.printStackTrace(writer);
				StackTraceElement[] elements = error.getStackTrace();
				for(int j=0; j<elements.length; j++)
				{
					writer.println(HtmlUtils.encode2HTML(elements[j].toString()));
				}
				writer.print("</pre>");
				writer.println("</dd>");
				
				error = error.getCause();
			}
			writer.println("</dl>");
			writer.println("</div>");
		}
		writer.println("</div>");
	}
}
