package com.ebm_ws.infra.bricks.impl.multiparts;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class MultiPartsProcessor extends InputStream implements IPart
{
	private static Log _trace = LogFactory.getLog(MultiPartsProcessor.class);
	
	private static Hashtable<String, String> _params = new Hashtable<String, String>();
	
	private static char CR = '\r';
	private static char LF = '\n';
	private static String CR_ONLY = "\r";
	private static String CRLF = "\r\n";
	private static String BLANK_CHARS = " \t";
	private static String HEADERSEP_CHARS = "=;:";
	private static String RET_CHARS = "\n\r";
	private static String NULL_VALUE = "NULL";

//	private static String ENCODING = "UTF8";
	private static int BUFFER_MAX_LEN = 128;

	private ByteArrayOutputStream _bytesBuffer = new ByteArrayOutputStream();
	
	private String encoding;
	private byte[] _boundary;
	private InputStream _input;
	private int _curByte;

	private int _fifoPos = 0;
	private byte[] _fifoBuffer;

	private Hashtable<String, String> _formData = new Hashtable<String, String>();
	private String _inputValue = null;
	private boolean _emptyStream = false;
	
	public MultiPartsProcessor(HttpServletRequest iRequest, String iCharSet, int bufferSize) throws IOException
	{
		this(iRequest.getContentType(), iRequest.getInputStream(), iCharSet, bufferSize);
	}
	public MultiPartsProcessor(String iContentType, InputStream iInput, String iCharSet, int bufferSize) throws IOException
	{
		_input = iInput;
		if(bufferSize > 0)
			_input = new BufferedInputStream(iInput, bufferSize);
		
		encoding = iCharSet;
		
		// --- extract boundary
		String boundary = getBoundary(iContentType);
		//System.out.println("ContentType: "+iContentType+" (indexof: "+iContentType.indexOf("boundary=")+"/lastindexof: "+iContentType.lastIndexOf("boundary=")+")");
		_trace.debug("new MultiPartsProcessor with boundary ["+boundary+"].");
		boundary = CRLF+"--"+boundary;
		_boundary = boundary.getBytes();
		
		// --- initialise fifo buffer with CR.LF.<boundary.len-2 bytes>
		_fifoBuffer = new byte[_boundary.length];
		_fifoPos = 0;
		_fifoBuffer[0] = (byte)CR;
		_fifoBuffer[1] = (byte)LF;
//		int nbread = _input.read(_fifoBuffer, 2, _boundary.length-2);
		fillBuffer(_fifoBuffer, 2, _boundary.length-2);
		// --- fifo buffer should contain the boundary. will be checked at first nextPart()
	}
	private String getBoundary(String iContentType)
	{
		int idx = iContentType.lastIndexOf("boundary=");
		if (idx < 0) return null;
	
		String boundary = iContentType.substring(idx+9);  // 9 for "boundary="
		if(boundary.charAt(0) == '"')
		{
			// --- The boundary is enclosed in quotes, strip them
			idx = boundary.lastIndexOf('"');
			boundary = boundary.substring(1, idx);
		}
		// The real boundary is always preceeded by an extra "--"
//		boundary = "--" + boundary;
		return boundary;
	}
	/**
	 * Determines whether if the current _fifoBuffer content matches the boundary.
	 */
	private boolean isOnBoundary()
	{
		for(int i=_boundary.length-1; i>=0; i--)
		{
			if(_boundary[i] != _fifoBuffer[(i+_fifoPos)%_boundary.length])
				return false;
		}
		return true;
	}
	/**
	 * This method must be called when a boundary was just read.
	 * @return
	 */
	private boolean internalHasMoreParts() throws IOException
	{
		if(_trace.isDebugEnabled()) _trace.debug("Info: look for next part...");
		
		int c, skipped;
		
		// =========== read bytes ===============================
		
		// --- reset current part
		_formData.clear();
		_inputValue = null;
		_emptyStream = false;
		
		// --- skip last part bytes until boundary
		_bytesBuffer.reset();
		skipped = 0;
		while((c = read()) >= 0)
		{
			if(skipped <= BUFFER_MAX_LEN) _bytesBuffer.write(c);
			skipped++;
		}
		if(skipped > 0)
		{
			if(_trace.isDebugEnabled())
			{
				// --- display skipped bytes
				if(skipped > _bytesBuffer.size())
					_trace.debug("Info: skipped "+skipped+" bytes from last part: ["+_bytesBuffer.toString()+"...].");
				else
					_trace.debug("Info: skipped "+skipped+" bytes from last part: ["+_bytesBuffer.toString()+"].");
			}
			else if(_trace.isDebugEnabled())
			{
				// --- display number of skipped bytes
				_trace.debug("Info: skipped "+skipped+" bytes from last part.");
			}
		}
		
		// --- boundary was just read
		int b1 = nextByte();
		int b2 = nextByte();
		if(b1 != CR || b2 != LF)
		{
			// --- this is the terminating boundary: normally the next 2 bytes are "--"
			if(b1 != '-' || b2 != '-')
				_trace.debug("Warning: unexpected bytes after terminating boundary: '"+b1+"' '"+b2+"'");
			
			// --- finish the stream: possibly read CRLF
			b1 = nextByte();
			b2 = nextByte();
			if(b1 < 0 || b2 < 0)
			{
				_trace.debug("Info: terminating boundary read. No more part.");
				return false;
			}
			
			_bytesBuffer.reset();
			skipped = 0;
			if(b1 != CR || b2 != LF)
			{
				_bytesBuffer.write(b1);
				_bytesBuffer.write(b2);
				skipped += 2;
			}
			while((c = nextByte()) >= 0)
			{
				if(skipped <= BUFFER_MAX_LEN) _bytesBuffer.write(c);
				skipped++;
			}
			/*
			if(skipped > 0)
			{
				if(_trace.isDebugEnabled())
				{
					// --- display skipped bytes
					if(skipped > _bytesBuffer.size())
						_trace.debug("Warning: skipped "+skipped+" bytes after terminating boundary: ["+_bytesBuffer.toString()+"...].");
					else
						_trace.debug("Warning: skipped "+skipped+" bytes after terminating boundary: ["+_bytesBuffer.toString()+"].");
				}
				else if(_trace.isDebugEnabled())
				{
					// --- display number of skipped bytes
					_trace.debug("Warning: skipped "+skipped+" bytes after terminating boundary.");
				}
			}
			*/
			_trace.debug("Info: terminating boundary read. No more part.");
			return false;
		}
		
		// =========== read chars (from Reader) ===============================
		// --- init next part
		if(_trace.isDebugEnabled()) _trace.debug("Info: init next part...");
		readHeaders();
		
		// =========== read bytes ===============================
		// --- reinit _fifoBuff
//		int nbread = _input.read(_fifoBuffer);
		int nbread = fillBuffer(_fifoBuffer, 0, _fifoBuffer.length);
		_fifoPos = 0;
		if(nbread < _boundary.length)
		{
			_trace.debug("Error: unexpected input stream end ("+nbread+" bytes read instead of "+_boundary.length+").");
			// TODO: throw exception?
			return false;
		}
		
		// --- is it a file upload or an input data part?
		if(isFileUpload())
		{
			// --- this is a file ulpoad
			_trace.debug("Info: file upload part ["+getInputName()+"]=["+getFileName()+"] ("+getFileContentType()+") initialised.");
			if(isOnBoundary())
			{
				// --- no file input?
				_emptyStream = true;
				_trace.debug("Warning: file input stream is empty."); 
			}
		}
		else
		{
			// --- this is an input data: read value
			_bytesBuffer.reset();
			while((c=read()) >= 0)
				_bytesBuffer.write(c);
			_inputValue = _bytesBuffer.toString(encoding);
			_trace.debug("Info: input data part ["+getInputName()+"]=["+_inputValue+"].");
			_params.put(getInputName(), _inputValue);
		}
		
		return true;
	}
	private void readHeaders() throws IOException
	{
		// read a list of <name>: <value>[;<name>="<val>"]*CRLF
		while(_curByte >= 0)
		{
			String headerLine = readUnicodeLine();
			// --- break on empty line
			if(headerLine == null)
				break;
			
			// --- decode header
			StringParser sa = new StringParser(headerLine);
			readHeader(sa);
			
			// --- skip rest of line if need be
			if(sa.available() > 0)
			{
				// --- display skipped chars
				_trace.debug("Warning: skipped "+sa.available()+" chars after terminating boundary: ["+sa.getAvailable()+"].");
			}
		}
	}
	/*
	 * reads a header line, and does not consume the CR line end.
	 */
	private void readHeader(StringParser iParser) throws IOException
	{
		// --- read header name
		String headerName = iParser.readStringUntil(BLANK_CHARS+HEADERSEP_CHARS+CR);
		if(headerName == null)
		{
			_trace.debug("Warning: null header name read. Skip rest of line.");
			return;
		}
		headerName = headerName.toLowerCase();
			
		if(!iParser.readChar(':', BLANK_CHARS))
		{
			_trace.debug("Warning: ':' expected after header name ["+headerName+"] : '"+_curByte+"' read instead. Skip rest of line.");
			return;
		}
		// --- read header value
		iParser.skipChars(BLANK_CHARS);
		String headerValue = iParser.readStringUntil(BLANK_CHARS+HEADERSEP_CHARS+CR);
		if(headerValue == null)
		{
			_trace.debug("Warning: null header value read for ["+headerName+"].");
			_formData.put(headerName, NULL_VALUE);
		}
		else
		{
			if(_trace.isDebugEnabled()) _trace.debug("Info: header read ["+headerName+"] = ["+headerValue+"].");
			_formData.put(headerName, headerValue);
		}
			
		// --- read optional fields
		iParser.skipChars(BLANK_CHARS);
		while(iParser.available() > 0)
		{
			// --- optional field begins with a ';'
			if(!iParser.readChar(';', BLANK_CHARS))
			{
				_trace.debug("Warning: ';' expected after header name and value ["+headerName+"] = ["+headerValue+"] : '"+_curByte+"' read instead. Skip rest of line.");
				return;
			}

			// --- read optional field name
			iParser.skipChars(BLANK_CHARS);
			String optName = iParser.readStringUntil(BLANK_CHARS+HEADERSEP_CHARS+CR);
			if(optName == null)
			{
				_trace.debug("Warning: null optional header name. Skip rest of line.");
				return;
			}
			// --- read '='
			if(!iParser.readChar('=', BLANK_CHARS))
			{
				_trace.debug("Warning: '=' expected after header optional name ["+optName+"]. '"+_curByte+"' read instead. Skip rest of line.");
				return;
			}
			// --- read a quote?
			boolean quotes = iParser.readChar('\"', BLANK_CHARS);
			// --- read optional field value
			String optValue = quotes ? iParser.readStringUntil("\""+CR) : iParser.readStringUntil(";"+CR);
			// --- read closing quote if need be
			if(quotes && !iParser.readChar('\"', null))
			{
				_trace.debug("Warning: closing quotes expected after header optional ["+optName+"] = ["+optValue+"]. '"+_curByte+"' read instead. Skip rest of line.");
				return;
			}
			
			if(_trace.isDebugEnabled()) _trace.debug("Info: optional header read ["+optName+"]=["+optValue+"]");
			if(optValue == null)
				_formData.put(headerName+"@"+optName, NULL_VALUE);
			else
				_formData.put(headerName+"@"+optName, optValue);

			iParser.skipChars(BLANK_CHARS);
		}
	}
	// +------------------------------------------------
	// | read helper functions
	// +------------------------------------------------
	private int nextByte() throws IOException
	{
		_curByte = _input.read();
		return _curByte;
	}
	private int fillBuffer(byte[] oBuffer, int iOffset, int iLen) throws IOException
	{
		int len = _input.read(oBuffer, iOffset, iLen);
		// --- are there remaining bytes?
		while(len < iLen)
		{
			int b = _input.read();
			if(b < 0) return len; // --- end of stream
//System.out.println("!!!!! rattrap readBytes() !!!!!!!");
			oBuffer[iOffset+len] = (byte)b;
			len++;
		}
		
		return len;
	}
	private String readUnicodeLine() throws IOException
	{
		_bytesBuffer.reset();
		int b1 = nextByte();
		int b2 = nextByte();
		while(b2 >= 0 && (b1 != CR || b2 != LF))
		{
			_bytesBuffer.write(b1);
			b1 = b2;
			b2 = nextByte();
		}
		if(_bytesBuffer.size() == 0)
			return null;
		return _bytesBuffer.toString(encoding);
	}
/*
	private boolean readByte(char c, String skip) throws IOException
	{
		if(skip != null)
			skipBytes(skip);
		if(curChar != c)
			return false;
		// --- expected char has been read
		nextByte();
		return true;
	}
	private void skipBytes(String chars) throws IOException
	{
		while(curChar >= 0 && chars.indexOf(curChar) >= 0)
			nextByte();
	}
	private String readStringUntil(String chars) throws IOException
	{
		_bytesBuffer.reset();
		while(curChar >= 0 && chars.indexOf(curChar) < 0)
		{
			_bytesBuffer.write(curChar);
			nextByte();
		}
		if(_bytesBuffer.size() == 0)
			return null;
		return _bytesBuffer.toString(ENCODING);
	}
*/
/* UTF
	private int nextChar() throws IOException
	{
		curChar = _reader.read();
		return curChar;
	}
	private boolean readChar(char c, String skip) throws IOException
	{
		if(skip != null)
			skipChars(skip);
		if(curChar != c)
			return false;
		// --- expected char has been read
		nextChar();
		return true;
	}
	private void skipChars(String chars) throws IOException
	{
		while(curChar >= 0 && chars.indexOf(curChar) >= 0)
			nextChar();
	}
	private String readStringUntil(String chars) throws IOException
	{
		_charsBuffer.reset();
		while(curChar >= 0 && chars.indexOf(curChar) < 0)
		{
			_charsBuffer.write(curChar);
			nextChar();
		}
		if(_charsBuffer.size() == 0)
			return null;
		return _charsBuffer.toString();
	}
*/
	// +------------------------------------------------
	// | InputStream
	// +------------------------------------------------
	/**
	 * @see InputStream#read()
	 */
	public int read() throws IOException
	{
		// --- if on boundary, return a <0 value
		if(isOnBoundary())
			return -1;
		
		// --- read byte from input stream
		int newc = nextByte();
		if(newc < 0)
		{
			_trace.debug("Error: end of input stream reached before terminating boundary!");
			throw new IOException("End of input stream reached before terminating boundary.");
		}
		// --- push new byte in FIFO, get oldest byte from FIFO
		int retc = _fifoBuffer[_fifoPos];
		if(retc < 0) retc += 256;
		_fifoBuffer[_fifoPos] = (byte)newc;
		
		// --- next FIFO pos
		_fifoPos++;
		if(_fifoPos >= _boundary.length) _fifoPos = 0;
		
		return retc;
	}
	// +------------------------------------------------
	// | IMultiPartsProcessor
	// +------------------------------------------------
	public boolean hasMoreParts()
	{
		try
		{
			return internalHasMoreParts();
		}
		catch (IOException e)
		{
			_trace.debug("Error: IOException in internalHasMoreParts():", e);
			return false;
		}
	}
	public IPart nextPart()
	{
		return this;
	}
	public String getParameter(String iName)
	{
		return (String)_params.get(iName);
	}
	// +------------------------------------------------
	// | IUploadPart
	// +------------------------------------------------
	private String getFormData(String iEntry)
	{
		String val = (String)_formData.get(iEntry);
		if(val == NULL_VALUE) return null;
		return val;
	}
	public String getFileName()
	{
		return getFormData("content-disposition@filename");
	}
	public InputStream getFileInputStream()
	{
		return this;
	}
	public String getInputName()
	{
		return getFormData("content-disposition@name");
	}
	public String getDataInputValue()
	{
		return _inputValue;
	}
	public String getFileContentType()
	{
		return getFormData("content-type");
	}
	public boolean isFileUpload()
	{
		return _formData.get("content-disposition@filename") != null;
	}
	public boolean looksLikeAnInvalidFile()
	{
		return _emptyStream;
	}
	// ===========================================================================
	// === StringParser
	// ===========================================================================
	private static class StringParser
	{
		private String _string;
		private int idx = 0;
		
		public StringParser(String iString)
		{
			_string = iString;
		}
		public int available()
		{
			return _string.length() - idx;
		}
		public String getAvailable()
		{
			return _string.substring(idx);
		}
		private int curChar()
		{
			if(idx >= _string.length())
				return -1;
			return _string.charAt(idx);
		}
		private int nextChar()
		{
			idx++;
			return curChar();
		}
		public boolean readChar(char c, String skip)
		{
			if(skip != null)
				skipChars(skip);
			if(curChar() != c)
				return false;
			// --- expected char has been read
			nextChar();
			return true;
		}
		public void skipChars(String chars)
		{
			while(curChar() >= 0 && chars.indexOf(curChar()) >= 0)
				nextChar();
		}
		public String readStringUntil(String chars) throws IOException
		{
			StringBuffer sb = new StringBuffer();
			while(curChar() >= 0 && chars.indexOf(curChar()) < 0)
			{
				sb.append((char)curChar());
				nextChar();
			}
			if(sb.length() == 0)
				return null;
			return sb.toString();
		}
	}
}
