package com.ebm_ws.infra.bricks.util;

import java.util.Enumeration;
import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import com.ebm_ws.infra.bricks.components.base.param.UrlParameterDef;
import com.ebm_ws.infra.bricks.session.BricksSession;
import com.ebm_ws.infra.bricks.session.IPageContext;

public class BricksUrlBuilder extends UrlBuilder
{
	/*
	--> http://localhost:9000/IPCentrex1/Main/other?arg1=val1&arg2=val2#anchor
	    req.getCharacterEncoding(): null
	    req.getContentType(): null
	    req.getContextPath(): /IPCentrex1
	    req.getLocalAddr(): 127.0.0.1
	    req.getLocalName(): localhost
	    req.getLocalPort(): 9000
	    req.getMethod(): GET
	    req.getPathInfo(): null
	    req.getPathTranslated(): null
	    req.getProtocol(): HTTP/1.1
	    req.getQueryString(): arg1=val1&arg2=val2
	    req.getRemoteAddr(): 127.0.0.1
	    req.getRemoteHost(): 127.0.0.1
	    req.getRemotePort(): 4352
	    req.getRemoteUser(): null
	    req.getRequestURI(): /IPCentrex1/Main/other
	    req.getScheme(): http
	    req.getServerName(): localhost
	    req.getServerPort(): 9000
	    req.getServletPath(): /Main/other
	    req.getLocale(): fr
	    
	    Format de requete choisi:
	    	http://<server>/<context_path>/<servlet>/<page>?<params>
	 */
	
	/**
	 * Constructs a BricksUrlBuilder that is a copy of the given request
	 * @param iRequest
	 */
	public static BricksUrlBuilder copyFromRequest(HttpServletRequest iRequest, boolean iCopyParameters)
	{
		return new BricksUrlBuilder(iRequest, iCopyParameters);
	}
	/**
	 * Creates a Bricks Url that refers to the given page (out of the current context).
	 * Input arguments still have to be added.
	 * @param iRequest
	 * @param iPage
	 * @return
	 */
	public static BricksUrlBuilder createPageUrl(HttpServletRequest iRequest, String iPage, boolean iKeepContext)
	{
		BricksUrlBuilder url = new BricksUrlBuilder(iRequest, false);
		url.setSelector(PAGE_SELECTOR);
		url.setPath(iPage);
		
		if(iKeepContext)
		{
			// --- keep context parameters
			BricksSession session = BricksSession.getSession(iRequest);
			IPageContext context = session.getCurrentPageContext();
			UrlParameterDef[] contextParameters = context.getContextDef().getInputParameters();
			if(contextParameters != null)
			{
				for(int i=0; i<contextParameters.length; i++)
				{
					String val = context.getInputParameter(contextParameters[i].getName());
					if(val != null)
						url.setParameter(contextParameters[i].getName(), val);
				}
			}
		}
		
		return url;
	}
	/**
	 * Creates a request url that point to an internal resource.
	 * @param iRequest
	 * @param iResourcePath
	 * @param iMimeType The resource MIME Type (may be null, will rely on the servlet MIME Types)
	 * @return
	 */
	public static BricksUrlBuilder createResourceUrl(HttpServletRequest iRequest, String iResourcePath, String iMimeType)
	{
		BricksUrlBuilder url = new BricksUrlBuilder(iRequest, false);
		url.setSelector(RESOURCE_SELECTOR);
		url.setPath(iResourcePath);
		if(iMimeType != null)
			url.setParameter("mime-type", iMimeType);
		return url;
	}
	
	// =========================================================================
	// ===
	// =========================================================================
	private String context;
	private String servlet;
	private String selector;
	private String path;
	
	public final static String PAGE_SELECTOR = "page";
	public final static String SERVICE_SELECTOR = "srv";
	public final static String RESOURCE_SELECTOR = "res";
	
	// --- blank constructor
	private BricksUrlBuilder()
	{
		super(null);
	}
	// --- constructor that copies 
	private BricksUrlBuilder(HttpServletRequest iRequest, boolean iCopyParameters)
	{
		super(null);
		
		// --- 1: analyse servlet path
		String[] path = iRequest.getRequestURI().split("/");
		int idx = 1;
		if(idx < path.length)
		{
			context = path[idx++];
			if(idx < path.length)
			{
				servlet = path[idx++];
				if(idx < path.length)
				{
					selector = path[idx++];
					if(idx < path.length)
					{
						setPathTokens(path, idx);
					}
				}
			}
		}
		
		if(!iCopyParameters)
			return;

		// --- 2: extract query string parameters
		boolean ispost = "post".equalsIgnoreCase(iRequest.getMethod());
		if(ispost)
		{
			// --- extract params from query string
			String query = iRequest.getQueryString();
			if(query == null)
				return;
			if(query.startsWith("?"))
				query = query.substring(1);
			StringTokenizer st = new StringTokenizer(query, "&");
			while(st.hasMoreTokens())
			{
				String name = st.nextToken();
				int idxEq = name.indexOf('=');
				if(idxEq >= 0)
				{
					// TODO: decode?
					name = name.substring(0, idxEq);
				}
				if(name.startsWith("_"))
					// --- discard
					continue;
				setParameter(name, iRequest.getParameter(name));
			}
		}
		else
		{
			// --- non POST: get params
			Enumeration params = iRequest.getParameterNames();
			if(params != null)
			{
				while(params.hasMoreElements())
				{
					String name = (String)params.nextElement();
					if(name.startsWith("_"))
						// --- discard
						continue;
					setParameter(name, iRequest.getParameter(name));
				}
			}
		}
	}
	public String getContext()
    {
    	return context;
    }
	public String getServlet()
    {
    	return servlet;
    }
	public String getSelector()
	{
		return selector;
	}
	public String getPath()
    {
    	return path;
    }
	public String[] getPathTokens()
	{
		if(path == null)
			return null;
		return path.split("/");
	}
	public void setPath(String path)
    {
    	this.path = path;
    }
	public void setPathTokens(String[] pathTokens, int offset)
	{
		if(pathTokens == null || pathTokens.length - offset <= 0)
		{
			setPath(null);
			return;
		}
		StringBuilder sb = new StringBuilder();
		for(int i=offset; i<pathTokens.length; i++)
		{
			if(i > offset)
				sb.append("/");
			sb.append(pathTokens[i]);
		}
		path = sb.toString();
	}
	public void setSelector(String selector)
    {
    	this.selector = selector;
    	// reset path
    	this.path = null;
    }
	@Override
	public String getRequestUri()
	{
		StringBuffer sb = new StringBuffer();
		sb.append("/");
		sb.append(context);
		sb.append("/");
		sb.append(servlet);
		if(selector != null)
		{
			sb.append("/");
			sb.append(selector);
			
			if(path != null)
			{
				sb.append("/");
				sb.append(path);
			}
		}
		return sb.toString();
	}
}
